/*
	coold.c - coold
	
	Cools Athlon-XP CPUs by setting enable HLT detect.
	Optionally stops processes when the temp gets too hot.
*/

/*
	Copyright (C) 2004 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
	Author grumble: Clause 9 of the GPL seems rather vague. I wish they'd omit it.
*/

/*
	I know about the following PCI registers.

	For the VIA KT333, 8235:
	> > enable:
	> > setpci -v -H1 -s 0:0.0 95=1e (HALT command detect) [Setting this will cool the CPU as long as the OS issues HLT instructions while idle]
	> > setpci -v -H1 -s 0:0.0 92=e9 (Disc When STPGNT# Detect.) [Not used in this program]
	> > setpci -v -H1 -s 0:0.0 70=86 (PCI Master Read Buffering) [Enable if you have a problems with certian devices while cooling]
	> >
	> > disable:
	> > setpci -v -H1 -s 0:0.0 70=82 (PCI Master Read Buffering) [See above]
	> > setpci -v -H1 -s 0:0.0 92=69 (Disc When STPGNT# Detect.) [Not used in this program]
	> > setpci -v -H1 -s 0:0.0 95=1c (HALT command detect) [Setting this will cool the CPU as long as the OS issues HLT instructions while idle]
*/

#include <sys/types.h>

#include <signal.h>
#include <sched.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <time.h>
#include <fcntl.h>

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/io.h>
#include <sys/stat.h>

#undef NODAEMON /* Become a daemon? */
#undef ENABLE_HLT_DETECT /* Enable HLT detect? */

#define TEMP_POLL_INTERVAL 1500000000 /* Number of nano (billionth) seconds to sleep between checking temperature */

struct mobotemp
{
	float temp;
	float warn;
	float danger;
};

static volatile int killflag = 0;

static inline void nano_to_timespec(struct timespec *ts, unsigned long long ns)
{
	if(!ts)
		return;
	ts->tv_sec = (int)(ns/1000000000);
	ts->tv_nsec = (int)(ns%1000000000);
}

void setkillflag(int n)
{
	killflag = 1;
}

void send_cool_cmd()
{
#ifdef ENABLE_HLT_DETECT
	/* This function should be replaced with actual code that sets the PCI registers */
	system("setpci -H1 -s 0:0.0 95=1e");
	/* This is so the system command has timeslice to execute */
	/* Maybe not needed? */
	sched_yield();
#ifdef NODAEMON
	printf("Cooling Enabled\n");
#endif
#endif
}

void send_hott_cmd()
{
#ifdef ENABLE_HLT_DETECT
	/* This function should be replaced with actual code that sets the PCI registers */
	system("setpci -H1 -s 0:0.0 95=1c");
	/* This is so the system command has timeslice to execute */
	/* Maybe not needed? */
	sched_yield();
#ifdef NODAEMON
	printf("Cooling Disabled\n");
#endif
#endif
}

/*
  This function writes 1 to /proc/cooldhlt to signal the kernel
  module to run the hlt loop.	Feel free to make this better
  (e.g. use ioctl, syscall, etc.).
*/
void hlt_loop()
{
	int cooldhlt_fd;

	if((cooldhlt_fd = open("/proc/cooldhlt", O_WRONLY)) == -1)
		return;

	/* This is where all the magic happens */
#ifdef NODAEMON
	fprintf(stdout, "Begin HLT\n");
	fflush(stdout);
#endif
	write(cooldhlt_fd, "1", 1); /* FIXME: Write the duration instead */
#ifdef NODAEMON
	fprintf(stdout, "End HLT\n");
	fflush(stdout);
#endif

	close(cooldhlt_fd);
}

void hlt_on_temp(const struct mobotemp *cpCPU, const struct mobotemp *cpcase)
{
	static int cooling = 0;
	
#ifdef NODAEMON
	/* For debugging new temp sensors */
	printf("Case: %.1f %.1f %.1f\n", cpcase->temp, cpcase->warn, cpcase->danger);
	printf("CPU: %.1f %.1f %.1f\n", cpCPU->temp, cpCPU->warn, cpCPU->danger);
#endif

	/* HLT when in a danger state	*/
	if((cpCPU->temp >= cpCPU->danger) || (cpcase->temp >= cpcase->danger))
	{
		cooling = 1;
		hlt_loop();
	}
	/* Keep HLTing until 1 below warning temperature is reached */
	else if(cooling && ((cpCPU->temp >= cpCPU->warn) || (cpcase->temp >= cpcase->warn)))
	{
		hlt_loop();
	}
	/* Go to sleep and let programs run */
	else
	{
		struct timespec ts;
		
		cooling = 0;
		
		nano_to_timespec(&ts, TEMP_POLL_INTERVAL);
		nanosleep(&ts, NULL);
	}
}

/*
	The temps should really be stored as an array of mobotemp's
	with some added information. Using libsensors.
*/

/*
  This function is not portable, it should be removed.
*/
int get_cpu_temp(struct mobotemp *pmt)
{
	int retval = 0;
	FILE *temp_proc;

	if(!(temp_proc = fopen("/proc/sys/dev/sensors/w83697hf-isa-0290/temp2", "r")))
		return -1;

	if(fscanf(temp_proc, "%f %f %f", &pmt->danger, &pmt->warn, &pmt->temp) != 3)
		retval = -1;

	fclose(temp_proc);

	return retval;
}

/*
  This function is not portable, it should be removed.
*/
int get_case_temp(struct mobotemp *pmt)
{
	int retval = 0;
	FILE *temp_proc;

	if(!(temp_proc = fopen("/proc/sys/dev/sensors/w83697hf-isa-0290/temp1", "r")))
		return -1;

	if(fscanf(temp_proc, "%f %f %f", &pmt->danger, &pmt->warn, &pmt->temp) != 3)
		retval = -1;

	fclose(temp_proc);

	return retval;
}

int main(void)
{
	if(getuid())
	{
		printf("Must be root!\n");
		return 1;
	}

	/* Create a realtime FIFO process with the highest static priority */
	struct sched_param sp;
	
	sp.sched_priority = sched_get_priority_max(SCHED_FIFO);
	
	if(sched_setscheduler(getpid(), SCHED_FIFO, &sp))
	{
		printf("Unable to set priority!\n");
		return 1;
	}

	signal(SIGTERM, setkillflag);
	signal(SIGINT, setkillflag);

#ifndef NODAEMON
	if(daemon(0, 0))
	{
		printf("Error daemonizing\n");
		return 1;
	}
#endif
	
	/* Set PCI registers that enable cooling */
	send_cool_cmd();
	
	while(!killflag)
	{
		struct mobotemp mtCPU, mtcase;

		if(!(get_cpu_temp(&mtCPU) || get_case_temp(&mtcase))) {
			hlt_on_temp(&mtCPU, &mtcase);
		}
	}

	/* Set PCI registers that disable cooling */
	send_hott_cmd();

	return 0;
}
