/*
	coold.c - coold
	
	Cools Athlon-XP CPUs by setting enable HLT detect.
	HLTs the system when the temperature gets too hot.
*/

/*
	Copyright (C) 2004-2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
	I know about the following PCI registers.

	For the VIA KT333, 8235 south bridge:
	> > enable:
	> > setpci -v -H1 -s 0:0.0 95=1e (HALT command detect) [Setting this will cool the CPU as long as the OS issues HLT instructions while idle]
	> > setpci -v -H1 -s 0:0.0 92=e9 (Disc When STPGNT# Detect.) [Not used in this program]
	> > setpci -v -H1 -s 0:0.0 70=86 (PCI Master Read Buffering) [Enable if you have a problems with certian devices while cooling]
	> >
	> > disable:
	> > setpci -v -H1 -s 0:0.0 70=82 (PCI Master Read Buffering) [See above]
	> > setpci -v -H1 -s 0:0.0 92=69 (Disc When STPGNT# Detect.) [Not used in this program]
	> > setpci -v -H1 -s 0:0.0 95=1c (HALT command detect) [Setting this will cool the CPU as long as the OS issues HLT instructions while idle]
*/

#include <sys/types.h>

#include <signal.h>
#include <sched.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <time.h>
#include <fcntl.h>

#include <sys/time.h>
#include <sys/resource.h>
#include <sys/io.h>
#include <sys/stat.h>
#include <sys/ioctl.h>

#undef USE_STATGRAB /* Use the libstatgrab library */
#undef NODAEMON /* Become a daemon? */
#undef ENABLE_HLT_DETECT /* Enable HLT detect? */

#define TEMP_POLL_INTERVAL 1500000000 /* Number of nano (billionth) seconds to sleep between checking temperature */

#define IDLE_PERCENT 85 /* Percent idle necessary before considering 'idle' */
#define HOTCOUNTER_RANGE 40 /* Number in intervals of TEMP_POLL_INTERVAL to remain in one state */

#ifdef USE_STATGRAB
#include <statgrab.h>
#endif

struct mobotemp
{
	float temp;
	float warn;
	float danger;
};

static volatile unsigned char killflag = 0;

static unsigned char iscooling = 0;
static unsigned char dangermode = 0;
/* WARNING: When USE_STATGRAB is defined the value of hotcounter is not well defined, but is usually HOTCOUNTER_RANGE/2. */
static unsigned int hotcounter = HOTCOUNTER_RANGE/2;
static int cooldhlt_fd = -1;

static inline void nano_to_timespec(struct timespec *ts, unsigned long long ns)
{
	if(!ts)
		return;
	ts->tv_sec = (int)(ns/1000000000);
	ts->tv_nsec = (int)(ns%1000000000);
}

void setkillflag(int n)
{
	killflag = 1;
}

void send_cool_cmd()
{
  if(iscooling)
    return;
  
  iscooling = 1;

#ifdef ENABLE_HLT_DETECT
	/* This function should be replaced with actual code that sets the PCI registers */
	system("setpci -H1 -s 0:0.0 95=1e");
#ifdef NODAEMON
	printf("Cooling Enabled\n");
#endif
#endif
}

void send_hott_cmd()
{
  if(!iscooling)
    return;

  iscooling = 0;

#ifdef ENABLE_HLT_DETECT
	/* This function should be replaced with actual code that sets the PCI registers */
	system("setpci -H1 -s 0:0.0 95=1c");
#ifdef NODAEMON
	printf("Cooling Disabled\n");
#endif
#endif
}

void hlt_loop()
{
  /* This is where all the magic happens */
#ifdef NODAEMON
  fprintf(stdout, "Begin HLT\n");
  fflush(stdout);
#endif
  ioctl(cooldhlt_fd, 0, 5000);
#ifdef NODAEMON
  fprintf(stdout, "End HLT\n");
  fflush(stdout);
#endif
}

void hlt_on_temp(const struct mobotemp *cpCPU, const struct mobotemp *cpcase)
{
#ifdef NODAEMON
	/* For debugging new temp sensors */
	printf("Case: %.1f %.1f %.1f\n", cpcase->temp, cpcase->warn, cpcase->danger);
	printf("CPU: %.1f %.1f %.1f\n", cpCPU->temp, cpCPU->warn, cpCPU->danger);
#endif

#ifdef USE_STATGRAB
  cpu_percent_t *cpu_use = cpu_percent_usage();
  if(cpu_use->idle >= IDLE_PERCENT)
  {
#ifdef NODAEMON
	printf("Idle\n");
#endif
    if(hotcounter > 0)
      hotcounter--;
  }
  else
  {
#ifdef NODAEMON
	printf("Not Idle\n");
#endif
    if(hotcounter < HOTCOUNTER_RANGE)
      hotcounter++;
  }
#endif
  
	/* HLT when in a danger state	*/
	if((dangermode && ((cpCPU->temp >= cpCPU->warn) || (cpcase->temp >= cpcase->warn))) || (cpCPU->temp >= cpCPU->danger) || (cpcase->temp >= cpcase->danger))
	{
    dangermode = 1;
    
    /* Force cooling */
    send_cool_cmd();
		hlt_loop();
	}
	/* Go to sleep and let programs run */
	else
	{
    dangermode = 0;

    if(hotcounter == 0)
      send_cool_cmd();
    else if(hotcounter == HOTCOUNTER_RANGE)
      send_hott_cmd();
    
		struct timespec ts;
		
		nano_to_timespec(&ts, TEMP_POLL_INTERVAL);
		nanosleep(&ts, NULL);
	}
}

/*
	The temps should really be stored as an array of mobotemp's
	with some added information. Using libsensors.
*/

/*
  This function is not portable, it should be replaced.
*/
int get_cpu_temp(struct mobotemp *pmt)
{
	int retval = 0;
	FILE *temp_proc;

	if(!(temp_proc = fopen("/proc/sys/dev/sensors/w83697hf-isa-0290/temp2", "r")))
		return -1;

	if(fscanf(temp_proc, "%f %f %f", &pmt->danger, &pmt->warn, &pmt->temp) != 3)
		retval = -1;

	fclose(temp_proc);

#if 0
  /* Manually Configure the CPU warn and danger temp */
	pmt->danger = 51;
	pmt->warn = 51;
#endif
	
	return retval;
}

/*
  This function is not portable, it should be replaced.
*/
int get_case_temp(struct mobotemp *pmt)
{
	int retval = 0;
	FILE *temp_proc;

	if(!(temp_proc = fopen("/proc/sys/dev/sensors/w83697hf-isa-0290/temp1", "r")))
		return -1;

	if(fscanf(temp_proc, "%f %f %f", &pmt->danger, &pmt->warn, &pmt->temp) != 3)
		retval = -1;

	fclose(temp_proc);

#if 0
  /* Manually Configure the case warn and danger temp */
	pmt->danger = 45;
	pmt->warn = 44;
#endif
  
	return retval;
}

int main(void)
{
	if(getuid())
	{
		fprintf(stderr, "Must be root!\n");
		return 1;
	}

	/* Create a realtime FIFO process with the highest static priority */
	struct sched_param sp;
	
	sp.sched_priority = sched_get_priority_max(SCHED_FIFO);
	
	if(sched_setscheduler(getpid(), SCHED_FIFO, &sp))
	{
		fprintf(stderr, "Unable to set priority!\n");
		return 1;
	}

	signal(SIGTERM, setkillflag);
	signal(SIGINT, setkillflag);

	if((cooldhlt_fd = open("/dev/coold", O_RDWR)) == -1)
  {
    fprintf(stderr, "Unable to open /dev/coold\n");
		return 1;
  }

#ifdef USE_STATGRAB
  /* Initialize the statgrab library */
	if(statgrab_init())
  {
		fprintf(stderr, "Failed to initialize statgrab library!\n");
		return 1;
  }
#endif

#ifndef NODAEMON
	if(daemon(0, 0))
	{
		fprintf(stderr, "Error daemonizing\n");
		return 1;
	}
#endif
  
	/* Set PCI registers that enable cooling */
  send_cool_cmd();

#ifdef USE_STATGRAB
  /* Throwaway junk CPU percentage */
  cpu_percent_usage();
#endif

  struct timespec ts;
  
  nano_to_timespec(&ts, TEMP_POLL_INTERVAL);
  nanosleep(&ts, NULL);

	while(!killflag)
	{
		struct mobotemp mtCPU, mtcase;

		if(!(get_cpu_temp(&mtCPU) || get_case_temp(&mtcase))) {
			hlt_on_temp(&mtCPU, &mtcase);
		}
	}

	/* Set PCI registers that disable cooling */
  send_hott_cmd();

 	close(cooldhlt_fd);
  
	return 0;
}
