/*
	cooldhlt.c - coold kernel module
	
	Interface to Kernel idle loop.
*/

/*
	Copyright (C) 2004-2005 Brian Gunlogson

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* Change this if it conflicts */
#define COOLD_MAJOR 251

#include <linux/module.h>
#include <linux/version.h>
#include <linux/sched.h>
#include <linux/fs.h>
#include <asm/uaccess.h>

static int coold_open_count = 0;
static spinlock_t coold_lock = SPIN_LOCK_UNLOCKED;

/* duration argument is in milliseconds */
static void coold_do_idle_loop(unsigned long duration)
{
  /* Check if HLTing works okay */
  if(/*current_cpu_data*/boot_cpu_data.hlt_works_ok)
  {
    /* FIXME: Should use floating point */
    unsigned long endtime = jiffies + ((HZ * duration)/1000);
    /* FIXME: Handle jiffies overflow. */
    /* Run the HLT loop for the specified duration */
    while(jiffies < endtime)
    {
      local_irq_disable();
      safe_halt();
    }
  }
}

static int coold_open(struct inode *inode, struct file *filep)
{
  spin_lock(&coold_lock);
  /* only allow 1 instance */
  if(coold_open_count >= 1)
  {
    spin_unlock(&coold_lock);
    return -ENOMEM;
  }
  coold_open_count++;
  spin_unlock(&coold_lock);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
  MOD_INC_USE_COUNT; /* Do not allow module to be removed. */
#endif

  return 0;
}

static int coold_release(struct inode *inode, struct file *filep)
{
  spin_lock(&coold_lock);
  if(coold_open_count <= 0)
  {
    spin_unlock(&coold_lock);
    return -ENOMEM;
  }
  coold_open_count--;
  spin_unlock(&coold_lock);

#if LINUX_VERSION_CODE < KERNEL_VERSION(2,6,0)
	MOD_DEC_USE_COUNT; /* Allow module to be removed. */
#endif

  return 0;
}

static int coold_ioctl(struct inode *inode, struct file *filep, unsigned int cmd, unsigned long arg)
{
  int ret = -ENOIOCTLCMD;

  /*
  Supported IOCTLs
  
  Run kernel idle loop for the specified duration.
    cmd = 0
    arg = duration to HLT in milliseconds
  */
  
  switch(cmd)
  {
    case 0:
      coold_do_idle_loop(arg);
      ret = 0;
      break;
  }

  return ret;
}

static struct file_operations coold_fops = {
  .owner   = THIS_MODULE,
  .ioctl   = coold_ioctl,
  .open    = coold_open,
  .release = coold_release,
};

static int __init init_coold(void)
{
  int ret;

  ret = register_chrdev(COOLD_MAJOR, "coold", &coold_fops);
  if(ret < 0)
  {
    printk("coold: could not register character device %d\n", COOLD_MAJOR);
    return ret;
  }

	return 0;
}

static void __exit cleanup_coold(void)
{
	unregister_chrdev(COOLD_MAJOR, "coold");
}

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Brian Gunlogson");
MODULE_DESCRIPTION("coold /dev interface to Kernel idle loop");

module_init(init_coold);
module_exit(cleanup_coold);
