/*
Copyright (C)  2006  Daniele Zelante

This file is part of copyblock.

copyblock is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

copyblock is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with copyblock; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/
/*@LICENSE*/
// $Id$

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <fcntl.h>
#include <string.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <algorithm>
#include <iostream>
#include <iomanip>

#ifdef __FreeBSD__
#include <sys/ioctl.h>
#include <sys/disk.h> 
#endif

#include <comf/globals.hxx>
#include <comf/string.hxx>

using namespace std;
using namespace comf;



static const char * COPYRIGHT = "Copyright (C)  2005  Daniele Zelante ";


void banner()
{
	string msg = Format(
		"product : %$ \n"
		"version : %$.%$.%$ \n"
		"build   : %$ \n"
		"author  : [%$] \n"
		"email   : <zeldan@email.it> \n\n"
		)
		
		% MAKROZ_PRODUCTSTR
		% utod(MAKROZ_VERSION) % utod(MAKROZ_RELEASE) % utod(MAKROZ_BUILDNR)
		% MAKROZ_TIMESTAMP
		% zeldan()
	;
	
	cerr << msg;
}
	
void license()
{
	string msg = Format("license : GPLv2 \n\n%$\n\n%$\n")
		% COPYRIGHT
		% gpl();
		
	cerr << msg;
}

	


void syntax()
{
	string msg = Format("syntax : %$ {-r[z]|-w[s]} [-b bsize] [-m msize] [-V] filename \n") % MAKROZ_PRODUCTSTR;
	cerr << msg << endl;
	cerr << "-r[z]    : read from file and writes to stdout ('z' to write sparse)" << endl;
	cerr << "-w[s]    : write to file reading from stdin ('s' to write sync)" << endl;
	cerr << "-b bsize : set blocksize in KiB, defaults to 64 KiB" << endl;
	cerr << "-m msize : set maximum size in KiB to read or write" << endl;
	cerr << "-V       : display version info" << endl;
	cerr << "-L       : display license info" << endl;
	cerr << "filename : block device or regular file to work on" << endl;
	cerr << endl;
	exit(3);
}


void myfatal(const char * str)
{
	cerr << str << endl;
	exit(2);
}
	
void myerror(const char * str)
{
	perror(str);
	exit(1);
}

void logerror(const char * str)
{
	perror(str);
}
		
size_t read_n(const int fd, char * buf, const size_t count)
{
	size_t done = 0;
	while(done<count)
	{
		ssize_t rd = read(fd,buf+done,count-done);
		if(rd==-1) logerror("\nerror: read");
		if(rd<=0) break;
		size_t rdu = static_cast<size_t>(rd);
		assert(rdu<=count);
		if(rdu < count-done)
		{
			string msg = Format("warn: slow read (%$ < %$)") %utod(rdu) %utod(count-done);
			cerr << endl << msg << endl;
		}
		done += rdu;
	}
	return done;
}

size_t write_n(const int fd, const char * buf, const size_t count)
{
	size_t done = 0;
	while(done<count)
	{
		ssize_t wr = write(fd,buf+done,count-done);
		if(wr==-1) logerror("\nerror: write");
		if(wr<=0) break;
		size_t wru = static_cast<size_t>(wr);
		if(wru < count-done) 
		{
			string msg = Format("warn: slow write (%$ < %$)") %utod(wru) %utod(count-done);	
			cerr << endl << msg << endl;
		}
		done += wru;
	}
	return done;
}




void printsize(const off_t size)
{
	const double sizeK = size/1024.0;
	const double sizeM = sizeK/1024.0;
	const double sizeG = sizeM/1024.0;
	
	cerr << fixed << setprecision(2);
	cerr << "size = " << size<<" B | " << sizeK<<" KiB | " << sizeM<<" MiB | " << sizeG<<" GiB" << endl;
}

void printprogress(time_t timebase, const off_t total, const off_t part, off_t * s0, time_t * t0)
{
	const time_t t1 = time(0);

	if(t1!=*t0)
	{
		const int progress = static_cast<int>((100.0*part) / total + 0.5);
		const double speed = static_cast<double>(part-*s0)/(t1-*t0);
		const double speedM = speed/(1024.0*1024.0);

		const double avgspeed = static_cast<double>(part)/(t1-timebase);
		const double avgspeedM = avgspeed/(1024.0*1024.0);

		int eta = static_cast<int>((total-part) / avgspeed);
		const int etaD = eta / (60*60*24);
		eta -= etaD * (60*60*24);
		const int etaH = eta / (60*60);
		eta -= etaH * (60*60);
		const int etaM = eta / 60;
		eta -= etaM * 60;
		const int etaS = eta;
		
		fprintf(stderr,"\r %2d%% : ETA = %dd,%02dh,%02dm,%02ds : AVG_RATE = %.02f MiB/s : RATE = %.02f MiB/s     \r",
			progress,etaD,etaH,etaM,etaS,avgspeedM,speedM);
	
		*s0 = part;
		*t0 = t1;
	}

}

struct fileinfo_t
{
	fileinfo_t() : _ok(false), _regular(false), _sizeable(false), _size(0) {}

	bool _ok;	
	bool _regular;
	bool _sizeable;
	off_t _size;
};


fileinfo_t fileinfo(const int fd)
{
	fileinfo_t fi;
	
	struct stat s;
	if(-1==fstat(fd,&s)) myerror("fstat");

	fi._ok = S_ISREG(s.st_mode) || S_ISBLK(s.st_mode) || S_ISCHR(s.st_mode);
	fi._regular = S_ISREG(s.st_mode);
	fi._size = s.st_size;
	if(fi._size) fi._sizeable = true;

	#ifdef __FreeBSD__
	if(fi._ok && !fi._regular && !fi._sizeable)
	{
		if(-1!=ioctl(fd,DIOCGMEDIASIZE,&fi._size))
			fi._sizeable = true;
	}
	#endif


	if(!fi._sizeable)
	{
		bool ok = true;
		off_t x = lseek(fd,0,SEEK_END);
		if(x==-1) ok=false;
		if(-1==lseek(fd,0,SEEK_SET)) ok=false;
		if(ok)
		{
			fi._size = x;
			if(fi._size) fi._sizeable = true;	
		}
	}

	return fi;
}






bool isdirty(const char * buf, size_t len)
{
	size_t n;
	for(n=0; n<len; ++n)
		if(buf[n]) return true;
	return false;
}

void blockread(const char * filename, const size_t blocksize, const off_t maxsize, bool sparse)
{
	fileinfo_t nfo1 = fileinfo(1);
	if(!nfo1._regular && sparse) 
	{
		cerr << "warn: ignoring [-z] when writing to non-regular file " << endl;
		sparse = false;
	}
	
	const int fdi = open(filename,O_RDONLY);
	if(fdi==-1) myerror("open");
	
	fileinfo_t fdinfo = fileinfo(fdi);
	if(!fdinfo._ok) myfatal("invalid input file type");
	
	off_t srcsize = 0;
		
	if(!fdinfo._sizeable)
	{
		if(maxsize)
			srcsize = maxsize;
		else
			myfatal("[-m size] is mandatory when source file size is unknown");
	}
	else
	{
		srcsize = fdinfo._size;
			
		if(maxsize)
		{
			if(maxsize > srcsize)
				cerr << "warn: requested size greater than real size " << endl;
			if(maxsize < srcsize)
			{
				string msg = Format("info: truncating size from %$ to %$")
					%utod(srcsize) %utod(maxsize);
				cerr << msg << endl;
				srcsize = maxsize;
			}
		}
	}
		
	printsize(srcsize);
	char * buffer = new char[blocksize];
	time_t t0 = time(0);
	time_t timebase = t0;
	off_t oldpart = 0;
	off_t part = 0;
	
	while(part<srcsize)
	{
		printprogress(timebase,srcsize,part,&oldpart,&t0);

		const size_t rdreq = static_cast<size_t>(std::min(static_cast<off_t>(blocksize), srcsize-part));
		const size_t rd = read_n(fdi, buffer, rdreq);
		
		if(rd<rdreq)
		{
			cerr << endl << "error: unexpected end of device" << endl;
			break;
		}

		part += rd;
		
		if(sparse && rd==blocksize && part<srcsize && !isdirty(buffer,rd))
		{
			if(-1==lseek(1,rd,SEEK_CUR)) 
				myerror("\nseek");
		}
		else
		{
			const size_t wr = write_n(1,buffer,rd);
			if(wr<rd)
			{
				cerr << endl << "error: unable to write" << endl;
				break;
			}
		}
		
	}

	delete[] buffer;
	
}


void blockwrite(const char * filename, const size_t blocksize, const off_t maxsize, int syncr)
{
	int fdo = open(filename,O_RDWR|(syncr ? O_SYNC : 0));
	if(fdo==-1) myerror("open");

	fileinfo_t fdonfo = fileinfo(fdo);
	if(!fdonfo._ok) myfatal("invalid output file type \n");
	
	fileinfo_t nfo0 = fileinfo(0);

	off_t devsize = 0;
	
	if(!fdonfo._sizeable && !nfo0._sizeable && !maxsize)
		myfatal("[-m size] is mandatory when both target and stream sizes are unknown");
		
	if(fdonfo._sizeable && nfo0._sizeable)
	{
		if(fdonfo._size!=nfo0._size)
		{
			string msg = Format("warn: target size is %$, stream size is %$, using smaller one")
				% utod(fdonfo._size)  % utod(nfo0._size);
			cerr << msg << endl;
		}		

		devsize=std::min(fdonfo._size,nfo0._size);
	}
	
	if(fdonfo._sizeable && !nfo0._sizeable) devsize=fdonfo._size;
	if(!fdonfo._sizeable && nfo0._sizeable) devsize=nfo0._size;
	

	if(maxsize)
	{
		if(maxsize > devsize)
			cerr << "warn: requested size greater than real size" << endl;
		if(maxsize < devsize)
		{
			string msg = Format("info: truncating size from %$ to $%")
				% utod(devsize)  % utod(maxsize);
			cerr << msg << endl;
			devsize = maxsize;
		}
	}


	printsize(devsize);
	char * buffer = new char[blocksize];
	time_t t0 = time(0);
	time_t timebase = t0;
	off_t oldpart = 0;
	off_t part = 0;

	while(part<devsize)
	{
		printprogress(timebase,devsize,part,&oldpart,&t0);
		
		const size_t rdreq = static_cast<size_t>(std::min(static_cast<off_t>(blocksize), devsize-part));
		const size_t rd = read_n(0, buffer, rdreq);
		if(rd<rdreq)
		{
			cerr << endl << "error: unexpected end of file" << endl;
			break;
		}

		part += rd;

		size_t wr = write_n(fdo, buffer, rd);
		if(wr<rd)
		{
			cerr << endl << "error: unable to write" << endl;
			break;
		}
				
		if(rd<blocksize) break;

	}

	delete[] buffer;
}	


int main(int argc, char ** argv)
{
	assert(sizeof(off_t)==8);
		
	const char * optstring = "LVrwszb:m:";
	bool writeop = false;
	bool readop = false;
	bool syncr = false;
	bool sparse = false;
	size_t blocksize = 0;
	off_t maxsize = 0;
	
	for(;;)
	{
		const int opt = getopt(argc,argv,optstring);
		if(opt==-1) break;
		if(opt=='V') banner();
		if(opt=='L') license();
		if(opt=='r') readop = true;
		if(opt=='w') writeop = true;
		if(opt=='s') syncr = true;
		if(opt=='z') sparse = true;
		if(opt=='b')
		{
			if(blocksize!=0) syntax();
			blocksize = atoi(optarg) * 1024;
			if(blocksize==0) syntax();
		}
	
		if(opt=='m')
		{
			if(maxsize!=0) syntax();
			maxsize = (static_cast<int64_t>(atoi(optarg))) * 1024;
			if(maxsize==0) syntax();
		}
	}


	if(optind+1!=argc) syntax();
	const char * filename = argv[optind];
	
	if(readop==writeop) syntax();
	if((sparse && writeop) || (readop && syncr)) syntax();
	
	if(blocksize==0) blocksize=64*1024;

	
	{
		std::string msg = comf::Format("%$ %$ ; blocksize %$ KiB")
			% (readop ? "reading" : "writing")
			% comf::quote(filename)
			% comf::utod(blocksize/1024);
		fprintf(stderr,msg.data());
	}
		

	if(maxsize)
		cerr << " ; maxsize " << (maxsize/1024) << " KiB";
	if(syncr)
		cerr << " ; sync";
	
	cerr << endl;

	if(readop)
		blockread(filename,blocksize,maxsize,sparse);
	if(writeop)
		blockwrite(filename,blocksize,maxsize,syncr);

	cerr << endl << "OK" << endl;
	
	return 0;
}
	
