/* 
    Copyright (C) 2024, 2025, Johannes Merten <coldemail@posteo.net>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <https://www.gnu.org/licenses/>.
    */

declare name "cptdronesynth";
declare description "drone synth";
declare author "Johannes Merten";
declare copyright "Johannes Merten (coldemail@posteo.net)";
declare version "0.19";
declare license "GPL 3.0 or later";


si = library("signals.lib");
ba = library("basics.lib");
ma = library("maths.lib");
en = library("envelopes.lib");
ef = library("misceffects.lib");
ho = library("hoa.lib");
os = library("oscillators.lib");
no = library("noises.lib");
fi = library("effect.lib");

constant = environment {
	     feedbackhighpassfactor = 0.9;
	   };

gate = button("[-1]gate");
gain=hslider("[00]gain",1,0,1,0.01);

oscGroup(x) = hgroup("[02]OSCs",x);
s = oscGroup(hslider("[0]os 1[lv2:integer] Off,Sawtooth, Square,  Sparse-Noise, Noise",0,0,4,2));
t = oscGroup(hslider("[1]os 2[lv2:integer] Off,  Sawtooth, Square,  Sparse-Noise, Noise",0,0,4,2));
l = oscGroup( hslider("[2]os 3[lv2:integer] Off, Sawtooth, Square, Triangle, Noise",0,0,4,2));

balanceGroup(x) = hgroup("[03]Balance",x);
osabal = balanceGroup (hslider("[0][lv2:integer]os 1 balance",0,-63,63,1));
osbbal = balanceGroup (hslider("[1][lv2:integer]os 2 balance",0,-63,63,1));
oscbal = balanceGroup (hslider("[2][lv2:integer]os 3 balance",0,-63,63,1));

mixGroup(x) = hgroup("[04]Mix",x);
osmix = mixGroup (hslider("[0][lv2:integer]os mix",50,0,100,1)/100);
attrel = mixGroup (hslider("[1][lv2:integer]os attack/release",1,0,127,1));
moddepth    = mixGroup (hslider("[2]os 3 modulation index",0,0,1000,0.1));

mixctFreq(x) = hgroup("[05]Cutoff freq",x);
osactfreq = mixctFreq (hslider("[0][unit:Hz][lv2:integer]os 1 lowpass/highpass", 0, -9000, 9000, 1));
osbctfreq = mixctFreq (hslider("[1][unit:Hz][lv2:integer]os 2 lowpass/highpass", 0, -9000, 9000, 1));
oscctfreq = mixctFreq (hslider("[2][unit:Hz][lv2:integer]os 3 lowpass/highpass", 0, -9000, 9000, 1));

noteGroup(x) = hgroup("[06]Note",x);
freq  =  noteGroup (hslider("[0][lv2:integer]key",60,0,127,1)) :  si.smooth(ba.tau2pole(glissandosec) ) : ba.midikey2hz;
glissandosec = noteGroup (hslider("[1][lv2:integer]glissando",0,0,60,0.1));
pitch = noteGroup (hslider("[2][unit:Hz][lv2:integer]os pitch",0,-63,+63,1));

oscGroup(x) = hgroup("[07]os 3",x);
bpm= oscGroup (hslider("[0][lv2:integer]os 3/feedback bpm",60,0,127,1));
oscshiftslider = oscGroup (hslider("[2][lv2:integer]os 3 shift",0,-63,63,1));


effectsGroup(x) = hgroup("[08]Effects",x);
feedback = effectsGroup (hslider("[1][lv2:integer]feedback",0,0,100,1)/99);
feedbackshift = effectsGroup (hslider("[2][lv2:integer]feedback shift",0,-24,24,1));

//all os's     
//os1-4 for os 1/2
//os5-8 for os 3
os1 = os.sawtooth(freq+pitch+(osc * moddepth));
os2 = os.square(freq+pitch+(osc * moddepth));
os3 = no.sparse_noise(freq+pitch+(osc * moddepth));
os4 =  no.noise;
os5 =  os.sawtooth(freq+pitch);
os6 = os.square(freq+pitch);
os7 = os.triangle(freq+pitch);
os8 =  no.noise;

//transpose os via "shift" slider
oscshift = ef.transpose(24,24,oscshiftslider);

// trigger attack and release of os
osexec(os,trigger,triggernumber)=  en.asr(attrel,1,attrel,(trigger==triggernumber))* os;

//os 1
osa = osexec(os1,s,1)
       + osexec(os2,s,2)
       + osexec(os3,s,3)
       + osexec(os4,s,4)
       :>/(4)<:attach(_,abs : ba.linear2db : hbargraph("[09]os 1 out",-60,0));

//os 2
osb = osexec(os1,t,1)
       + osexec(os2,t,2)
       + osexec(os3,t,3)
       + osexec(os4,t,4)
       :>/(4)<:attach(_,abs : ba.linear2db : hbargraph("[10]os 2 out",-60,0));

//os 3
osc = osexec(os5,l,1) : oscshift
       + osexec(os6,l,2) : oscshift
       + osexec(os7,l,3) : oscshift
       + osexec(os8,l,4) : oscshift
       :>/(4)<:attach(_,abs : ba.linear2db : hbargraph("[11]os 3 out",-60,0));

//pulse sound for os 3
pulse = en.arfe(atrt, atrt, (1-finallevel),time) 
with {
    finallevel= oscGroup (hslider("[1]os 3 level",1,0.1,1,0.01));
    atrt = ba.samp2sec(ma.SR)*60/bpm;
    time =  ba.pulsen(ma.SR*60/bpm,(ma.SR*60/bpm)*2);
};

//feedback
feedbackeffect = +~@(delay) * feedback : feedbackhighpass 
with {
  delay = (bpm + feedbackshift,1:max)  / 60 * ma.SR;
  feedbackhighpass = _ <: (fi.highpass(1,freq)) , (_) :> _
  with {
    freq = 10000 * constant.feedbackhighpassfactor;
  };
};

//balance calculation
balanceleft(osbal) = ((osbal/63) * (-1)) + 1 : (sqrt);
balanceright(osbal) = ((osbal / 63)) + 1 : (sqrt);

osaballeft = balanceleft(osabal);
osabalright = balanceright(osabal);

osbballeft = balanceleft(osbbal);
osbbalright = balanceright(osbbal);

oscballeft = balanceleft(oscbal);
oscbalright = balanceright(oscbal);

//high- and lowpass calculation
lowpass(freq) = fi.lowpass(1, (10000+freq));
highpass(freq) = fi.highpass(1, ( (freq >0) * freq)+1);

//process the os's, pulse for os 3, gain, balance, low- and highpass and feedback for channel 1/2

process =
//left channel
           osc  * pulse *(1-osmix) * gain  * oscballeft
           :lowpass(oscctfreq):highpass(oscctfreq)

	+ (osa * osmix/2 * gain  * osaballeft
	   :lowpass(osactfreq):highpass(osactfreq))

        +(osb * osmix/2  * gain  * osbballeft
	  :lowpass(osbctfreq) :highpass(osbctfreq))

	    //feedback left channel
	    : feedbackeffect

	     //bar left channel
	     <:attach(_,abs : ba.linear2db : hbargraph("[12]total out left",-60,0))/2,
	
//right channel
	   (
	     osc  * pulse *(1-osmix) *gain * oscbalright
	     :lowpass(oscctfreq) :highpass(oscctfreq)

	  + (osa * osmix/2 * gain *osabalright 
     	     :lowpass(osactfreq) :highpass(osactfreq))
	  
	  +(osb * osmix/2  * gain  *osbbalright
	    :lowpass(osbctfreq) :highpass(osbctfreq))

	     //feedback right channel
	     : feedbackeffect
	       //bar right channel
	       <:attach(_,abs : ba.linear2db : hbargraph("[13]total out right",-60,0))/2
);
