/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dom/cria-slide-priv.h>

#include <inttypes.h>
#include <string.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <gdk/gdk.h>

#include <cdebug/cdebug.h>

#include "background.h"

enum {
	PROP_0,
	PROP_BACKGROUND,
	PROP_COMMENT,
	PROP_TITLE,
	PROP_MASTER_SLIDE,
	PROP_THEME
};

enum {
	SIGNAL,
	N_SIGNALS
};

static GObjectClass* parent_class = NULL;

static void	  cria_slide_get_property      (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void	  cria_slide_init	       (CriaSlide	* self);
static void cria_slide_must_have_background    (CriaSlide	* self);
static void	  cria_slide_set_property      (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
static	guint	cria_slide_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_signal	       (CriaSlide	* template,
						const	gchar	* string);
#endif

/**
 * cria_slide_add_block:
 * @self: a #CriaSlide
 * @block: a #CriaBlock
 *
 * Adds a #Block to this slide
 */
void
cria_slide_add_block(CriaSlide* self, CriaBlock* block) {
	g_return_if_fail(CRIA_IS_SLIDE(self));
	g_return_if_fail(CRIA_IS_BLOCK(block));
	g_return_if_fail(cria_block_get_name(block) != NULL);
        g_return_if_fail(g_hash_table_lookup(self->blocks, (cria_block_get_name(block))) == NULL);
	/* FIXME: check that the block is defined in the layout
	 * this shouldn't be neccessary with the new rendering
	 * framework*/

	g_hash_table_insert(self->blocks,
			    g_strdup(cria_block_get_name(block)),
			    block);
}

static void
cs_finalize(GObject* object) {
	CriaSlide* self = CRIA_SLIDE(object);

	g_hash_table_destroy(self->blocks);
	self->blocks = NULL;

	if(self->master_slide) {
		g_object_unref(self->master_slide);
	}

	if(self->theme) {
		g_object_unref(self->theme);
	}

	if(self->comment) {
		g_free(self->comment);
	}

	if(self->background) {
		g_object_unref(self->background);
	}

	if(self->presentation) {
		g_object_remove_weak_pointer(G_OBJECT(self->presentation), (gpointer*)&(self->presentation));
	}
}

static void
cria_slide_class_init(CriaSlideClass* cria_slide_class) {
	GObjectClass	* g_object_class;

	parent_class = g_type_class_peek_parent(cria_slide_class);

	/* initialize the object class */
	g_object_class = G_OBJECT_CLASS(cria_slide_class);
	g_object_class->finalize = cs_finalize;
	g_object_class->get_property = cria_slide_get_property;
	g_object_class->set_property = cria_slide_set_property;

	g_object_class_install_property(g_object_class,
					PROP_BACKGROUND,
					g_param_spec_object("background",
							    "Slide Background",
							    "The background for this slide",
							    CRIA_TYPE_BACKGROUND,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_COMMENT,
					g_param_spec_string("comment",
							    "Comment",
							    "Some descriptive comment for the slide, typically used for "
							    "master slides",
							    "",
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_MASTER_SLIDE,
					g_param_spec_object("master-slide",
							    "Master Slide",
							    "A Slide that specifies drawing details",
							    CRIA_TYPE_SLIDE,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_TITLE,
					g_param_spec_string("title",
							    "Title",
							    "The title of that slide",
							    _("untitled"),
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_THEME,
					g_param_spec_object("theme",
							    "Theme",
							    "The theme associated with this slide",
							    CRIA_TYPE_THEME,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

/**
 * cria_slide_get_background:
 * @self: a #CriaSlide
 * @no_recurse: tell the function whether to recuse through master slides or not
 *
 * Get the background of a slide.
 *
 * Returns the #CriaBackground of a slide.
 */
CriaBackground*
cria_slide_get_background(CriaSlide* self, gboolean no_recurse) {
	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);

	if(!no_recurse && !self->background && self->master_slide && cria_slide_get_background(self->master_slide, FALSE)) {
		return cria_slide_get_background(self->master_slide, FALSE);
	}

	if(no_recurse && !self->background) {
		cria_slide_must_have_background(self);
	}

	cdebugo(self, "getBackground()", "returning background 0x%x", (uintptr_t)self->background);
	return self->background;
}

CriaBlock*
cria_slide_get_block(CriaSlide* self, const gchar* block) {
	CriaBlock * cblock;
	
	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);
	g_return_val_if_fail(block != NULL, NULL);

	/* search for the block in here */
	cblock = cria_slide_get_block_no_recurse(self, block);

	if(!cblock && CRIA_IS_SLIDE(self->master_slide)) {
		cblock = cria_slide_get_block(self->master_slide, block);
	}

	cdebugo(self, "getBlock()", "self (0x%x) returns \"%s\" 0x%x now", (uintptr_t)self, block, (uintptr_t)cblock);
	return cblock;
}

CriaBlock*
cria_slide_get_block_no_recurse(CriaSlide* self, const gchar* block) {
	CriaBlock	* cblock;

	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);
	g_return_val_if_fail(block != NULL, NULL);
	
	cblock = CRIA_BLOCK(g_hash_table_lookup(self->blocks, block));
	cdebugo(self, "getBlockNoRecurse()", "\"%s\" is 0x%x", block, (uintptr_t)cblock);
	return cblock;
}

struct vector {
	gchar	**vector;
	guint	  length;
};

static void
populate_array_with_keys(gpointer key, gpointer value, gpointer user_data) {
	struct vector* v = (struct vector*)user_data;
	gchar	* skey  = (gchar*)key;
	guint	  i;

	if(!v) {
		return;
	}

	for(i = 0; v->vector[i]; i++) {
		if (!strcmp(v->vector[i], skey)) {
			/* we already contain that key */
			return;
		}
		/* stepping to the first NULL */
	}

	if(i == (v->length - 1)) {
		int j;
		
		v->length += 10;
		v->vector = g_renew(gchar*,v->vector,v->length);

		for(j = i; j < v->length; j++) {
			v->vector[j] = NULL;
		}
	}
	
	v->vector[i] = g_strdup(skey);
}

static void
cria_slide_get_block_namesv(CriaSlide* self, struct vector* v) {
	if(CRIA_IS_SLIDE(self->master_slide)) {
		cria_slide_get_block_namesv(self->master_slide, v);
	}

	g_hash_table_foreach(self->blocks,
			     populate_array_with_keys,
			     v);
}

/**
 * cria_slide_get_block_names:
 * @self: the slide to get the block names from
 *
 * Get the names of blocks from this slide.
 *
 * Returns a string array of block names. Free it after using with g_strfreev()
 */
gchar**
cria_slide_get_block_names(CriaSlide* self) {
	gchar	**names = NULL;
	struct vector* v = g_new0(struct vector, 1);
	
	/* we will need at least that much space */
	v->length = g_hash_table_size(self->blocks) + 1;
	v->vector = g_new0(gchar*,v->length);

	cria_slide_get_block_namesv(self, v);

	names = v->vector;
	g_free(v);

	return names;
}

/**
 * cria_slide_get_master_slide:
 * @self: The slide to the get master slide from
 *
 * The master slide defines several rendering details that the slide doesn't
 * need to contain (because it's shared across all similar looking slides).
 * This method returns the master slide of a slide or NULL if the given slide
 * is a master slide.
 *
 * Returns the master slide of a slide
 */
#warning "Slide::getMasterSlide(): FIXME: Think about this: can we have a tree of master slides so NULL may even be returned when the given slide is a master slide?"
CriaSlide*
cria_slide_get_master_slide(CriaSlide* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);
	
	return self->master_slide;
}

static void
cria_slide_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlide	* self;

	self = CRIA_SLIDE(object);

	switch(prop_id) {
	case PROP_BACKGROUND:
		g_value_set_object(value, self->background);
		break;
	case PROP_COMMENT:
		g_value_set_string(value, self->comment);
		break;
	case PROP_MASTER_SLIDE:
		g_value_set_object(value, self->master_slide);
		break;
	case PROP_TITLE:
		g_value_set_string(value, self->title);
		break;
	case PROP_THEME:
		g_value_set_object(value, self->theme);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

/**
 * cria_slide_get_size:
 * @self: a #CriaSlide
 *
 * ...
 *
 * Returns a #GoPoint defining the size of the slide in master coordinates
 * (576dpi). Don't forget to g_free it.
 */
GoPoint*
cria_slide_get_size(CriaSlide* self) {
#warning "Slide::getSize(): FIXME return value from presentation"
	GoPoint* size = g_new0(GoPoint,1);
	size->x = 5760;
	size->y = 4320;
	return size;
}

CriaTheme*
cria_slide_get_theme(CriaSlide* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);
	
	return self->theme;
}

const char*
cria_slide_get_title(CriaSlide* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->title;
}

static void
cria_slide_init(CriaSlide* self) {
	cdebugo(self, "init()", "initializing Slide at 0x%x", (uintptr_t)self);
	
	self->blocks = g_hash_table_new_full(g_str_hash,
						   g_str_equal,
						   g_free,
						   g_object_unref);
}

/**
 * cria_slide_new:
 * @presentation: a #CriaSlideList
 *
 * Creates a new empty slide in @presentation at the end.
 *
 * Returns the new #CriaSlide.
 */
CriaSlide*
cria_slide_new(CriaSlideList* container) {
	return cria_slide_new_pos(container, -1);
}

/**
 * cria_slide_new_pos:
 * @presentation: a #CriaPresentation
 * @pos: a position
 *
 * Creates a new empty slide in @presentation at @pos.
 *
 * Returns the new #CriaSlide.
 */
CriaSlide*
cria_slide_new_pos(CriaSlideList* container, gint pos) {
	CriaSlide* self;

	g_return_val_if_fail(!container || CRIA_IS_SLIDE_LIST(container), NULL);
	
	self = g_object_new(CRIA_TYPE_SLIDE, NULL);

	if(container) {
		self->presentation = container;
		g_object_add_weak_pointer(G_OBJECT(container), (gpointer*)&(self->presentation));
		cria_slide_list_insert(container, self, pos);
	}

	return self;
}

/**
 * cria_slide_copy:
 */
CriaSlide*
cria_slide_copy(CriaSlide* slide, CriaSlideList* container, gint pos) {
	CriaSlide* self = cria_slide_new_pos(container, pos);

	/* copy properties from one slide to the other */
	return self;
}

/*
 * cria_slide_must_have_background:
 * @self: a #CriaSlide
 *
 * Ensures that @self has a valid background structure
 */
static void
cria_slide_must_have_background(CriaSlide* self) {
	g_return_if_fail(CRIA_IS_SLIDE(self));
	
	if(!self->background) {
		self->background = cria_background_new();
	}
}

/**
 * cria_slide_set_background:
 * @self: the slide to set the background for
 * @background: the background to be set
 *
 * Specify a background for a slide
 */
void
cria_slide_set_background(CriaSlide* self, CriaBackground* background) {
	g_return_if_fail(CRIA_IS_SLIDE(self));
	g_return_if_fail(!background || CRIA_IS_BACKGROUND(background));

	if(background == self->background) {
		return;
	}

	if(self->background) {
		g_object_unref(self->background);
		self->background = NULL;
	}

	if(background) {
		self->background = g_object_ref(background);
	}
	
	g_object_notify(G_OBJECT(self), "background");
}

/**
 * cria_slide_set_comment:
 * @self: The slide to set a comment for
 * @comment: The comment, a plain NUL terminated UTF8 string
 *
 * Set a comment for this slide. This is usually used for master slides
 * to set some descriptive text for a layout.
 */
void
cria_slide_set_comment(CriaSlide* self, const gchar* comment) {
	g_return_if_fail(CRIA_IS_SLIDE(self));
	g_return_if_fail(comment);

	cdebugo(self, "setComment()", "start");

	if(self->comment && !strcmp(comment, self->comment)) {
		return;
	}

	if(self->comment != NULL) {
		g_free(self->comment);
	}

	self->comment = g_strdup(comment);

	g_object_notify(G_OBJECT(self), "comment");

	cdebugo(self, "setComment()", "end");
}

/**
 * cria_slide_set_master_slide:
 * @self: the slide to set the master slide for
 * @master_slide: the slide to be set as the master slide
 *
 * Defines a slide containing common rendering details shared across several
 * slides.
 */
void
cria_slide_set_master_slide(CriaSlide* self, CriaSlide* master_slide) {
	g_return_if_fail(CRIA_IS_SLIDE(self));
	g_return_if_fail(!master_slide || CRIA_IS_SLIDE(master_slide));

	if(self->master_slide == master_slide) {
		return;
	}

	if(self->master_slide != NULL) {
		g_object_unref(self->master_slide);
	}

	if(master_slide) {
		self->master_slide = g_object_ref(master_slide);
	} else {
		self->master_slide = master_slide;
	}

	g_object_notify(G_OBJECT(self), "master_slide");
}

static void
cria_slide_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlide	* self = CRIA_SLIDE(object);
	
	switch(prop_id) {
	case PROP_BACKGROUND:
		cria_slide_set_background(self, g_value_get_object(value));
		break;
	case PROP_COMMENT:
		cria_slide_set_comment(self, g_value_get_string(value));
		break;
	case PROP_MASTER_SLIDE:
		cria_slide_set_master_slide(self, g_value_get_object(value));
		break;
	case PROP_TITLE:
		cria_slide_set_title(self, g_value_get_string(value));
		break;
	case PROP_THEME:
		cria_slide_set_theme(self, g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

void
cria_slide_set_theme(CriaSlide* self, CriaTheme* theme) {
	g_return_if_fail(CRIA_IS_SLIDE(self));
	g_return_if_fail(!theme || CRIA_IS_THEME(theme));

	if(self->theme != NULL) {
		g_object_unref(self->theme);
	}

	if(theme) {
		self->theme = g_object_ref(theme);
	} else {
		self->theme = theme;
	}

	g_object_notify(G_OBJECT(self), "theme");
}

void
cria_slide_set_title(CriaSlide* self, const gchar* title) {
	g_return_if_fail (CRIA_IS_SLIDE(self));

	if (self->title != NULL) {
		g_free (self->title);
	}

	cdebugo(self, "setTitle()", "\"%s\"", title);
	self->title = g_strdup(title);

	g_object_notify(G_OBJECT(self), "title");
}

static CriaSlide*
cs_get(CriaSlideList* container, guint num) {
	return g_list_nth_data(CRIA_SLIDE(container)->sub_slides, num);
}

static gint
cs_index(CriaSlideList* container, const CriaSlide* slide) {
	return g_list_index(CRIA_SLIDE(container)->sub_slides, slide);
}

static void
cs_insert(CriaSlideList* container, CriaSlide* slide, guint position) {
	g_list_insert(CRIA_SLIDE(container)->sub_slides, slide, position);
}

static guint
cs_n_slides(CriaSlideList* container) {
	return g_list_length(CRIA_SLIDE(container)->sub_slides);
}

static void
cs_container_init(gpointer interface) {
	CriaSlideListIface* iface = interface;

	iface->get      = cs_get;
	iface->index    = cs_index;
	iface->insert   = cs_insert;
	iface->n_slides = cs_n_slides;
}

G_DEFINE_TYPE_WITH_CODE(CriaSlide, cria_slide, G_TYPE_OBJECT, G_IMPLEMENT_INTERFACE(CRIA_TYPE_SLIDE_CONTAINER, cs_container_init));

