/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *       Ezequiel Prez       <eazel7@yahoo.com.ar>
 *
 * Copyright (C) 2004 Sven Herzberg
 * Copyright (C) 2004 Ezequiel Prez
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "presentation-writer.h"

#include <glib.h>
#include <libgnomevfs/gnome-vfs-uri.h>
#include <gsf-gnome/gsf-output-gnomevfs.h>
#include <gsf/gsf-libxml.h>
#include <goffice/utils/go-color.h>

#include <cdebug.h>

#include <dom/cria-format.h>

#include "presentation.h"
#include "theme.h"
#include <dom/cria-slide.h>

static	void	write_block_xml			(CriaBlock *self,
						 GsfXMLOut *xmlout);
static	void	write_slide_xml			(CriaSlide *self,
						 GsfXMLOut *xmlout,
						 gboolean theme);
static	void	write_theme_xml			(CriaTheme *self,
						 GsfXMLOut *xmlout);
static	void	write_presentation_xml		(CriaPresentation *self,
						 GsfXMLOut *xmlout);
static	void	write_theme_hfunc		(gpointer key,
						 gpointer value,
						 gpointer user_data);
static	void	write_master_slide_hfunc	(gpointer key,
						 gpointer value,
						 gpointer user_data);

static void
write_theme_xml(CriaTheme *self, GsfXMLOut *xmlout) {
	cdebug("PresentationWriter::writeThemeXml()", "start");
	gsf_xml_out_start_element(xmlout, "theme");
	cdebug("PresentationWriter::writeThemeXml()", "theme element created");
	
	gsf_xml_out_add_cstr(xmlout, "name", cria_theme_get_name(self));
 	cdebug("PresentationWriter::writeThemeXml()", "name attribute set");

	g_hash_table_foreach(cria_theme_get_master_slides(self),
			     write_master_slide_hfunc,
			     xmlout);
	cdebug("PresentationWriter::writeThemeXml()", "master slides written");
	gsf_xml_out_end_element(xmlout);
	cdebug("PresentationWriter::writeMasterSlideHFunc()", "finish");
}

static void
write_master_slide_hfunc(gpointer key, gpointer value, gpointer user_data) {
	CriaSlide *self = CRIA_SLIDE(value);
	GsfXMLOut *xmlout = GSF_XML_OUT(user_data);

	cdebug("PresentationWriter::writeMasterSlideHFunc()", "start");
	write_slide_xml(self, xmlout, TRUE);
	cdebug("PresentationWriter::writeMasterSlideHFunc()", "master slide written");
	cdebug("PresentationWriter::writeMasterSlideHFunc()", "finish");
}

static void
write_theme_hfunc(gpointer key, gpointer value, gpointer user_data) {
	CriaTheme *theme = CRIA_THEME(value);
	GsfXMLOut *xmlout = GSF_XML_OUT(user_data);

	cdebug("PresentationWriter::writeThemeHFunc()", "start");
	write_theme_xml(theme, xmlout);
	cdebug("PresentationWriter::writeThemeHFunc()", "theme written");
}

static void
write_block_xml(CriaBlock *self, GsfXMLOut *xmlout) {
	GoRect* rect;
	gchar * text;

	cdebug("PresentationWriter::writeBlockXML()", "started");
	gsf_xml_out_start_element(xmlout, "block");
	cdebug("PresentationWriter::writeBlockXML()", "block element created");
	gsf_xml_out_add_cstr(xmlout, "title", cria_block_get_name(self));
	cdebug("PresentationWriter::writeBlockXML()", "title attribute set");

#warning "writeBlockXml(): FIXME: Remember to reimplement the block color when the API changes"
	
	switch (cria_block_get_alignment (self)) {
		case CRIA_ALIGNMENT_LEFT:
			gsf_xml_out_add_cstr(xmlout, "align", "left");
			cdebug("PresentationWriter::writeBlockXML()", "align attribute set left");
			break;
		case CRIA_ALIGNMENT_CENTER:
			gsf_xml_out_add_cstr(xmlout, "align", "center");
			cdebug("PresentationWriter::writeBlockXML()", "align attribute set center");
			break;
		case CRIA_ALIGNMENT_RIGHT:
			gsf_xml_out_add_cstr(xmlout, "align", "right");
			cdebug("PresentationWriter::writeBlockXML()", "align attribute set right");
			break;
		case CRIA_ALIGNMENT_JUSTIFY:
			gsf_xml_out_add_cstr(xmlout, "align", "justify");
			cdebug("PresentationWriter::writeBlockXML()", "align attribute set justify");
			break;
		case CRIA_ALIGNMENT_UNSET:
			break;
	}

	switch (cria_block_get_valignment (self)) {
		case CRIA_ALIGNMENT_TOP:
			gsf_xml_out_add_cstr(xmlout, "valign", "top");
			cdebug("PresentationWriter::writeBlockXML()", "valign attribute set top");
			break;
		case CRIA_ALIGNMENT_MIDDLE:
			gsf_xml_out_add_cstr(xmlout, "valign", "middle");
			cdebug("PresentationWriter::writeBlockXML()", "valign attribute set middle");
			break;
		case CRIA_ALIGNMENT_BOTTOM:
			gsf_xml_out_add_cstr(xmlout, "valign", "bottom");
			cdebug("PresentationWriter::writeBlockXML()", "valign attribute set bottom");
			break;
		case CRIA_VALIGNMENT_UNSET:
			break;
	}

/*	rect = cria_block_get_position (self);
	gsf_xml_out_add_cstr (xmlout, "x", g_strdup_printf ("%lli%%", rect->left));
	cdebug("PresentationWriter::writeBlockXML()", "x attribute set");
	gsf_xml_out_add_cstr (xmlout, "y", g_strdup_printf ("%lli%%", rect->top));
	cdebug("PresentationWriter::writeBlockXML()", "y attribute set");
	gsf_xml_out_add_cstr (xmlout, "width", g_strdup_printf ("%lli%%", rect->right));
	cdebug("PresentationWriter::writeBlockXML()", "width attribute set");
	gsf_xml_out_add_cstr (xmlout, "height", g_strdup_printf ("%lli%%", rect->bottom));
	cdebug("PresentationWriter::writeBlockXML()", "height attribute set");
*/

#warning "writeBlockXml(): FIXME: Remember to reimplement the block coordinates when the API changes"

	text = cria_format_get(cria_block_get_format(self), "font-family");
	gsf_xml_out_add_cstr(xmlout, "family", text);
	g_free(text);
	cdebug("PresentationWriter::writeBlockXML()", "family attribute set");

	text = cria_format_get(cria_block_get_format(self), "font-size");
	gsf_xml_out_add_cstr(xmlout, "size", text);
	g_free(text);
	cdebug("PresentationWriter::writeBlockXML()", "size attribute set");
	
	gsf_xml_out_add_cstr(xmlout, NULL, cria_block_get_markup (self));
	cdebug("PresentationWriter::writeBlockXML()", "block contents written");
	gsf_xml_out_end_element(xmlout);
	cdebug("PresentationWriter::writeBlockXML()", "block element closed");
	cdebug("PresentationWriter::writeBlockXML()", "finished");
}

static void
write_slide_xml(CriaSlide *self, GsfXMLOut *xmlout, gboolean theme) {
	gchar **block_names;
	guint block_n;

	cdebug("PresentationWriter::writeSlideXML()", "start");
	if (!theme) {
		gsf_xml_out_start_element(xmlout, "slide");
		cdebug("PresentationWriter::writeSlideXML()", "slide element created");
		gsf_xml_out_add_cstr(xmlout, "title", cria_slide_get_title (self));
		cdebug("PresentationWriter::writeSlideXML()", "title attribute set");
		gsf_xml_out_add_cstr(xmlout, "layout", cria_slide_get_title (cria_slide_get_master_slide(self)));
		cdebug("PresentationWriter::writeSlideXML()", "layout attribute set");
		gsf_xml_out_add_cstr(xmlout, "theme", cria_theme_get_name (cria_slide_get_theme (self)));
		cdebug("PresentationWriter::writeSlideXML()", "theme attribute set");
	} else {
		gsf_xml_out_start_element(xmlout, "layout");
		cdebug("PresentationWriter::writeSlideXML()", "layout element created");
		gsf_xml_out_add_cstr(xmlout, "name", cria_slide_get_title (self));
		cdebug("PresentationWriter::writeSlideXML()", "name attribute set");
	}

#warning "writeSlideXml(): FIXME: Implement the background element"

	block_names = cria_slide_get_block_names(self);
	cdebug("PresentationWriter::writeSlideXML()", "got block names");
	for (block_n = 0; block_names[block_n] != NULL; block_n++) {
		write_block_xml(cria_slide_get_block(self, block_names [block_n]), xmlout);
		cdebug("PresentationWriter::writeSlideXML()", "got block written");
	}

	gsf_xml_out_end_element(xmlout);
	cdebug("PresentationWriter::writeSlideXML()", "block element closed");
	cdebug("PresentationWriter::writeSlideXML()", "finished");
}

static void
write_presentation_xml(CriaPresentation *self, GsfXMLOut *xmlout) {
	guint  slide_n;

	cdebug("PresentationWriter::writePresentationXML()", "start");
	g_hash_table_new_full(g_str_hash, g_str_equal, g_free, NULL);
	cdebug("PresentationWriter::writePresentationXML()", "themes hash table created");
	gsf_xml_out_start_element(xmlout, "presentation");
	cdebug("PresentationWriter::writePresentationXML()", "presentation element created");
	gsf_xml_out_add_cstr(xmlout, "title", cria_presentation_get_title(self));
	cdebug("PresentationWriter::writePresentationXML()", "title attribute set");
#warning "writePresentationXml(): FIXME: Implement the author element"

	g_hash_table_foreach(cria_presentation_get_themes (self),
			     write_theme_hfunc,
			     (gpointer) xmlout);
	cdebug("PresentationWriter::writePresentationXML()", "theme elements written");

	for(slide_n = 0; slide_n < cria_slide_list_n_slides(CRIA_SLIDE_LIST(self)); slide_n++) {
		CriaSlide *slide = cria_slide_list_get(CRIA_SLIDE_LIST(self), slide_n);
		write_slide_xml(slide, xmlout, FALSE);
		cdebug("PresentationWriter::writePresentationXML()", "slide written");
	}
	gsf_xml_out_end_element(xmlout);
	cdebug("PresentationWriter::writePresentationXml()", "presentation element closed");
	cdebug("PresentationWriter::writePresentationXml()", "finished");
}

void
cria_presentation_save_to_file(CriaPresentation* self, GnomeVFSURI* uri, GError** error) {
	GsfXMLOut *xmlout;
	GsfOutput *gsfout;

	g_return_if_fail(!error || !*error);

	cdebug("PresentationWriter::saveToFile()", "start");

	g_return_if_fail(uri != NULL);
	/*g_return_if_fail(GNOME_IS_VFS_URI(uri));*/

	gnome_vfs_uri_ref(uri);

	{
		gchar* str = gnome_vfs_uri_to_string(uri, GNOME_VFS_URI_HIDE_USER_NAME | GNOME_VFS_URI_HIDE_PASSWORD);
		cdebug("PresentationWriter::saveToFile()", "Uri is %s", str);
		g_free(str);
	}
	gsfout = gsf_output_gnomevfs_new_uri(uri, error);

	if(error && *error) {
		return;
	}
	
	cdebug("PresentationWriter::saveToFile()", "gsf output opened");
	xmlout = gsf_xml_out_new(gsfout);
	cdebug("PresentationWriter::saveToFile()", "xml gsf output created");
#warning "saveToFile(): FIXME: Add a doctype"
/*	gsf_xml_out_set_doc_type () */
	write_presentation_xml(self, xmlout);
	cdebug("PresentationWriter::saveToFile()", "presentation xml is written");

	gsf_output_close(gsfout);
	cdebug ("PresentationWriter::saveToFile()", "gsf output closed");
	gnome_vfs_uri_unref(uri);
}

