/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <glib-object.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <gnome.h>
#include <glade/glade.h>

#include "application.h"
#include "debug.h"
#include "libglade-support.h"
#include "main-window.h"
#include "slide-show.h"


/* method prototypes */
  // file selection handling
static void cria_open_presentation_from_filename (char const *filename);

  // GObject used methods
static void cria_application_init (CriaApplication* self);

//int x = -1, y, w = -1, h = -1, d = 0;
//char *debug_modules = NULL;

gboolean	start_off = 0;

static const struct poptOption options[] =
{
//	{NULL, 'x', POPT_ARG_INT, &x, 0, N_("X position of window"), N_("X")},
//	{NULL, 'y', POPT_ARG_INT, &y, 0, N_("Y position of window"), N_("Y")},
//	{NULL, 'w', POPT_ARG_INT, &w, 0, N_("width of window"), N_("WIDTH")},
//	{NULL, 'h', POPT_ARG_INT, &h, 0, N_("height of window"), N_("WIDTH")},
	{"start-off", 's', POPT_ARG_NONE, &start_off, 0, N_("Start a presentation (and open a file selection if no file is given at the commend line)"), NULL},
//	{"debug", '\0', POPT_ARG_NONE, &d, 0, N_("Enable debugging"), NULL},
/*	{"debug-modules", '\0', POPT_ARG_STRING, &debug_modules, 0,
		N_("Modules to enable debugging in"), N_("MODULES")},
*/	{NULL, '\0', 0, NULL, 0}
};

enum {
	PROP_0
};

enum {
	SIGNAL,
	N_SIGNALS
};

struct _CriaApplicationPriv {
	GList		* windows;
};

static	CriaApplication	* instance = NULL;

const gchar*	  cria_application_get_attribute (CriaApplication	* self);
void		  cria_application_set_attribute (CriaApplication	* self,
						const char	* attribute);
static	void	cria_application_get_property	       (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_application_set_property        (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_application_signals[N_SIGNALS] = { 0 };

static	void	cria_application_signal	       (CriaApplication	* template,
						const	gchar	* string);

void
criawips_init(int *argc, char ***argv) {
	const gchar	**filenames;
	GnomeProgram	* gnome_program;
	GValue		* value;
	poptContext	  popt_context;

	g_set_prgname (PACKAGE);
	g_debug("criawips_init(): start");

#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	/* initializing libraries */
	  /* GTK+ */
	gtk_init (argc, argv);

	  /* libgnome(ui) */
	gnome_program = gnome_program_init (
			PACKAGE, PACKAGE_VERSION,
			LIBGNOMEUI_MODULE, // module info
			*argc, *argv,
			GNOME_PARAM_POPT_TABLE, options,
			GNOME_PARAM_POPT_FLAGS, 0,
			NULL);

	  /* libglade */
	glade_init();
	glade_set_custom_handler(
			criawips_libglade_custom_handler,
			NULL);

	value = g_new0 (GValue, 1);
	g_value_init (
			value,
			G_TYPE_POINTER);
	g_object_get_property(
			G_OBJECT (gnome_program),
			GNOME_PARAM_POPT_CONTEXT,
			value);

	popt_context = (poptContext) g_value_get_pointer (value);
	filenames = poptGetArgs(popt_context);

	cria_application_get_instance();

	if (start_off) {
		g_debug ("criawips_init(): opening presentation mode");
		
		if (filenames && filenames[0]) {
			if (filenames[1]) {
				// open a warning dialog that says that we're only displaying the first file
			}
			
			cria_open_presentation_from_filename (filenames[0]);
		} else {
			GtkWidget	* dialog;
			gchar		* filename;

			dialog = gtk_file_chooser_dialog_new (
					"Open File",
					NULL,
					GTK_FILE_CHOOSER_ACTION_OPEN,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);
			switch(gtk_dialog_run(GTK_DIALOG(dialog))) {
			case GTK_RESPONSE_ACCEPT:
				gtk_widget_hide(dialog);
				while (gtk_events_pending()) {
					gtk_main_iteration();
				}
				filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
				cria_open_presentation_from_filename(filename);
				g_free(filename);
				gtk_object_destroy(GTK_OBJECT(dialog));
				break;
			case GTK_RESPONSE_CANCEL:
				criawips_quit();
				break;
			default:
				g_assert_not_reached();
				break;
			};

		}
	} else {
		g_debug("criawips_init(): opening main window(s)");
		if (!filenames) {
			/* Open the criawips main window here */
			cria_main_window_new (NULL);
		} else {
			/* open each file in a new window */
			const gchar	**iterator;
			g_debug("criawips_init(): filenames is a char** at 0x%x", (int)filenames);
			for (iterator = filenames; iterator && *iterator; iterator++) {
				g_debug("criawips_init(): iterator is now 0x%x >> 0x%x (%s)", (int)iterator, (int)(*iterator), *iterator);
				cria_main_window_new(*iterator);
			}
		}
	}

	g_debug("criawips_init(): end");
	return;
}

void
criawips_quit (void) {
	// get the application object and quit all window instances
	if (gtk_main_level() > 0) {
		gtk_main_quit ();
	} else {
		exit(0);
	}
}

/*****************************************************************************
 * file selection handling stuff                                             *
 *****************************************************************************/

static gchar*
construct_uri_from_absolute_path (const gchar* path) {
	gchar	* uri;

	g_assert(path != NULL);

	uri = g_malloc(strlen("file://") + strlen(path) + 1);
	g_snprintf(uri, strlen("file://") + strlen(path) + 1, "file://%s", path);

	return uri;
}

static void
cria_open_presentation_from_filename (const gchar* filename) {
	CriaPresentation	* pres;
	CriaSlideShow		* show;
	gchar			* text_uri;

	g_assert(filename != NULL);
	g_debug("Opening file '%s'", filename);

	if (filename[0] == '/') {
		text_uri = construct_uri_from_absolute_path (filename);
	} else {
		//text_uri = constrcut_uri_from_relative_path (filename);
	}
	
	pres = cria_presentation_new_from_file (filename);

	show = g_object_new (
			CRIA_TYPE_SLIDE_SHOW,
			"presentation", pres,
			NULL);
	g_signal_connect (
			G_OBJECT (show),
			"destroy",
			G_CALLBACK (criawips_quit),
			NULL);
}

//----------------------------------------------------------------------------

static void
cria_application_class_init (CriaApplicationClass	* cria_application_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_application_class);
#if 0
	/* setting up signal system */
	cria_application_class->signal = cria_application_signal;

	cria_application_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_APPLICATION,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaApplicationClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_application_set_property;
	g_object_class->get_property = cria_application_get_property;
}

void
cria_application_register_window(CriaMainWindow	* window) {
	CriaApplication	* self;

	g_debug("cria_application_register_window(): start");

	g_return_if_fail(window != NULL);
	g_assert(CRIA_IS_MAIN_WINDOW(window));

	self = cria_application_get_instance();

	self->priv->windows = g_list_prepend(
			self->priv->windows,
			g_object_ref(window));

	//g_object_notify(self, "windows");

	g_debug("cria_application_register_window(): end");
}

CriaApplication*
cria_application_get_instance (void) {
	if (!instance) {
		instance = g_object_new(CRIA_TYPE_APPLICATION, NULL);
	}

	return instance;
}

static void
cria_application_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaApplication	* self;

	self = CRIA_APPLICATION (object);

	switch (prop_id)
	{
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

static void
cria_application_init (CriaApplication* self) {
	self->priv = g_new0(CriaApplicationPriv,1);
	self->priv->windows = NULL;
	
	if (!instance) {
		instance = self;
	} else {
		g_critical("%s", "CriaApplication can only be instantiated once");
	}
}
#if 0
void
cria_application_open_window(CriaApplication* self, gchar const* filename) {
	CriaMainWindow	* mainwin =  cria_main_window_new(filename);
	self->priv->windows = g_list_prepend(self->priv->windows, mainwin);
}
#endif
static void
cria_application_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaApplication	* self;
	
	self = CRIA_APPLICATION (object);
	
	switch (prop_id)
	{
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_application_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaApplicationClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_application_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaApplication),
			0,
			(GInstanceInitFunc)cria_application_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaApplication",
				&info,
				0);
	}

	return type;
}

