/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <math.h>

#include <glib-object.h>
#include <glib.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomecanvas/libgnomecanvas.h>
#include <gdk/gdkkeysyms.h>
#include <gtkgesture.h>

#include "debug.h"
#include "slide-show.h"

#define TITLE_STRING _("%s - criawips")

struct _CriaSlideShowPrivate
{
	GtkWidget		* canvas;
	GnomeCanvasGroup	* group;
	GtkGestureHandler	* gesture_handler;
};

enum
{
	PROP_0,
	PROP_PRESENTATION,
	PROP_SLIDE
};

enum
{
	SIGNAL,
	N_SIGNALS
};

static	void	cria_slide_show_init	       (CriaSlideShow	* slide_show);
static	void	cria_slide_show_get_property   (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void	cria_slide_show_next_page_event(GtkGestureHandler *gh,
						gpointer user_data_a,
						gpointer user_data_b);
static void	cria_slide_show_prev_page_event(GtkGestureHandler *gh,
						gpointer user_data_a,
						gpointer user_data_b);
static	void	cria_slide_show_render	       (CriaSlideShow	* slide_show);
static	void	cria_slide_show_set_property   (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_slide_show_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_show_signal	       (CriaSlideShow	* template,
						const	gchar	* string);

enum
{
	KEYS_NONE,
	KEYS_FORWARD,
	KEYS_BACK,
	KEYS_CLOSE
};

static gboolean
cria_key_press_event_cb (
		GtkWidget	* widget,
		GdkEventKey	* event,
		gpointer	  data)
{
	gboolean  retval = FALSE;
	guint	  action = KEYS_NONE;
	guint	  slide  = 0;

	g_return_val_if_fail (CRIA_IS_SLIDE_SHOW (widget), FALSE);

	slide = cria_slide_show_get_slide (CRIA_SLIDE_SHOW (widget));

	// TODO If we want to make it configurable we have to use a
	// method instead of a switch statement
	switch (event->keyval)
	{
	case GDK_space:
	case GDK_Right:
	case GDK_KP_Enter:
	case GDK_Return:
	case GDK_Page_Down:
		if (slide == cria_presentation_n_slides (cria_slide_show_get_presentation (CRIA_SLIDE_SHOW (widget))))
			action = KEYS_CLOSE;
		else
			action = KEYS_FORWARD;
		break;
	case GDK_BackSpace:
	case GDK_Left:
	case GDK_Page_Up:
		action = KEYS_BACK;
		break;
	case GDK_Escape:
		action = KEYS_CLOSE;
		break;
	default:
		/* after show ? */
		if (slide == cria_presentation_n_slides (cria_slide_show_get_presentation (CRIA_SLIDE_SHOW (widget))))
		{
			action = KEYS_CLOSE;
		}
		else
		{
			g_debug ("cria_key_press_event(): this key is not handled");
		}
		break;
	}

	switch (action)
	{
	case KEYS_FORWARD:
		if (slide < cria_presentation_n_slides (cria_slide_show_get_presentation (CRIA_SLIDE_SHOW (widget))))
		{
			cria_slide_show_set_slide (
					CRIA_SLIDE_SHOW (widget),
					slide + 1);
		}
		break;
	case KEYS_BACK:
		if (slide > 0)
		{
			cria_slide_show_set_slide (
					CRIA_SLIDE_SHOW (widget),
					slide - 1);
		}
		break;
	case KEYS_CLOSE:
		gtk_object_destroy (GTK_OBJECT (widget));
	}

	return retval;
}

static void
cria_slide_show_class_init (CriaSlideShowClass	* cria_slide_show_class)
{
	GObjectClass	* g_object_class;

	g_debug ("cria_slide_show_class_init(%i): start", __LINE__);

	g_object_class = G_OBJECT_CLASS(cria_slide_show_class);

	// TODO make this reflect the expected behavior
	cria_slide_show_class->screen_width = gdk_screen_get_width (gdk_screen_get_default ());
	cria_slide_show_class->screen_height= gdk_screen_get_height (gdk_screen_get_default ());

	/* setting up signal system */
//	cria_slide_show_class->signal = cria_slide_show_signal;

//	cria_slide_show_signals[SIGNAL] = g_signal_new (
//			"signal",
//			CRIA_TYPE_SLIDE_SHOW,
//			G_SIGNAL_RUN_LAST,
//			G_STRUCT_OFFSET (
//				CriaSlideShowClass,
//				signal),
//			NULL,
//			NULL,
//			g_cclosure_marshal_VOID__STRING,
//			G_TYPE_NONE,
//			0);

	/* setting up property system */
	g_object_class->set_property = cria_slide_show_set_property;
	g_object_class->get_property = cria_slide_show_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_PRESENTATION,
			g_param_spec_object (
				"presentation",
				"Presentation",
				"The presentation currently shown by this CriaSlideShow",
				CRIA_TYPE_PRESENTATION,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_SLIDE,
			g_param_spec_uint (
				"slide",
				"Slide",
				"The slide that's presented in this moment or that we're "
				"currently switching to",
				0,
				G_MAXUINT,
				0,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);

	g_debug ("cria_slide_show_class_init(%i): end", __LINE__);
}

static void
cria_slide_show_clear (CriaSlideShow *self)
{
	g_return_if_fail (CRIA_IS_SLIDE_SHOW (self));
	g_return_if_fail (GNOME_IS_CANVAS (self->priv->canvas));

	g_debug ("cria_slide_show_clear(%i): start", __LINE__);

	if (self->priv->group)
	{
		g_return_if_fail (GNOME_IS_CANVAS_ITEM (self->priv->group));
		
		gnome_canvas_item_hide (GNOME_CANVAS_ITEM (self->priv->group));
		g_object_unref (G_OBJECT (self->priv->group));
	}

	self->priv->group = GNOME_CANVAS_GROUP (
			gnome_canvas_item_new (
				gnome_canvas_root (GNOME_CANVAS (self->priv->canvas)),
				gnome_canvas_group_get_type (),
				"x", (gdouble) 0.,
				"y", (gdouble) 0.,
				NULL)
			);
	g_object_ref (G_OBJECT (self->priv->group));
	gtk_object_sink (GTK_OBJECT (self->priv->group));

	g_debug ("cria_slide_show_clear(%i): end", __LINE__);
}

guint
cria_slide_show_get_slide (CriaSlideShow*self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE_SHOW (self), 0);

	return self->slide;
}

CriaPresentation *
cria_slide_show_get_presentation (
		CriaSlideShow	* self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE_SHOW(self), NULL);
	
	return self->presentation;
}

static void
cria_slide_show_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaSlideShow	* self;

	self = CRIA_SLIDE_SHOW (object);

	switch (prop_id)
	{
	case PROP_PRESENTATION:
		g_value_set_object (
				value,
				self->presentation);
		break;
	case PROP_SLIDE:
		g_value_set_uint (
				value,
				self->slide);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_slide_show_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaSlideShowClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_show_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideShow),
			0,
			(GInstanceInitFunc)cria_slide_show_init,
			0
		};

		type = g_type_register_static (
				GTK_TYPE_WINDOW,
				"CriaSlideShow",
				&info,
				0);
	}

	return type;
}

static void
cria_slide_show_init (CriaSlideShow *self)
{
	GError		* error = NULL;
	GdkColor	* color = g_new0 (GdkColor, 1);

	gdk_color_parse (
			"black",
			color);
	g_return_if_fail (CRIA_IS_SLIDE_SHOW (self));
	
	g_debug ("cria_slide_show_init(%i): start", __LINE__);

	/* setting up our private data */
	self->priv = g_new0 (CriaSlideShowPrivate, 1);

	/* setting up the canvas */
	self->priv->canvas = gnome_canvas_new ();
	gdk_colormap_alloc_color (
			gtk_widget_get_colormap (self->priv->canvas),
			color,
			FALSE,
			TRUE);
	gtk_widget_modify_bg (
			self->priv->canvas,
			GTK_STATE_NORMAL,
			color);
	gnome_canvas_set_pixels_per_unit(GNOME_CANVAS (self->priv->canvas), 1.0);
	gnome_canvas_set_scroll_region (GNOME_CANVAS (self->priv->canvas), 0, 0, CRIA_SLIDE_SHOW_GET_CLASS(self)->screen_width, CRIA_SLIDE_SHOW_GET_CLASS(self)->screen_height);

	/* setting up gestures */
	self->priv->gesture_handler = gtk_gesture_handler_new (GTK_WIDGET (self->priv->canvas));
	/* adding gestures */
	gtk_gesture_add_callback (self->priv->gesture_handler, "012",
			GTK_GESTURE_FUNC (cria_slide_show_next_page_event),
			self, NULL);
	gtk_gesture_add_callback (self->priv->gesture_handler, "210",
			GTK_GESTURE_FUNC (cria_slide_show_prev_page_event),
			self, NULL);
	
	/* setting up the window */
	gtk_window_set_default_icon_from_file (
			PACKAGE_DATA_DIR "/pixmaps/criawips.png",
			&error);
	if (error)
	{
		g_message (_("Couldn't set window icon, you have to live without; reason: %s"),
				error->message);
		g_error_free (error);
	}

	gtk_window_fullscreen (GTK_WINDOW (self));
	gtk_container_add (
			GTK_CONTAINER (self),
			self->priv->canvas);
	
	cria_slide_show_clear (self);
	cria_slide_show_render (self);

	g_signal_connect_swapped (
			G_OBJECT (self->priv->canvas),
			"key-release-event",
			G_CALLBACK (cria_key_press_event_cb),
			self);

	gtk_widget_show_all (GTK_WIDGET (self));
	g_debug ("cria_slide_show_init(%i): end", __LINE__);
}

static void
cria_slide_show_next_page_event (
		GtkGestureHandler *gh,
		gpointer user_data_a,
		gpointer user_data_b)
{
	CriaSlideShow	* self;
	
	g_return_if_fail (CRIA_IS_SLIDE_SHOW (user_data_a));
	g_return_if_fail (CRIA_SLIDE_SHOW(user_data_a)->slide <= cria_presentation_n_slides (cria_slide_show_get_presentation (CRIA_SLIDE_SHOW(user_data_a))));

	self = CRIA_SLIDE_SHOW (user_data_a);

	if (self->slide < cria_presentation_n_slides (cria_slide_show_get_presentation (self)))
	{
		cria_slide_show_set_slide (
				self,
				self->slide + 1);
	}
	else
	{
		gtk_object_destroy (GTK_OBJECT (self));
	}
}

GtkWidget*
cria_slide_show_new(CriaPresentation* p) {
	g_assert(p != NULL);
	g_assert(CRIA_IS_PRESENTATION(p));

	return g_object_new(CRIA_TYPE_SLIDE_SHOW, "presentation", p, NULL);
}

static void
cria_slide_show_prev_page_event (
		GtkGestureHandler *gh,
		gpointer user_data_a,
		gpointer user_data_b)
{
	CriaSlideShow   * self;

	g_return_if_fail (CRIA_IS_SLIDE_SHOW (user_data_a));

	self = CRIA_SLIDE_SHOW (user_data_a);

	if (self->slide > 0)
	{
		cria_slide_show_set_slide (
				self,
				self->slide - 1);
	}
		
}

static void
cria_slide_show_render_after_show (CriaSlideShow *self)
{
	gchar		* title;
	GnomeCanvasItem	* rectangle,
			* text;
	
	g_debug ("cria_slide_show_render_after_show(%i): start", __LINE__);
	g_return_if_fail (CRIA_IS_SLIDE_SHOW (self));
	
	cria_slide_show_clear (self);
	
	/* black slide */
	rectangle = gnome_canvas_item_new (
			self->priv->group,
			GNOME_TYPE_CANVAS_RECT,
			"x1", 0.,
			"y1", 0.,
			// TODO update this to reflect slide instead of screen size
			"x2", (gdouble) CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width,
			"y2", (gdouble) CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height,
			"fill-color", "black",
			NULL);
	gnome_canvas_item_show (rectangle);
	
	/* message */
	text = gnome_canvas_item_new (
			self->priv->group,
			gnome_canvas_text_get_type (),
			"anchor", GTK_ANCHOR_CENTER,
			"text", _("press a key or click to close this window..."),
			"fill-color", "white",
			// TODO update this to reflect slide instead of screen size
			"x", (gdouble) (CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width / 2),
			"y", (gdouble) (CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height / 2),
			NULL);
	gnome_canvas_item_show (text);

	title = g_strdup_printf (TITLE_STRING, _("End of Presentation"));
	gtk_window_set_title (
			GTK_WINDOW (self),
			title);
	g_free (title);

	// TODO set key and mouse handler
	
	g_debug ("cria_slide_show_render_after_show(%i): end", __LINE__);
	return;
}

static void
cria_slide_show_render_slide (CriaSlideShow *self)
{
	gchar		* title;
	CriaLayout	* layout;
	CriaPresentation* presentation;
	CriaSlide	* slide;
	CriaTheme	* theme;

	g_return_if_fail (CRIA_IS_SLIDE_SHOW (self));
	
	cria_slide_show_clear (self);

	presentation	= cria_slide_show_get_presentation (self);
	slide		= cria_presentation_get_slide (presentation, self->slide);
	theme 		= cria_presentation_get_theme (presentation, cria_slide_get_theme (slide));
	layout		= cria_theme_get_layout (theme, cria_slide_get_layout (slide));
	// hmmm... these long lines make me want to write this in java

	g_debug ("cria_slide_show_render_slide(): theme=0x%x layout=0x%x", (guint)theme, (guint)layout);
	
	if (theme && layout) {
		char	**blocks,
			**iterator;
		
		/* 1 - render slide background */
		/* TODO listen more directly to the layout */
		g_debug ("cria_slide_show_render_slide(): %s", G_OBJECT_TYPE_NAME (cria_layout_get_background (layout)));
		gnome_canvas_item_new (
				self->priv->group,
				gnome_canvas_pixbuf_get_type (),
				"x", (gdouble) 0.,
				"y", (gdouble) 0.,
				// TODO update this to reflect slide size instead of screen size
				"height", (gdouble) CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height,
				"height-in-pixels", TRUE,
				"height-set", TRUE,
				// TODO store image in layout and get it from there
				"pixbuf", cria_layout_get_background (layout),
				"width", (gdouble) CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width,
				"width-in-pixels", TRUE,
				"width-set", TRUE,
				NULL);
		
		/* 2 - render slide blocks */
		/* 2.1 - get a list of block names from the layout */
		blocks = cria_layout_get_block_names (layout);
		
		/* 2.2 - render one block after the other */
		for (iterator = blocks; iterator && *iterator; iterator++)
		{
			g_debug ("cria_slide_show_render_slide(): rendering block \"%s\"", *iterator);
			
			CriaAlignment	  alignment = cria_layout_block_get_alignment (cria_layout_get_block (layout, *iterator));
			GtkJustification  gtk_align = GTK_JUSTIFY_LEFT;
			GtkAnchorType	  gtk_anchor= GTK_ANCHOR_NORTH_WEST;
			gdouble		  xpos  = cria_layout_block_get_pos_x (cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width,
					  ypos  = cria_layout_block_get_pos_y (cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height,
					  width = cria_layout_block_get_width (cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width,
					  height= cria_layout_block_get_height(cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height,
					  font_size = cria_layout_block_get_font_size (cria_layout_get_block (layout, *iterator));

			font_size = (font_size <= 0)?1:font_size;

			if (alignment == CRIA_ALIGNMENT_LEFT)
			{
				g_debug ("cria_slide_show_render_slide(%s:%i): left rendering:", __FILE__, __LINE__);
				xpos = xpos;
				ypos = ypos;
				gtk_anchor= GTK_ANCHOR_NORTH_WEST;
				gtk_align = GTK_JUSTIFY_LEFT;
			}
			else if (alignment == CRIA_ALIGNMENT_CENTER)
			{
				g_debug ("cria_slide_show_render_slide(%s:%i): centered rendering:", __FILE__, __LINE__);
				xpos = xpos + (0.5 * width);
				ypos = ypos;
				gtk_anchor= GTK_ANCHOR_NORTH;
				gtk_align = GTK_JUSTIFY_CENTER;
			}
			else if (alignment == CRIA_ALIGNMENT_RIGHT)
			{
				g_debug ("cria_slide_show_render_slide(%s:%i): right rendering:", __FILE__, __LINE__);
				xpos = xpos + width;
				ypos = ypos;
				gtk_anchor= GTK_ANCHOR_NORTH_EAST;
				gtk_align = GTK_JUSTIFY_RIGHT;
			}
			else
				g_debug ("cria_slide_show_render_slide(%s:%i): unknown alignment value", __FILE__, __LINE__);
			
			if (cria_slide_get_block (slide, *iterator))
			{
				/* 2.2.a - render slide's block */
				g_debug ("cria_slide_show_render_slide(): rendering from slide");

				gnome_canvas_item_new (
						self->priv->group,
						gnome_canvas_text_get_type (),
						"anchor", gtk_anchor,
					//	"clip", TRUE,
					//	"clip-height", cria_layout_block_get_height (cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height,
					//	"clip-width", cria_layout_block_get_width (cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width,
						"family", cria_layout_block_get_font_family (cria_layout_get_block (layout, *iterator)),
						"justification", gtk_align,
						"markup", cria_slide_block_get_processed_content (cria_slide_get_block (slide, *iterator)),
						"size-points", font_size,
						"size-set", TRUE,
#warning 4 assertions that unit is percent
						// TODO use something like cria_slide_show_unit_to_double (slide_show, unit, length)
						"x", xpos,
						"y", ypos,
						NULL
						);
			}
			else
			{
				/* 2.2.b - render default block */
				g_debug ("cria_slide_show_render_slide(): rendering from layout");

				gnome_canvas_item_new (
						self->priv->group,
						gnome_canvas_text_get_type (),
						"anchor", gtk_anchor,
						"clip", TRUE,
						"clip-height", cria_layout_block_get_height (cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height,
						"clip-width", cria_layout_block_get_width (cria_layout_get_block (layout, *iterator)) / 100 * CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width,
						"family", cria_layout_block_get_font_family (cria_layout_get_block (layout, *iterator)),
						"justification", gtk_align,
						"size-points", font_size,
						"size-set", TRUE,
						"text", (cria_layout_block_get_default_text (cria_layout_get_block (layout, *iterator))==_("block has no content"))?"":cria_layout_block_get_default_text (cria_layout_get_block (layout, *iterator)),
						// TODO use something like cria_slide_show_unit_to_double (slide_show, unit, length)
						"x", xpos,
						"y", ypos,
						NULL
						);
			}
		}

		/* done */
	}
	else
	{
		/* render slide background */
		gnome_canvas_item_new (
				self->priv->group,
				gnome_canvas_rect_get_type (),
				"x1", (gdouble) 0.,
				"y1", (gdouble) 0.,
				// TODO update this to reflect slide size instead of screen size
				"x2", (gdouble) CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width,
				"y2", (gdouble) CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height,
				"fill-color", "darkslategray",
				NULL);

		/* render slide foreground */
		title = g_strdup_printf (_("Slide %i (\"%s\") has no layout"), self->slide + 1, cria_slide_get_title (slide));
		gnome_canvas_item_new (
				self->priv->group,
				gnome_canvas_text_get_type (),
				"anchor", GTK_ANCHOR_CENTER,
				"text", title,
				"fill-color", "white",
				// TODO update this to reflect slide instead of screen size
				"x", (gdouble) (CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_width / 2),
				"y", (gdouble) (CRIA_SLIDE_SHOW_GET_CLASS (self)->screen_height / 2),
				NULL);
		g_free (title);
	}

	/* update window's title */
	title = g_strdup_printf (TITLE_STRING, cria_slide_get_title (slide));
	g_debug ("cria_slide_show_render(%i): title[%i] (dev-count): \"%s\"", __LINE__, cria_slide_show_get_slide (self), title);
	gtk_window_set_title (
			GTK_WINDOW (self),
			title);
	g_free (title);
}

static void
cria_slide_show_render (CriaSlideShow *self)
{
	g_debug ("cria_slide_show_render(%i): start", __LINE__);
	
	g_return_if_fail (CRIA_IS_SLIDE_SHOW (self));

	if (cria_slide_show_get_presentation (self) == NULL)
	{
		cria_slide_show_render_after_show (self);
		g_debug ("cria_slide_show_render(%i): end (no presentation)", __LINE__);
		return;
	}

	if (self->slide < cria_presentation_n_slides (cria_slide_show_get_presentation (self)))
	{
		cria_slide_show_render_slide (self);
	}
	else if (self->slide == cria_presentation_n_slides (cria_slide_show_get_presentation (self)))
	{
		cria_slide_show_render_after_show (self);
	}
	else
	{
		g_assert_not_reached ();
	}

	g_debug ("cria_slide_show_render(%i): end", __LINE__);

	return;
}

void
cria_slide_show_set_presentation (
		CriaSlideShow		* self,
		CriaPresentation	* presentation)
{
	g_return_if_fail (CRIA_IS_SLIDE_SHOW(self));

	g_debug ("cria_slide_show_set_presentation(%i): start", __LINE__);

	if (self->presentation != NULL)
		g_object_unref (self->presentation);

	self->presentation = presentation;
	if (self->presentation)
		g_object_ref (presentation);

	cria_slide_show_render (self);

	g_object_notify (G_OBJECT (self), "presentation");

	g_debug ("cria_slide_show_set_presentation(%i): set presentation 0x%x", __LINE__, (guint)self->presentation);
	g_debug ("cria_slide_show_set_presentation(%i): end", __LINE__);
}

static void
cria_slide_show_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlideShow	* self;
	
	self = CRIA_SLIDE_SHOW (object);
	
	switch (prop_id)
	{
	case PROP_PRESENTATION:
		cria_slide_show_set_presentation (
				self,
				g_value_get_object (value));
		break;
	case PROP_SLIDE:
		cria_slide_show_set_slide (
				self,
				g_value_get_uint (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

void
cria_slide_show_set_slide (
		CriaSlideShow	* self,
		guint		  slide)
{
	if (slide != self->slide)
	{
		self->slide = slide;
		g_object_notify (
				G_OBJECT (self),
				"slide");
		cria_slide_show_render (self);
	}
	
	return;
}
