/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>

#include "debug.h"
#include "slide-list.h"

enum {
	PROP_0,
	PROP_PRESENTATION
};

enum {
	SIGNAL,
	N_SIGNALS
};

enum {
	COLUMN_PREVIEW,
	COLUMN_TITLE,
	COLUMN_INDEX,
	N_COLUMNS
};

struct _CriaSlideListPrivate {
	CriaPresentation	* presentation;
	GtkTreeStore		* store;
};

static	void	cria_slide_list_init	       (CriaSlideList	* self);
static	void	cria_slide_list_get_property   (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_slide_list_parse_presentation (
						CriaSlideList	* self);
static	void	cria_slide_list_set_property   (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);

static	guint	cria_slide_list_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_list_signal	       (CriaSlideList	* template,
						const	gchar	* string);

static void
cria_slide_list_class_init (CriaSlideListClass	* cria_slide_list_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_slide_list_class);
#if 0
	/* setting up signal system */
	cria_slide_list_class->signal = cria_slide_list_signal;

	cria_slide_list_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SLIDE_LIST,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideListClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_slide_list_set_property;
	g_object_class->get_property = cria_slide_list_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_PRESENTATION,
			g_param_spec_object (
				"presentation",
				"Presentation",
				"The presentation that's currently shown in the slide list",
				CRIA_TYPE_PRESENTATION,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

CriaPresentation*
cria_slide_list_get_presentation (
		CriaSlideList	* self)
{
	g_return_val_if_fail (CRIA_IS_SLIDE_LIST(self), NULL);
	
	return self->priv->presentation;
}

static void
cria_slide_list_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaSlideList	* self;

	self = CRIA_SLIDE_LIST (object);

	switch (prop_id)
	{
	case PROP_PRESENTATION:
		g_value_set_object (
				value,
				self->priv->presentation);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_slide_list_get_type (void)
{
	static GType	type = 0;

	if (!type)
	{
		const GTypeInfo info = {
			sizeof (CriaSlideListClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_list_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideList),
			0,
			(GInstanceInitFunc)cria_slide_list_init,
			0
		};

		type = g_type_register_static (
				GTK_TYPE_TREE_VIEW,
				"CriaSlideList",
				&info,
				0);
	}

	return type;
}

static void
cria_slide_list_init(CriaSlideList *self) {
	GtkTreeViewColumn	* column;
	GtkCellRenderer		* renderer;
	
	/* setting up private data */
	self->priv = g_new0(CriaSlideListPrivate,1);
	self->priv->store = gtk_tree_store_new(
			N_COLUMNS,
			GDK_TYPE_PIXBUF,
			G_TYPE_STRING,
			G_TYPE_UINT);
	
	/* setting up the tree view */
	renderer = gtk_cell_renderer_text_new();
	column = gtk_tree_view_column_new();
	gtk_tree_view_column_pack_start(
			column,
			renderer,
			FALSE);
	gtk_tree_view_column_add_attribute(
			column,
			renderer,
			"text",
			COLUMN_TITLE);
	gtk_tree_view_column_set_title(
			column,
			"Slides");
	gtk_tree_view_append_column(
			GTK_TREE_VIEW(self),
			column);
	gtk_tree_view_set_model(
			GTK_TREE_VIEW(self),
			GTK_TREE_MODEL(self->priv->store));
	return;
}

GtkWidget*
cria_slide_list_new(void) {
	CriaSlideList	* self = g_object_new(CRIA_TYPE_SLIDE_LIST, NULL);

	g_assert(GTK_IS_WIDGET(self));

	return GTK_WIDGET(self);
}

static void
cria_slide_list_parse_presentation (CriaSlideList*self) {
	guint		  i;

	g_debug("cria_slide_list_parse_presentation(): start");
	
	// FIXME make one islide-view for each slide
	for (i = 0; i < cria_presentation_n_slides(cria_slide_list_get_presentation(self)); i++) {
		GtkTreeIter	  iter;
		gtk_tree_store_append(self->priv->store, &iter, NULL);
		gtk_tree_store_set(self->priv->store, &iter,
				COLUMN_PREVIEW, NULL,
				COLUMN_TITLE, cria_slide_get_title(cria_presentation_get_slide(cria_slide_list_get_presentation(self), i)),
				COLUMN_INDEX, i,
				-1);
		g_debug("cria_slide_list_parse_presentation(): added [%i]: %s", i, cria_slide_get_title(cria_presentation_get_slide(cria_slide_list_get_presentation(self), i)));
	}
	
	g_debug("cria_slide_list_parse_presentation(): end");
}

void
cria_slide_list_set_presentation (
		CriaSlideList	* self,
		CriaPresentation* presentation) {
	g_debug("cria_slide_list_set_presentation(): start");
	
	g_return_if_fail (CRIA_IS_SLIDE_LIST(self));
	g_assert(presentation == NULL || CRIA_IS_PRESENTATION(presentation));

	if (self->priv->presentation != NULL) {
		g_object_unref (self->priv->presentation);
	}

	if (presentation != NULL) {
		g_debug("cria_slide_list_set_presentation(): got a real presentation");
		self->priv->presentation = g_object_ref (presentation);

		cria_slide_list_parse_presentation(self);
	} else {
		g_debug("cria_slide_list_set_presentation(): presentation is null :(");
		self->priv->presentation = NULL;
	}

	g_object_notify (G_OBJECT (self), "presentation");

	g_debug("cria_slide_list_set_presentation(): end");
}

static void
cria_slide_list_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlideList	* self;
	
	self = CRIA_SLIDE_LIST (object);
	
	switch (prop_id)
	{
	case PROP_PRESENTATION:
		cria_slide_list_set_presentation (
				self,
				g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

