/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <string.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>

#include "debug.h"
#include "layout.h"
#include "presentation.h"
#include "slide-view.h"

#define SLIDE_VIEW_PADDING	12
#define SLIDE_VIEW_SHADOW_OFFSET 6

enum {
	PROP_0,
	PROP_SLIDE,
	PROP_ZOOM,
	PROP_ZOOM_AUTO
};

enum {
	SIGNAL_SLIDE_CHANGED,
	SIGNAL_ZOOM_CHANGED,
	N_SIGNALS
};

struct _CriaSlideViewPriv {
	GtkWidget	* canvas;
	GnomeCanvasGroup* top_group;
	GnomeCanvasGroup* group;
	GnomeCanvasItem	* shadow,
			* border;
	CriaSlide	* slide;
	gdouble		  zoom;
	gboolean	  zoom_auto;
};

static	void	cria_slide_view_get_property           (GObject		* object,
					        	guint		  prop_id,
					        	GValue		* value,
					        	GParamSpec	* param_spec);
static	void	cria_slide_view_init	               (CriaSlideView	* self);
static	void	cria_slide_view_set_property           (GObject		* object,
						        guint		  prop_id,
						        const	GValue	* value,
						        GParamSpec	* param_spec);
static	void	cria_slide_view_slide_changed          (CriaSlideView	* template,
						        gpointer	  data);
static void	cria_slide_view_zoom_changed	       (CriaSlideView	* self,
							gpointer	  data);

static gdouble get_slide_width(void) {
	gdouble	  screen_width = gdk_screen_get_width(gdk_screen_get_default()),
		  screen_height= gdk_screen_get_height(gdk_screen_get_default());

	if ((screen_width/screen_height) >= (4.0/3.0)) {
		/* screen is too wide */
		return screen_height*4.0/3.0;
	} else {
		/* screen is too high */
		return screen_width;
	}
}

static gdouble get_slide_height(void) {
	gdouble   screen_width = gdk_screen_get_width(gdk_screen_get_default()),
		  screen_height= gdk_screen_get_height(gdk_screen_get_default());

	if ((screen_width/screen_height) >= (4.0/3.0)) {
		/* screen is too wide */
		return screen_height;
	} else {
		/* screen is too high */
		return screen_width*3.0/4.0;
	}
}

static	guint	cria_slide_view_signals[N_SIGNALS] = { 0 };

static void
cria_slide_view_class_init (CriaSlideViewClass* cria_slide_view_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_slide_view_class);

	/* setting up signal system */
	cria_slide_view_class->slide_changed = cria_slide_view_slide_changed;
	cria_slide_view_class->zoom_changed = cria_slide_view_zoom_changed;

	cria_slide_view_signals[SIGNAL_SLIDE_CHANGED] = g_signal_new (
			"slide-changed",
			CRIA_TYPE_SLIDE_VIEW,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideViewClass,
				slide_changed),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE,
			0);
	cria_slide_view_signals[SIGNAL_ZOOM_CHANGED] = g_signal_new (
			"zoom-changed",
			CRIA_TYPE_SLIDE_VIEW,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideViewClass,
				zoom_changed),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__VOID,
			G_TYPE_NONE,
			0);

	/* setting up property system */
	g_object_class->set_property = cria_slide_view_set_property;
	g_object_class->get_property = cria_slide_view_get_property;

	g_object_class_install_property (
			g_object_class,
			PROP_SLIDE,
			g_param_spec_object (
				"slide",
				"Slide",
				"The slide that's currently displayed in this view",
				CRIA_TYPE_SLIDE,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_ZOOM,
			g_param_spec_double (
				"zoom",
				"Zoom",
				"Defines the zoom factor that should be applied to display the slide",
				0.01,
				G_MAXDOUBLE,
				0.73, // FIXME this is a value that works well on my machine
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
	g_object_class_install_property (
			g_object_class,
			PROP_ZOOM_AUTO,
			g_param_spec_boolean (
				"zoom-auto",
				"Automatic zoom",
				"Defines whether the slide view changes the zoom factor on resizing",
				FALSE,
				G_PARAM_READWRITE | G_PARAM_CONSTRUCT)
			);
}

CriaSlide*
cria_slide_view_get_slide (CriaSlideView* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE_VIEW(self), NULL);
	
	return self->priv->slide;
}

static void
cria_slide_view_get_property (GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlideView	* self;

	self = CRIA_SLIDE_VIEW (object);

	switch (prop_id) {
	case PROP_SLIDE:
		g_value_set_object(value, self->priv->slide);
		break;
	case PROP_ZOOM:
		g_value_set_double(value, self->priv->zoom);
		break;
	case PROP_ZOOM_AUTO:
		g_value_set_boolean(value, self->priv->zoom_auto);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_slide_view_get_type (void) {
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaSlideViewClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_view_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlideView),
			0,
			(GInstanceInitFunc)cria_slide_view_init,
			0
		};

		type = g_type_register_static (
				GTK_TYPE_SCROLLED_WINDOW,
				"CriaSlideView",
				&info,
				0);
	}

	return type;
}

static void
cria_slide_view_init(CriaSlideView   * self) {
	/* prepare private data */
	self->priv = g_new0(CriaSlideViewPriv,1);

	/* prepare the canvas */
	self->priv->canvas = gnome_canvas_new_aa();
	gnome_canvas_set_scroll_region(
			GNOME_CANVAS(self->priv->canvas),
			0.0 - SLIDE_VIEW_PADDING,
			0.0 - SLIDE_VIEW_PADDING,
			get_slide_width() + SLIDE_VIEW_SHADOW_OFFSET + SLIDE_VIEW_PADDING,
			get_slide_height() + SLIDE_VIEW_SHADOW_OFFSET + SLIDE_VIEW_PADDING);
	gnome_canvas_set_center_scroll_region(
			GNOME_CANVAS(self->priv->canvas),
			TRUE);

	/* prepare the canvas groups */
	self->priv->top_group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS(self->priv->canvas)),
						   			 gnome_canvas_group_get_type(),
									 "x", 0.0,
									 "y", 0.0,
									 NULL));
	self->priv->group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(self->priv->top_group,
								     gnome_canvas_group_get_type(),
								     "x", 0.0,
								     "y", 0.0,
								     NULL));
	gnome_canvas_item_show(GNOME_CANVAS_ITEM(self->priv->group));
	gnome_canvas_item_show(GNOME_CANVAS_ITEM(self->priv->top_group));

	/* add the shadow */
	self->priv->shadow = gnome_canvas_item_new(self->priv->top_group,
						   gnome_canvas_rect_get_type(),
						   "fill-color", "darkgray",
						   "x1", 0.0 + SLIDE_VIEW_SHADOW_OFFSET,
						   "x2", get_slide_width() + SLIDE_VIEW_SHADOW_OFFSET,
						   "y1", 0.0 + SLIDE_VIEW_SHADOW_OFFSET,
						   "y2", get_slide_height() + SLIDE_VIEW_SHADOW_OFFSET,
						   NULL);
	gnome_canvas_item_show(self->priv->shadow);

	/* add the slide border */
	self->priv->border = gnome_canvas_item_new(self->priv->top_group,
						   gnome_canvas_rect_get_type(),
						   "outline-color", "black",
						   "width-pixels", 1,
						   "x1", 0.0,
						   "x2", get_slide_width(),
						   "y1", 0.0,
						   "y2", get_slide_height(),
						   NULL);
	gnome_canvas_item_show(self->priv->shadow);
	gtk_widget_show(self->priv->canvas);
	gtk_scrolled_window_set_hadjustment(GTK_SCROLLED_WINDOW(self), gtk_layout_get_hadjustment(GTK_LAYOUT(self->priv->canvas)));
	gtk_scrolled_window_set_vadjustment(GTK_SCROLLED_WINDOW(self), gtk_layout_get_vadjustment(GTK_LAYOUT(self->priv->canvas)));
	gtk_container_add(GTK_CONTAINER(self), self->priv->canvas);
}

GtkWidget*
cria_slide_view_new(void) {
	return g_object_new(CRIA_TYPE_SLIDE_VIEW, NULL);
}

void
cria_slide_view_set_slide (CriaSlideView* self, CriaSlide* slide) {
	g_debug("cria_slide_view_set_slide(): start");
	
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	g_assert(slide == NULL || CRIA_IS_SLIDE(slide));

	if (self->priv->slide != NULL) {
		g_object_unref (self->priv->slide);
	}

	if (slide) {
		self->priv->slide = g_object_ref (slide);
	} else {
		self->priv->slide = NULL;
	}

	g_signal_emit_by_name(self,
			      "slide-changed");
	g_object_notify (G_OBJECT (self), "slide");

	g_debug("cria_slide_view_set_slide(): end");
}

static void
cria_slide_view_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaSlideView	* self;
	
	self = CRIA_SLIDE_VIEW (object);
	
	switch (prop_id)
	{
	case PROP_SLIDE:
		cria_slide_view_set_slide(self, g_value_get_object (value));
		break;
	case PROP_ZOOM:
		cria_slide_view_set_zoom(self, g_value_get_double(value));
		break;
	case PROP_ZOOM_AUTO:
		cria_slide_view_set_zoom_auto(self, g_value_get_boolean(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(
				object,
				prop_id,
				param_spec);
		break;
	}
}

void
cria_slide_view_set_zoom(CriaSlideView* self, gdouble zoom) {
	g_assert(G_IS_OBJECT(self));
	g_assert(CRIA_IS_SLIDE_VIEW(self));

	if(self->priv->zoom != zoom) {
		self->priv->zoom = zoom;

		gnome_canvas_set_pixels_per_unit(GNOME_CANVAS(self->priv->canvas), zoom);
		
		g_object_notify(G_OBJECT(self), "zoom");
		g_signal_emit_by_name(G_OBJECT(self), "zoom-changed");
	}
}

void
cria_slide_view_set_zoom_auto(CriaSlideView* self, gboolean zoom_auto) {
	g_assert(G_IS_OBJECT(self));
	g_assert(CRIA_IS_SLIDE_VIEW(self));

	if(self->priv->zoom_auto != zoom_auto) {
		self->priv->zoom_auto = zoom_auto;

		g_object_notify(G_OBJECT(self), "zoom-auto");
		/* FIXME think whether we really need this */
		//g_signal_emit_by_name(G_OBJECT(self), "zoom-auto-toggle");
	}
}

static void
cria_slide_view_clear_canvas(CriaSlideView* self) {
	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->group != NULL);
	g_assert(GNOME_IS_CANVAS_GROUP(self->priv->group));
	g_assert(self->priv->shadow != NULL);
	g_assert(GNOME_IS_CANVAS_ITEM(self->priv->shadow));
	g_assert(self->priv->border != NULL);
	g_assert(GNOME_IS_CANVAS_ITEM(self->priv->border));

	gnome_canvas_item_hide(GNOME_CANVAS_ITEM(self->priv->group));
	gtk_object_destroy(GTK_OBJECT(self->priv->group));
	self->priv->group = NULL;

	self->priv->group = GNOME_CANVAS_GROUP(gnome_canvas_item_new(
				                  self->priv->top_group,
						  gnome_canvas_group_get_type(),
						  NULL));
	/* move the slide border over the group and the shadow below it */
	gnome_canvas_item_raise_to_top(self->priv->border);
	gnome_canvas_item_lower_to_bottom(self->priv->shadow);
}

static void
cria_slide_view_render_background_default(CriaSlideView* self) {
	GnomeCanvasItem	* c_item;

	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->group != NULL);
	g_assert(GNOME_IS_CANVAS_GROUP(self->priv->group));

	c_item = gnome_canvas_item_new (self->priv->group,
					gnome_canvas_rect_get_type(),
					"fill-color", "white",
					"x1", 0.0,
					"x2", get_slide_width(),
					"y1", 0.0,
					"y2", get_slide_height(),
					NULL);
	gnome_canvas_item_show(c_item);
}

static void
cria_slide_view_render_background_pixmap(CriaSlideView* self) {
	CriaSlide	* slide = cria_slide_view_get_slide(self);
	CriaLayout	* layout = cria_slide_get_layout(slide);
	
	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->group != NULL);
	g_assert(GNOME_IS_CANVAS_GROUP(self->priv->group));
	g_assert(CRIA_IS_SLIDE(slide));
	g_assert(CRIA_IS_LAYOUT(layout));

	if (TRUE) { // is_stretched
		GnomeCanvasItem	* c_item;
		c_item = gnome_canvas_item_new (
				self->priv->group,
				gnome_canvas_pixbuf_get_type (),
				"x", 0.0,
				"y", 0.0,
				"height", get_slide_height(),
				"height-set", TRUE,
				"pixbuf", cria_layout_get_background (layout),
				"width", get_slide_width(),
				"width-set", TRUE,
				NULL);
		gnome_canvas_item_show(c_item);
	} else if (FALSE) { // is scaled
	} else if (FALSE) { // is tiled
	} else if (FALSE) { // is aligned
	} else {
		g_assert_not_reached();
	}
}

static void
cria_slide_view_render_background(CriaSlideView* self) {
	CriaSlide	* slide;
	CriaLayout	* layout;
	
	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->group != NULL);
	g_assert(GNOME_IS_CANVAS_GROUP(self->priv->group));

	slide = cria_slide_view_get_slide(self);
	g_assert(slide == NULL || CRIA_IS_SLIDE(slide));

	if (!slide) {
		/* If there's no slide left, just display the empty background */
		cria_slide_view_render_background_default(self);
		return;
	}
	
	layout = cria_slide_get_layout(slide);
	g_assert(layout != NULL || CRIA_IS_LAYOUT(layout));

	if (!layout) {
		/* If (for some stupid reason) there's no layout associated to
		 * the slide, just render the empty background */
		cria_slide_view_render_background_default(self);
		return;
	}
	
	if (GDK_IS_PIXBUF(cria_layout_get_background(layout))) {
		/* render a pixbuf background */
		cria_slide_view_render_background_pixmap(self);
	} else if (FALSE) { // BACKGROUND_IS_COLOR
		/* render a colored background */
	} else if (FALSE) {
		/* render a gradient background */
	} else {
		g_assert_not_reached();
	}
}

static void
cria_slide_view_render_block(CriaSlideView* self, CriaSlide* slide, gchar const* blockname) {
	GnomeCanvasItem	* c_item;
	CriaLayout	* layout;
	GtkAnchorType	  anchor;
	GtkJustification  justification;
	gdouble		  x,
			  width,
			  y,
			  height,
			  font_size;

	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	g_assert(slide != NULL);
	g_assert(CRIA_IS_SLIDE(slide));
	g_assert(blockname != NULL);
	g_assert(strlen(blockname) > 0);

	layout = cria_slide_get_layout(slide);
	g_assert(layout != NULL);
	g_assert(CRIA_IS_LAYOUT(layout));

	x = cria_layout_block_get_pos_x (cria_layout_get_block (layout, blockname)) / 100 * get_slide_width();
	y = cria_layout_block_get_pos_y (cria_layout_get_block (layout, blockname)) / 100 * get_slide_height();
	width = cria_layout_block_get_width (cria_layout_get_block (layout, blockname)) / 100 * get_slide_width();
	height = cria_layout_block_get_height(cria_layout_get_block (layout, blockname)) / 100 * get_slide_height();

	font_size = cria_layout_block_get_font_size (cria_layout_get_block (layout, blockname));

	switch(cria_layout_block_get_alignment (cria_layout_get_block (layout, blockname))) {
	case CRIA_ALIGNMENT_CENTER:
		x = x + (0.5 * width);
		anchor= GTK_ANCHOR_NORTH;
		justification = GTK_JUSTIFY_CENTER;
		break;
	case CRIA_ALIGNMENT_RIGHT:
		x = x + width;
		anchor= GTK_ANCHOR_NORTH_EAST;
		justification = GTK_JUSTIFY_RIGHT;
		break;
	case CRIA_ALIGNMENT_LEFT:
	default:
		anchor= GTK_ANCHOR_NORTH_WEST;
		justification = GTK_JUSTIFY_LEFT;
		break;
	};
		
	if (cria_slide_get_block(slide, blockname)) {
		/* there's block information in the slide, use this information
		 * for rendering */
		c_item = gnome_canvas_item_new (self->priv->group,
						gnome_canvas_text_get_type (),
						"anchor", anchor,
						"clip", TRUE,
						"clip-height", height,
						"clip-width", width,
						"family", cria_layout_block_get_font_family (cria_layout_get_block (layout, blockname)),
						"fill-color", "black",
						"justification", justification,
						"size-points", font_size,
						"size-set", TRUE,
						"markup", cria_slide_block_get_processed_content (cria_slide_get_block (slide, blockname)),
						// TODO use something like cria_slide_show_unit_to_double (slide_show, unit, length)
						"x", x,
						"y", y,
						NULL);
	} else {
		/* there's no block information in the slide, just use the one
		 * from the slide layout */
		g_debug("cria_slide_view_render_block(): adding canvas item from layout");
		gchar const	* text = cria_layout_block_get_default_text (cria_layout_get_block (layout, blockname));
		
		if (strcmp(text, _("the block has no content")) == 0) {
			text = "";
		}

		c_item = gnome_canvas_item_new (self->priv->group,
						gnome_canvas_text_get_type (),
						"anchor", anchor,
						"clip", TRUE,
						"clip-height", height,
						"clip-width", width,
						"family", cria_layout_block_get_font_family (cria_layout_get_block (layout, blockname)),
						"fill-color", "black",
						"justification", justification,
						"size-points", font_size,
						"size-set", TRUE,
						"text", text,
//						"weight", PANGO_WEIGHT_BOLD,
//						"weight-set", TRUE,
						// TODO use something like cria_slide_show_unit_to_double (slide_show, unit, length)
						"x", x,
						"y", y,
						NULL);
	}
	g_debug("cria_slide_view_render_block(): showing canvas item");
	gnome_canvas_item_show(c_item);
	g_debug("cria_slide_view_render_block(): done");
}

static void
cria_slide_view_render_blocks(CriaSlideView* self) {
	gchar		**blocks,
			**iterator;
	CriaLayout	* layout;
	CriaSlide	* slide;

	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_VIEW(self));

	slide = cria_slide_view_get_slide(self);
	if(!CRIA_IS_SLIDE(slide)){
		return;
	}
	
	layout = cria_slide_get_layout(slide);
	g_assert(CRIA_IS_LAYOUT(layout));

	blocks = cria_layout_get_block_names(layout);

	for (iterator = blocks; iterator && *iterator; iterator++) {
		cria_slide_view_render_block(self, slide, *iterator);
	}

	g_free(blocks);
}

static void
cria_slide_view_slide_changed(CriaSlideView* self, gpointer data) {
	g_assert(self != NULL);
	g_assert(CRIA_IS_SLIDE_VIEW(self));
	
	cria_slide_view_clear_canvas(self);
	cria_slide_view_render_background(self);
	cria_slide_view_render_blocks(self);
}

static void
cria_slide_view_zoom_changed(CriaSlideView* self, gpointer data) {
	g_assert(CRIA_IS_SLIDE_VIEW(self));
}

