/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdint.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <libgnomeui/gnome-about.h>

#include "debug.h"
#include "application.h"
#include "main-window.h"
#include "presentation.h"
#include "slide-list.h"
#include "slide-show.h"
#include "slide-view.h"

enum {
	PROP_0,
	PROP_PRESENTATION
};

enum {
	SIGNAL,
	N_SIGNALS
};

struct _CriaMainWindowPriv {
	GladeXML	* xml;
	CriaPresentation* presentation;
};

static	void	cria_main_window_get_property  (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	cria_main_window_init	       (CriaMainWindow	* self);
static	void	cria_main_window_open	       (CriaMainWindow	* self,
						gpointer	  data);
static	void	cria_main_window_selected_slide_changed(
						CriaMainWindow	* self,
						GtkTreeSelection* selection);
static	void	cria_main_window_set_property  (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
static void	  cria_main_window_set_zoom_from_spin  (CriaMainWindow  * template,
							GtkSpinButton	* spinner);
static	void	cria_main_window_start_off	       (CriaMainWindow	* self);
#if 0
static	void	cria_main_window_signal		       (CriaMainWindow	* template,
							const	gchar	* string);

static	guint	cria_main_window_signals[N_SIGNALS] = { 0 };
#endif
static void
cria_main_window_class_init (CriaMainWindowClass	* cria_main_window_class)
{
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_main_window_class);
#if 0
	/* setting up signal system */
	cria_main_window_class->signal = cria_main_window_signal;

	cria_main_window_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_MAIN_WINDOW,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaMainWindowClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_main_window_set_property;
	g_object_class->get_property = cria_main_window_get_property;

	g_object_class_install_property(g_object_class,
					PROP_PRESENTATION,
					g_param_spec_object("presentation",
							    "Presentation",
							    "This is the presentation that's currently assotiated with this main window",
							    CRIA_TYPE_PRESENTATION,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

/*
 * cria_main_window_displayed_slide_changed:
 * @self: ...
 * @slide_view: ...
 *
 * ...
 */
static void
cria_main_window_displayed_slide_changed(CriaMainWindow* self, CriaSlideView* slide_view) {
	gchar	* message;
	
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(slide_view != NULL && CRIA_IS_SLIDE_VIEW(slide_view));
	g_assert(self->priv != NULL);
	g_assert(self->priv->xml != NULL && GLADE_IS_XML(self->priv->xml));
	g_assert(!self->priv->presentation || CRIA_IS_PRESENTATION(self->priv->presentation));

	if(!self->priv->presentation) {
		message = _("No Presentation assigned");
	} else if(cria_presentation_n_slides(self->priv->presentation) < 1) {
		message = _("The Presentation doesn't contains any slides");
	} else if(!cria_slide_view_get_slide(slide_view)) {
		message = _("No Slide selected");
	} else if(cria_presentation_get_slide_index(self->priv->presentation, cria_slide_view_get_slide(slide_view)) >= 0) {
		message = g_strdup_printf("<span weight=\"bold\">%s %d/%d \"%s\"</span>",
					  _("Slide"),
					  cria_presentation_get_slide_index(self->priv->presentation, cria_slide_view_get_slide(slide_view)) + 1,
					  cria_presentation_n_slides(self->priv->presentation),
					  cria_slide_get_title(cria_slide_view_get_slide(slide_view)));
	} else {
		/* the slide could not be found in the presentation */
		g_assert_not_reached();
	}

	gtk_label_set_markup(GTK_LABEL(glade_xml_get_widget(self->priv->xml, "main_label_displayed_title")), message);

	g_free(message);
}

CriaPresentation*
cria_main_window_get_presentation (
		CriaMainWindow	* self)
{
	g_return_val_if_fail (CRIA_IS_MAIN_WINDOW(self), NULL);
	
	return self->priv->presentation;
}

static void
cria_main_window_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaMainWindow	* self;

	self = CRIA_MAIN_WINDOW (object);

	switch (prop_id)
	{
	case PROP_PRESENTATION:
		g_value_set_object (
				value,
				self->priv->presentation);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_main_window_get_type (void)
{
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaMainWindowClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_main_window_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaMainWindow),
			0,
			(GInstanceInitFunc)cria_main_window_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaMainWindow",
				&info,
				0);
	}

	return type;
}

static void
cria_main_window_init(CriaMainWindow* self) {
	g_debug("cria_main_window_init(): start");
	
	/* creating the private data */
	self->priv = g_new0(CriaMainWindowPriv,1);
	self->priv->xml = glade_xml_new(PACKAGE_DATA_DIR "/" PACKAGE "/data/criawips.glade", NULL, NULL);

	if (!self->priv->xml) {
		gchar	* error_message = g_strdup_printf("<span weight=\"bold\">%s</span>\n\n%s",
				                          _("Couldn't load GUI description"),
							  _("The file containing the GUI description of this application could not "
							    "be found, this is usually an error in your installation."));
		cria_application_show_error_dialog(NULL,
				_("Error opening file"),
				error_message);
		g_free(error_message);
		criawips_quit();
	}

	gtk_paned_set_position(
			GTK_PANED(glade_xml_get_widget(self->priv->xml, "main_hpaned")),
			150);

	/* connecting signals */
	  /* window */
	g_signal_connect(glade_xml_get_widget(self->priv->xml, "main_window"),
			"delete-event",
			G_CALLBACK(criawips_quit),
			NULL);
	  /* menus */
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_file_new"), FALSE);
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "main_menu_file_open"), "activate",
				 G_CALLBACK(cria_main_window_open), self);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_file_save"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_file_save_as"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_file_properties"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_file_close"), FALSE);
	g_signal_connect(glade_xml_get_widget(self->priv->xml, "main_menu_file_quit"), "activate",
			 G_CALLBACK(criawips_quit), NULL);

	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_edit_cut"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_edit_copy"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_edit_paste"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_edit_clear"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_edit_preferences"), FALSE);

	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_zoom_in"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_zoom_out"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_view_zoom_100"), FALSE);
	gtk_widget_set_sensitive(glade_xml_get_widget(self->priv->xml, "main_menu_view_zoom_fit"), FALSE);
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "main_menu_view_start_off"), "activate",
				 G_CALLBACK(cria_main_window_start_off), self);

	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "main_menu_help_about"), "activate",
				 G_CALLBACK(cria_main_window_show_about), self);
	  /* toolbars */
	g_signal_connect_swapped(
			glade_xml_get_widget(self->priv->xml, "main_tool_button_open"),
			"clicked",
			G_CALLBACK(cria_main_window_open),
			self);
	g_signal_connect_swapped(
			glade_xml_get_widget(self->priv->xml, "main_tool_button_start_off"),
			"clicked",
			G_CALLBACK(cria_main_window_start_off),
			self);
	gtk_spin_button_set_value(GTK_SPIN_BUTTON(glade_xml_get_widget(self->priv->xml, "main_spinbutton_zoom")),
				  cria_slide_view_get_zoom(CRIA_SLIDE_VIEW(glade_xml_get_widget(self->priv->xml, "main_slide_view")))*100);
	g_signal_connect_swapped(
			glade_xml_get_widget(self->priv->xml, "main_spinbutton_zoom"),
			"value-changed",
			G_CALLBACK(cria_main_window_set_zoom_from_spin),
			self);

	/* slide list */
	g_signal_connect_swapped(gtk_tree_view_get_selection(GTK_TREE_VIEW(glade_xml_get_widget(self->priv->xml, "main_slide_list"))), "changed",
				 G_CALLBACK(cria_main_window_selected_slide_changed), self);

	/* slide view */
	g_signal_connect_swapped(glade_xml_get_widget(self->priv->xml, "main_slide_view"), "slide-changed",
				 G_CALLBACK(cria_main_window_displayed_slide_changed), self);

	/* display the window */
	gtk_widget_show(glade_xml_get_widget(self->priv->xml, "main_window"));

	g_debug("cria_main_window_init(): end");
}

/**
 * cria_main_window_new:
 * @filename: filename of the new window
 *
 * Creates a new main window displaying the presentation that @filename
 * points to. It automatically registers the window for the application
 * to make it managed by the application. @filename can be NULL to create
 * a window without a presentation.
 *
 * Returns a new main window
 */
CriaMainWindow*
cria_main_window_new(const gchar* filename) {
	CriaMainWindow	* self;

	g_debug("cria_main_window_new(): start");
	
	self = g_object_new(CRIA_TYPE_MAIN_WINDOW, NULL);
	
	if (filename != NULL) {
		CriaPresentation	* p;
		GError			* error = NULL;

		g_debug("cria_main_window_new(): got filename 0x%x(\"%s\")", (uintptr_t)filename, filename);
		
		p = cria_presentation_new_from_file(filename, &error);

		if (error) {
			/* NOTE: keep these strings in sync with the ones from application.c */
			char            * error_message = g_strdup_printf("<span weight=\"bold\">%s</span>\n\n%s %s:\n<span style=\"italic\">%s</span>.",
									  _("The Presentation could not be opened"),
									  _("For some reason the presentation you wanted to open could not be opened."),
									  _("The reason was:"),
									  error->message);
			cria_application_show_error_dialog(GTK_WINDOW(glade_xml_get_widget(self->priv->xml, "main_window")),
					 		   _("Error opening file"),
							   error_message);
			g_free(error_message);
			g_error_free(error);
		} else {
			g_debug ("cria_main_window_new(): setting presentation");
		
			g_object_set(G_OBJECT(self), "presentation", p, NULL);

			g_debug ("cria_main_window_new(): done");
		}
	} else {
		g_debug("cria_main_window_new(): filename was NULL, we're done");
	}
	
	g_debug("cria_main_window_new(): in between");
	
	cria_application_register_window(self);

	g_debug("cria_main_window_new(): end");
	return self;
}

static void
cria_main_window_open(CriaMainWindow* self, gpointer d) {
	GtkWidget	* dialog = gtk_file_chooser_dialog_new("Open File",
							       NULL,
							       GTK_FILE_CHOOSER_ACTION_OPEN,
							       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
							       GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
							       NULL);
	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), TRUE);

	if(gtk_dialog_run(GTK_DIALOG(dialog))) {
		gtk_widget_hide(dialog);

		while(gtk_events_pending()) {
			gtk_main_iteration();
		}
		
		GSList			* uris	   = gtk_file_chooser_get_uris(GTK_FILE_CHOOSER(dialog));
		/*g_object_unref(dialog); dialog = NULL;*/

		if(uris && g_slist_length(uris) == 1) {
			GError	* error = NULL;
			
			cria_main_window_set_presentation(self, cria_presentation_new_from_uri((gchar*)uris->data, &error));

			if(error) {
				/* NOTE: keep the string in sync with the ones from application.c */
				char            * error_message = g_strdup_printf("<span weight=\"bold\">%s</span>\n\n%s %s:\n<span style=\"italic\">%s</span>.",
										  _("The Presentation could not be opened"),
										  _("For some reason the presentation you wanted to open could not be opened."),
										  _("The reason was:"),
										  error->message);
				cria_application_show_error_dialog(GTK_WINDOW(glade_xml_get_widget(self->priv->xml, "main_window")),
								   _("Error opening file"),
								   error_message);
				gtk_dialog_run(GTK_DIALOG(dialog));
				
				g_free(error_message);
				g_error_free(error);
			}
		} else {
			GSList	* it = NULL;
			GError	* error = NULL;
			for(it = uris; it; it = it->next) {
				CriaMainWindow*	mainwin = cria_main_window_new(NULL);
				cria_main_window_set_presentation(mainwin, cria_presentation_new_from_uri((gchar*)it->data, &error));
				gtk_widget_show(GTK_WIDGET(glade_xml_get_widget(mainwin->priv->xml, "main_window")));

				if(error) {
					/* NOTE: keep the string in sync with the ones from application.c */
					char            * error_message = g_strdup_printf("<span weight=\"bold\">%s</span>\n\n%s %s:\n<span style=\"italic\">%s</span>.",
											  _("The Presentation could not be opened"),
											  _("For some reason the presentation you wanted to open could not be opened."),
											  _("The reason was:"),
											  error->message);
					cria_application_show_error_dialog(GTK_WINDOW(glade_xml_get_widget(mainwin->priv->xml, "main_window")),
							_("Error opening file"),
							error_message);
					gtk_dialog_run(GTK_DIALOG(dialog));
					
					g_free(error_message);
					g_error_free(error);
				}
			}
		}
	}

	gtk_object_destroy(GTK_OBJECT(dialog));
}

static void
cria_main_window_selected_slide_changed(CriaMainWindow* self, GtkTreeSelection* selection) {
	GtkWidget	* slide_view;
	CriaSlide	* slide;
	GtkTreeIter	  iter;
	gint		  index;
	gboolean	  has_selection;

	g_assert(CRIA_IS_MAIN_WINDOW(self));
	g_assert(GTK_IS_TREE_SELECTION(selection));

	slide_view = glade_xml_get_widget(self->priv->xml, "main_slide_view");

	has_selection = gtk_tree_selection_get_selected(
			selection,
			NULL,
			&iter);

	if (has_selection) {
		gtk_tree_model_get(
				gtk_tree_view_get_model(GTK_TREE_VIEW(glade_xml_get_widget(self->priv->xml, "main_slide_list"))), &iter,
				COLUMN_INDEX, &index,
				-1);

		g_debug("cria_main_window_selected_slide_changed(): selected slide[%i]", index);

		slide = cria_presentation_get_slide(
				cria_main_window_get_presentation(self),
				index);
	} else {
		slide = NULL;
	}

	cria_slide_view_set_slide(
			CRIA_SLIDE_VIEW(slide_view),
			slide);
}

void
cria_main_window_set_presentation (
		CriaMainWindow	* self,
		CriaPresentation* presentation) {
	g_debug("cria_main_window_set_presentation(): start");
	
	g_return_if_fail (CRIA_IS_MAIN_WINDOW(self));
	g_assert(presentation == NULL || CRIA_IS_PRESENTATION(presentation));

	if (self->priv->presentation != NULL) {
		g_object_unref (self->priv->presentation);
	}

	if (presentation) {
		self->priv->presentation = g_object_ref (presentation);
		cria_slide_list_set_presentation(CRIA_SLIDE_LIST(glade_xml_get_widget(self->priv->xml, "main_slide_list")), cria_main_window_get_presentation(self));
		cria_slide_view_set_slide(CRIA_SLIDE_VIEW(glade_xml_get_widget(self->priv->xml, "main_slide_view")), cria_presentation_get_slide(cria_main_window_get_presentation(self), 0));
	}

	g_object_notify (G_OBJECT (self), "presentation");

	g_debug("cria_main_window_set_presentation(): end");
}

static void
cria_main_window_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec) {
	CriaMainWindow	* self;
	
	self = CRIA_MAIN_WINDOW (object);
	
	switch (prop_id) {
	case PROP_PRESENTATION:
		cria_main_window_set_presentation (
				self,
				g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

static void
cria_main_window_set_zoom_from_spin(CriaMainWindow* self, GtkSpinButton* spinner) {
	cria_slide_view_set_zoom(CRIA_SLIDE_VIEW(glade_xml_get_widget(self->priv->xml, "main_slide_view")), gtk_spin_button_get_value(spinner)/100);
}

void
cria_main_window_show_about(CriaMainWindow* self) {
	const gchar	* authors[] = {
		"Sven Herzberg",
		NULL
	};

	GdkPixbuf	* pixbuf = NULL;

	const gchar	* documenters[] = {NULL};

	GtkWidget	* dialog = gnome_about_new(
				PACKAGE,
				VERSION,
				_("Copyright (C) 2004 - The criawips development team"),
				_("Criawips is a presentation application for the GNOME "
				"desktop environment which should integrate smoothly into "
				"the desktop"),
				authors,
				documenters,
				(strcmp("translator-credits", _("translator-credits")) == 0) ? NULL : _("translator-credits"),
				pixbuf);
	
	gtk_window_set_transient_for(
			GTK_WINDOW(dialog),
			GTK_WINDOW(glade_xml_get_widget(self->priv->xml, "main_window")));

	gtk_widget_show(dialog);

	if (pixbuf) {
		g_object_unref(pixbuf);
	}
}

static void
cria_main_window_start_off (CriaMainWindow* self) {
	CriaPresentation	* p;

	g_debug("cria_main_window_start_off(): start");
	
	p = cria_main_window_get_presentation(self);

	if (p != NULL && CRIA_IS_PRESENTATION(p)) {
		gtk_widget_show(cria_slide_show_new(p));
	}

	g_debug("cria_main_window_start_off(): end");
}

