/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <inttypes.h>
#include <string.h>

#include <glib.h>
#include <gsf/gsf-libxml.h>

#include "debug.h"
#include "presentation-parser.h"

typedef struct {
	GList			* tags;
	guint			  unknown;
	GString			* cdata;
	GError			* error;
} Sax001Data;

typedef void (*cria_sax_start_element) (GHashTable	* attributes,
					Sax001Data	* sax_data);
typedef void (*cria_sax_end_element)   (Sax001Data	* sax_data);

typedef struct _cria_tag_data cria_tag_data;

struct _cria_tag_data
{
	const	gchar		* tag_name;
	cria_sax_start_element	  tag_start;
	cria_sax_end_element	  tag_end;
};

static xmlSAXHandler cria_0_0_1_handler;

CriaPresentation*
cria_presentation_parse_input_xml (GsfInput *input, GError**error) {
	CriaPresentation* presentation = NULL;
	xmlParserCtxt	* context;
	
	g_debug ("cria_presentation_parse_input_xml(%i): start", __LINE__);

	context = gsf_xml_parser_context (input);
	context->sax = &cria_0_0_1_handler;
	context->userData = g_new0 (Sax001Data, 1);

#warning "FIXME validate against DTD"
	xmlParseDocument (context);

	if (((Sax001Data*)context->userData)->error) {
		*error = ((Sax001Data*)context->userData)->error;
		g_free(context->userData);
		return NULL;
	}

	if (context->wellFormed) {
		presentation = CRIA_PRESENTATION (((Sax001Data*)context->userData)->tags);
		g_debug ("cria_presentation_parse_input_xml(%i): really good file", __LINE__);
	} else {
		g_debug ("cria_presentation_parse_input_xml(%i): really bad file", __LINE__);
	}

	g_free (context->userData);
	g_debug ("cria_presentation_parse_input_xml(%i): end", __LINE__);

	return presentation;
}

static void
cria_sax_0_0_1_characters (
		gpointer	* user_data,
		const	xmlChar	* string,
		int		  length) {
	char		* str;
	Sax001Data	* data = (Sax001Data*) user_data;
	
	/* return if empty and if we're in some unknown space */
	if (data->unknown > 0 || !string) {
		return;
	}

	str = g_strndup (string, length);
	g_strstrip (str);
	
	if (strlen (str)) {
		/* just do this if we really have some data to care about */
		char	**strv,
			**iterator;

		/* TODO this code totally misbehaved (see the if-statement above)
		 * we need some new cdata parsing code. something like
		 * 1. replace any whitespace character with a space
		 * 2. replace any two spaces with one until there are no duplicated
		 *    spaces
		 * 3. cut trailing space
		 * 4. cut leading space if cdata was empty
		 */
		/* split the string into single lines to be processed */
		strv = g_strsplit (
				str,
				"\n",
				0);

		if (!data->cdata) {
			data->cdata = g_string_new ("");
		}

		for (iterator = strv; *iterator; iterator++) {
			g_strstrip (*iterator);

			/* append one whitespace between lines */
			/* but don't do if we already have '\n' there */
/*			if (strlen (data->cdata->str) && (data->cdata->str[strlen (data->cdata->str) - 1] != '\n'))
				g_string_append (data->cdata, " ");
*/
			g_string_append (data->cdata, *iterator);
		}
	}
}

static void
cria_sax_0_0_1_element_background_start (
		GHashTable	* hash_table,
		Sax001Data	* data) {
	GError	* error = NULL;
	g_return_if_fail (CRIA_IS_LAYOUT (g_list_last (data->tags)->data));

	data->tags = g_list_append (data->tags,
				    gdk_pixbuf_new_from_file (g_hash_table_lookup (hash_table, "file"), &error));

	if (error) {
		data->error = error;
	}
}

static void
cria_sax_0_0_1_element_background_end (
		Sax001Data	* data)
{
	g_return_if_fail (GDK_IS_PIXBUF	(g_list_last (data->tags)->data));
	g_return_if_fail (CRIA_IS_LAYOUT (g_list_nth (data->tags, g_list_length(data->tags) - 2)->data));
	
	g_object_set (
			G_OBJECT (g_list_nth (data->tags, g_list_length(data->tags) - 2)->data),
			"background", GDK_PIXBUF (g_list_last (data->tags)->data),
			NULL);

	data->tags = g_list_remove (
			data->tags,
			g_list_last (data->tags)->data);
}

static void
cria_sax_0_0_1_element_block_start (
		GHashTable	* hash_table,
		Sax001Data	* data)
{
	/* block may happen both inside of a slide and inside of a layout */
	g_return_if_fail (CRIA_IS_LAYOUT (g_list_last (data->tags)->data) || CRIA_IS_SLIDE (g_list_last (data->tags)->data));

	if (CRIA_IS_LAYOUT (g_list_last (data->tags)->data))
	{
		/* this is the layout block */
		gdouble	  fsize = 0.;
		gchar	* fonts = g_hash_table_lookup (hash_table, "size");

		g_debug ("cria_sax_0_0_1_element_block_start(): layout block with font size delivered as \"%s\"", fonts);

		if (fonts && g_ascii_strtod (fonts, NULL))
		{
			fsize = g_ascii_strtod (fonts, NULL);
			g_debug ("cria_sax_0_0_1_element_block_start(): got a cool font size, double %f", fsize);
		}
		else
		{
			g_debug ("cria_sax_0_0_1_element_block_start(): sucking font size, blll :P");
		}
		
		data->tags = g_list_append (
				data->tags,
				g_object_new (
					CRIA_TYPE_LAYOUT_BLOCK,
					"alignment", cria_alignment_from_string (g_hash_table_lookup (hash_table, "align")),
					"family", g_strdup (g_hash_table_lookup (hash_table, "family")),
					"height", cria_unit_val_from_string (g_hash_table_lookup (hash_table, "height")),
					"height-unit", cria_unit_unit_from_string (g_hash_table_lookup (hash_table, "height")),
					"pos-x", cria_unit_val_from_string (g_hash_table_lookup (hash_table, "x")),
					"pos-x-unit", cria_unit_unit_from_string (g_hash_table_lookup (hash_table, "x")),
					"pos-y", cria_unit_val_from_string (g_hash_table_lookup (hash_table, "y")),
					"pos-y-unit", cria_unit_unit_from_string (g_hash_table_lookup (hash_table, "y")),
					"size", fsize,
					"title", g_hash_table_lookup (hash_table, "title"),
					"valignment", cria_valignment_from_string (g_hash_table_lookup (hash_table, "valign")),
					"width", cria_unit_val_from_string (g_hash_table_lookup (hash_table, "width")),
					"width-unit", cria_unit_unit_from_string (g_hash_table_lookup (hash_table, "width")),
					NULL
					)
				);
	}
	else
	{
		/* this is the slide block */
		data->tags = g_list_append (
				data->tags,
				g_object_new (
					CRIA_TYPE_SLIDE_BLOCK,
					"title", g_hash_table_lookup (hash_table, "title"),
					NULL
					)
				);
	}
}

static void
cria_sax_0_0_1_element_block_end (Sax001Data * data)
{
	g_return_if_fail(
				(
				 	CRIA_IS_LAYOUT_BLOCK (g_list_last (data->tags)->data)
					&&
					CRIA_IS_LAYOUT (g_list_nth (data->tags, g_list_length(data->tags) - 2)->data)
				)
				||
				(
					CRIA_IS_SLIDE_BLOCK (g_list_last (data->tags)->data)
					&&
					CRIA_IS_SLIDE (g_list_nth (data->tags, g_list_length(data->tags) - 2)->data)
				)
			);

	if (CRIA_IS_LAYOUT (g_list_nth (data->tags, g_list_length(data->tags) - 2)->data)) {
		if (data->cdata) {
			cria_layout_block_set_default_text (
					CRIA_LAYOUT_BLOCK (g_list_last (data->tags)->data),
					data->cdata->str);
			g_string_free (data->cdata, TRUE);
			data->cdata = NULL;
		}
		
		cria_layout_add_block(CRIA_LAYOUT (g_list_nth (data->tags, g_list_length(data->tags) - 2)->data),
				      CRIA_LAYOUT_BLOCK (g_list_last (data->tags)->data));
	} else {
		/* slide block */
		if (data->cdata)
		{
			cria_slide_block_set_content (
					CRIA_SLIDE_BLOCK (g_list_last (data->tags)->data),
					data->cdata->str);
			g_string_free (data->cdata, TRUE);
			data->cdata = NULL;
		}

		cria_slide_add_block(CRIA_SLIDE (g_list_nth (data->tags, g_list_length(data->tags) - 2)->data),
				     CRIA_SLIDE_BLOCK (g_list_last (data->tags)->data));
	}

	data->tags = g_list_remove (
			data->tags,
			g_list_last (data->tags)->data);
}

static void
cria_sax_0_0_1_element_br_end (Sax001Data* data) {
	if (data->cdata) {
		g_message("cria_sax_0_0_1_element_br_end(): data->cdata = 0x%x (%s)", (uintptr_t)data->cdata, (data->cdata==NULL)?NULL:data->cdata->str);
		g_string_append(data->cdata, "\n");
	}
}

static void
cria_sax_0_0_1_element_description_start (
		GHashTable	* hash_table,
		Sax001Data	* data)
{
	g_debug ("cria_sax_0_0_1_element_description_start(): start");
	data->tags = g_list_append (
			data->tags,
			g_strdup ("description"));
	g_debug ("cria_sax_0_0_1_element_description_start(): end");
}

static void
cria_sax_0_0_1_element_description_end (
		Sax001Data	* data)
{
	g_debug ("cria_sax_0_0_1_element_description_end(): start");
	data->tags = g_list_remove (
			data->tags,
			g_list_last (data->tags)->data);

	g_object_set (
			G_OBJECT (g_list_last (data->tags)->data),
			"description", data->cdata->str,
			NULL);

	g_string_free (data->cdata, TRUE);
	data->cdata = NULL;
	g_debug ("cria_sax_0_0_1_element_description_end(): end");
}

static void
cria_sax_0_0_1_element_layout_start (
		GHashTable	* hash_table,
		Sax001Data	* data)
{
	g_debug ("cria_sax_0_0_1_element_layout_start(): begin");

	data->tags = g_list_append (
			data->tags,
			g_object_new (
				CRIA_TYPE_LAYOUT,
				"name", g_hash_table_lookup (hash_table, "name"),
				NULL)
			);
	g_debug ("cria_sax_0_0_1_element_layout_start(): end");
}

static void
cria_sax_0_0_1_element_layout_end (
		Sax001Data	* data)
{
	g_debug ("cria_sax_0_0_1_element_layout_end(): begin");
	g_return_if_fail (CRIA_IS_THEME (g_list_nth (data->tags, g_list_length (data->tags) - 2)->data));
	g_return_if_fail (CRIA_IS_LAYOUT (g_list_last (data->tags)->data));

	cria_theme_add_layout (
			CRIA_THEME (g_list_nth (data->tags, g_list_length (data->tags) - 2)->data),
			CRIA_LAYOUT (g_list_last (data->tags)->data));

	data->tags = g_list_remove (
			data->tags,
			g_list_last (data->tags)->data);
	g_debug ("cria_sax_0_0_1_element_layout_end(): end");
}

static void
cria_sax_0_0_1_element_presentation_start (
		GHashTable	* hash_table,
		Sax001Data	* data)
{
	g_debug ("cria_sax_element_presentation_start(%i): start", __LINE__);
	
	data->tags = g_list_append (
			data->tags,
			g_object_new (
				CRIA_TYPE_PRESENTATION,
				"title", g_hash_table_lookup (hash_table, "title"),
				NULL)
			);
	
	g_debug ("cria_sax_element_presentation_start(%i): end", __LINE__);
}

static void
cria_sax_0_0_1_element_presentation_end (
		Sax001Data	* data)
{
	CriaPresentation	* presentation;

	g_return_if_fail (CRIA_IS_PRESENTATION (g_list_last (data->tags)->data));
	
	presentation = CRIA_PRESENTATION (
			g_list_last (data->tags)->data
			);

	data->tags = g_list_remove (
			data->tags,
			g_list_last (data->tags)->data);

	g_return_if_fail (data->tags == NULL);

	data->tags = (GList*) presentation;
}

static void
cria_sax_0_0_1_element_slide_start (
		GHashTable	* hash_table,
		Sax001Data	* data)
{
	g_assert (g_hash_table_lookup (hash_table, "title") != NULL);
	g_assert (g_hash_table_lookup (hash_table, "theme") != NULL);
	g_assert (g_hash_table_lookup (hash_table, "layout")!= NULL);

	data->tags = g_list_append (
			data->tags,
			g_object_new (
				CRIA_TYPE_SLIDE,
				"title", g_hash_table_lookup (hash_table, "title"),
				"theme", cria_presentation_get_theme(CRIA_PRESENTATION(data->tags->data), g_hash_table_lookup (hash_table, "theme")),
				"layout", cria_theme_get_layout(cria_presentation_get_theme(CRIA_PRESENTATION(data->tags->data), g_hash_table_lookup (hash_table, "theme")),g_hash_table_lookup (hash_table, "layout")),
				NULL)
			);
}

static void
cria_sax_0_0_1_element_slide_end (
		Sax001Data	* data)
{
	g_return_if_fail (CRIA_IS_SLIDE (g_list_last (data->tags)->data));
	g_return_if_fail (CRIA_IS_PRESENTATION (data->tags->data));

	cria_presentation_append_slide (
			data->tags->data,
			CRIA_SLIDE (g_list_last (data->tags)->data));

	data->tags = g_list_remove (
			data->tags,
			g_list_last (data->tags)->data);
}

static void
cria_sax_0_0_1_element_theme_start (
		GHashTable      * hash_table,
		Sax001Data      * data)
{
	g_debug ("cria_sax_0_0_1_element_theme_start(): start");
	data->tags = g_list_append (
			data->tags,
			g_object_new (
				CRIA_TYPE_THEME,
				"name", g_hash_table_lookup (hash_table, "name"),
				NULL)
			);
	g_debug ("cria_sax_0_0_1_element_theme_start(): end");
}

static void
cria_sax_0_0_1_element_theme_end (Sax001Data* data)
{
	g_debug ("cria_sax_0_0_1_element_theme_end(): start");
	g_return_if_fail (CRIA_IS_PRESENTATION (data->tags->data));
	g_return_if_fail (CRIA_IS_THEME (g_list_last (data->tags)->data));
	
	cria_presentation_add_theme (
			data->tags->data,
			CRIA_THEME (g_list_last (data->tags)->data));
			
	data->tags = g_list_remove (
			data->tags,
			g_list_last (data->tags)->data);
	g_debug ("cria_sax_0_0_1_element_theme_end(): end");
}

static cria_tag_data cria_0_0_1_tag_data[] =
{
	{"background",		&cria_sax_0_0_1_element_background_start,	&cria_sax_0_0_1_element_background_end		},
	{"block",		&cria_sax_0_0_1_element_block_start,		&cria_sax_0_0_1_element_block_end		},
	{"br",			NULL,						&cria_sax_0_0_1_element_br_end			},
	{"description",		&cria_sax_0_0_1_element_description_start,	&cria_sax_0_0_1_element_description_end		},
	{"layout",		&cria_sax_0_0_1_element_layout_start,		&cria_sax_0_0_1_element_layout_end		},
	{"presentation",	&cria_sax_0_0_1_element_presentation_start,	&cria_sax_0_0_1_element_presentation_end	},
	{"slide",		&cria_sax_0_0_1_element_slide_start,		&cria_sax_0_0_1_element_slide_end		},
	{"theme",		&cria_sax_0_0_1_element_theme_start,		&cria_sax_0_0_1_element_theme_end		},
	{NULL, NULL, NULL}
};

static GHashTable*
hash_table_from_strarray (gchar const **strarray)
{
	GHashTable	* hash_table;
	gchar	const	**iterator;
	gchar	const	* key = NULL;

	hash_table = g_hash_table_new_full (
			g_str_hash,
			g_str_equal,
			g_free,
			g_free);

	for (iterator = strarray; iterator && *iterator; iterator++)
	{
		if (!key)
			key = *iterator;
		else
		{
			g_debug ("hash_table_from_strarray(%i): \"%s\" => \"%s\"", __LINE__, key, *iterator);
			g_hash_table_insert (
					hash_table,
					g_strdup (key),
					g_strdup (*iterator));
			key = NULL;
		}
	}

	return hash_table;
}

static void
cria_0_0_1_start_element (
		gpointer	  userdata,
		gchar	const	* tagname,
		gchar	const	**attributes)
{
	guint		  index = 0;
	Sax001Data	* sax_data;
	
	sax_data = (Sax001Data*) userdata;

	if (sax_data->error != NULL) {
		/* if we already have entered some invalid state the just do not try to make
		 * more damage happen
		 */
		return;
	}

	if (sax_data->unknown == 0) {
		/* We know where we are */
		for (index = 0; cria_0_0_1_tag_data[index].tag_name; index++) {
			if (strcmp (tagname, cria_0_0_1_tag_data[index].tag_name) == 0) {
				break;
			}
		}
	}

	if (sax_data->unknown > 0 || cria_0_0_1_tag_data[index].tag_name == NULL) {
		g_debug ("cria_0_0_1_start_element(%i): unknown tag \"%s\" (Unknown depth: %d)", __LINE__, tagname, sax_data->unknown);
		sax_data->unknown++;
	} else {
		if (cria_0_0_1_tag_data[index].tag_start) {
			GHashTable	* attrs = NULL;

			if (attributes != NULL) {
				attrs = hash_table_from_strarray (attributes);
			}
			
			cria_0_0_1_tag_data[index].tag_start (attrs, sax_data);
		} else {
			g_debug ("cria_0_0_1_start_element(%i): skipping known tag \"%s\"", __LINE__, tagname);
		}
	}
}

static void
cria_0_0_1_end_element (
		gpointer	  userdata,
		gchar   const   * tagname)
{
	guint		  index;
	Sax001Data      * sax_data;
	
	sax_data = (Sax001Data*) userdata;

	if (sax_data->error != NULL) {
		/* ignore this (see explanation in cria_0_0_1_start_element) */
		return;
	}

	if (sax_data->unknown > 0)
	{
		g_debug ("cria_0_0_1_end_element(%i): unknown tag \"%s\"", __LINE__, tagname);
		sax_data->unknown--;
	}
	else
	{
		for (index = 0; cria_0_0_1_tag_data[index].tag_name; index++)
		{
			if (strcmp (tagname, cria_0_0_1_tag_data[index].tag_name) == 0)
				break;
		}

		if (cria_0_0_1_tag_data[index].tag_end)
		{
			cria_0_0_1_tag_data[index].tag_end (sax_data);
		}
		else
		{
			g_debug ("cria_0_0_1_end_element(%i): skipping known tag \"%s\"", __LINE__, tagname);
		}
	}
}

static void
cria_0_0_1_start_document (gpointer userdata) {
	Sax001Data	* sax_data;

	/* ensure that context->userData is correctly filled */
	sax_data = (Sax001Data*) userdata;
	sax_data->tags = NULL;
}

static void
cria_0_0_1_end_document (gpointer userdata) {
	/* anything to do here? */
}

static xmlSAXHandler cria_0_0_1_handler = {
	NULL, /* internalSubsetSAXFunc internalSubset;			*/
	NULL, /* isStandaloneSAXFunc isStandalone;			*/
	NULL, /* hasInternalSubsetSAXFunc hasInternalSubset;		*/
	NULL, /* hasExternalSubsetSAXFunc hasExternalSubset;		*/
	NULL, /* resolveEntitySAXFunc resolveEntity;			*/
	NULL, /* getEntitySAXFunc getEntity;				*/
	NULL, /* entityDeclSAXFunc entityDecl;				*/
	NULL, /* notationDeclSAXFunc notationDecl;			*/
	NULL, /* attributeDeclSAXFunc attributeDecl;			*/
	NULL, /* elementDeclSAXFunc elementDecl;			*/
	NULL, /* unparsedEntityDeclSAXFunc unparsedEntityDecl;		*/
	NULL, /* setDocumentLocatorSAXFunc setDocumentLocator;		*/
	cria_0_0_1_start_document,
	cria_0_0_1_end_document,
	(startElementSAXFunc)cria_0_0_1_start_element,
	(endElementSAXFunc)cria_0_0_1_end_element,
	NULL, /* referenceSAXFunc reference;				*/
	(charactersSAXFunc)cria_sax_0_0_1_characters,
	NULL, /* ignorableWhitespaceSAXFunc ignorableWhitespace;	*/
	NULL, /* processingInstructionSAXFunc processingInstruction;	*/
	NULL, /* commentSAXFunc comment;				*/
	NULL, /* warningSAXFunc warning;				*/
	NULL, /* errorSAXFunc error;					*/
	NULL  /* fatalErrorSAXFunc fatalError;				*/
};
