/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *       Adrien Beaucreux     <informancer@afturgurluk.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 * Copyright (C) 2004 Adrien Beuacreux
 * Copyright (C) 2004 Keith Sharp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <stdint.h>

#include <glib.h>
#include <glib-object.h>
#include <gconf/gconf-client.h>
#include <gtk/gtk.h>
#include <gnome.h>
#include <glade/glade.h>

#include "application.h"
#include "debug.h"
#include "libglade-support.h"
#include "main-window.h"
#include "slide-show.h"

enum {
	PROP_0,
	PROP_DEFAULT_FONT,
	PROP_GCONF_CLIENT
};

enum {
	CHANGED_DEFAULT_FONT_SIGNAL,
	N_SIGNALS
};

struct _CriaApplicationPriv {
	GList		     * windows;
	EggRecentModel	     * recent_model;
	PangoFontDescription * default_font;
	GConfClient          * gconf_client;
};

/* method prototypes */

/*int x = -1, y, w = -1, h = -1;*/
/*char *debug_modules = NULL;	*/

gboolean	start_off = 0,
		debug     = 0;

static CriaApplication	* self = NULL;

static const struct poptOption options[] = {
	{"debug", '\0', POPT_ARG_NONE, &debug, 0, N_("Enable debugging"), NULL},
	{"start-off", 's', POPT_ARG_NONE, &start_off, 0,
		N_("Start a presentation (and open a file selection if no file "
			"is given at the command line)"), NULL},
/*	{NULL, 'x', POPT_ARG_INT, &x, 0, N_("X position of window"), N_("X")},
	{NULL, 'y', POPT_ARG_INT, &y, 0, N_("Y position of window"), N_("Y")},
	{NULL, 'w', POPT_ARG_INT, &w, 0, N_("width of window"), N_("WIDTH")},
	{NULL, 'h', POPT_ARG_INT, &h, 0, N_("height of window"), N_("WIDTH")},
	{"debug-modules", '\0', POPT_ARG_STRING, &debug_modules, 0,
		N_("Modules to enable debugging in"), N_("MODULES")},
*/	{NULL, '\0', 0, NULL, 0}
};

static void cria_application_get_property      (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static void cria_application_set_property      (GObject		* object,
						guint		  prop_id,
						const GValue	* value,
						GParamSpec	* param_spec);
static void cria_open_presentation_from_filename
					       (char const	* filename);
static void cria_application_init	       (CriaApplication	* self);

static void notify_default_font_changed        (GConfClient *client,
						guint cnxn_id,
						GConfEntry *entry,
						gpointer user_data);

static guint		  cria_application_signals[N_SIGNALS] = { 0 };

/**
 * criawips_init:
 * @argc: pointer to argc as delivered to the main() method
 * @argv: pointer to the argument vector as delivered to the main() method
 *
 * Initializes the application. #criawips_init parses the command line arguments
 * and strips those arguments it understood from argc and argv.
 */
void
criawips_init(int *argc, char ***argv) {
#warning "criawips_init(): FIXME: check for a runnning instance, delegate command line tasks to it"
	const gchar	**filenames;
	GnomeProgram	* gnome_program;
	GValue		* value;
	poptContext	  popt_context;
	GError		* error = NULL;

	g_set_prgname (PACKAGE);
	g_debug("criawips_init(): start");

	/* initialize i18n */
#ifdef ENABLE_NLS
	bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
	textdomain (GETTEXT_PACKAGE);
#endif

	/* initializing libraries */
	  /* GTK+ */
	gtk_init (argc, argv);

	gtk_window_set_default_icon_from_file(PACKAGE_DATA_DIR "/pixmaps/criawips.png", &error);
	if(error) {
		gchar	* message = g_strdup_printf(_("Couldn't set window icon, you have to live without; reason: %s"), error->message);
		cria_application_show_error_dialog(NULL, _("Error"), message);
		g_error_free(error);
		g_free(message);
		error = NULL;
	}

	  /* libgnome(ui) */
	gnome_program = gnome_program_init (
			PACKAGE, PACKAGE_VERSION,
			LIBGNOMEUI_MODULE,
			*argc, *argv,
			GNOME_PARAM_POPT_TABLE, options,
			GNOME_PARAM_POPT_FLAGS, 0,
			NULL);

	  /* libglade */
	glade_init();
	glade_set_custom_handler(criawips_libglade_custom_handler, NULL);

	/* libgoffice */
	go_font_init();

	value = g_new0(GValue, 1);
	g_value_init(value,
		     G_TYPE_POINTER);
	g_object_get_property(G_OBJECT(gnome_program),
			      GNOME_PARAM_POPT_CONTEXT,
			      value);

	popt_context = (poptContext) g_value_get_pointer(value);
	filenames = poptGetArgs(popt_context);

	cria_application_get_instance();

	if(start_off) {
		g_debug("criawips_init(): opening presentation mode");
		
		if(filenames && filenames[0]) {
			if(filenames[1]) {
				GtkWidget* dialog = gtk_message_dialog_new_with_markup(NULL,
									  	      0,
									  	      GTK_MESSAGE_WARNING,
									  	      GTK_BUTTONS_CLOSE,
									   	      "<span weight=\"bold\">%s</span>\n\n%s",
									   	      _("Criawips can only show one presentation"),
										      _("You tried to show more than one presentation. Criawips cannot handle this yet."));
				gtk_widget_show(dialog);
				gtk_dialog_run(GTK_DIALOG(dialog));
				gtk_widget_hide(dialog);

				while(gtk_events_pending()) {
					gtk_main_iteration();
				}
				gtk_object_destroy(GTK_OBJECT(dialog));
			}
			
			cria_open_presentation_from_filename (filenames[0]);
		} else {
			GtkWidget	* dialog;
			GtkFileFilter	* filter;
			gchar		* filename;

			dialog = gtk_file_chooser_dialog_new(_("Select a Presentation"),
							     NULL,
							     GTK_FILE_CHOOSER_ACTION_OPEN,
							     GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
							     GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
							     NULL);
			filter = gtk_file_filter_new();
			gtk_file_filter_set_name(filter, _("Criawips Presentations"));
			gtk_file_filter_add_pattern(filter, "*.criawips");
			gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog),
						    filter);
			switch(gtk_dialog_run(GTK_DIALOG(dialog))) {
			case GTK_RESPONSE_ACCEPT:
				gtk_widget_hide(dialog);
				while (gtk_events_pending()) {
					gtk_main_iteration();
				}
				filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(dialog));
				cria_open_presentation_from_filename(filename);
				g_free(filename);
				gtk_object_destroy(GTK_OBJECT(dialog));
				break;
			case GTK_RESPONSE_CANCEL:
				criawips_quit();
				break;
			default:
				g_assert_not_reached();
				break;
			};
		}
	} else {
		g_debug("criawips_init(): opening main window(s)");
		
		if(!filenames) {
			/* Open the criawips main window here */
			gtk_widget_show(GTK_WIDGET(cria_main_window_new(NULL)));
		} else {
			/* open each file in a new window */
			const gchar	**iterator;
			g_debug("criawips_init(): filenames is a char** at 0x%x", (uintptr_t)filenames);
			for(iterator = filenames; iterator && *iterator; iterator++) {
				g_debug("criawips_init(): iterator is now 0x%x >> 0x%x (%s)", (uintptr_t)iterator, (uintptr_t)(*iterator), *iterator);
				gtk_widget_show(GTK_WIDGET(cria_main_window_new(*iterator)));
			}
		}
	}

	g_debug("criawips_init(): end");
}

/**
 * criawips_quit:
 *
 * Quit Criawips, close all application windows and then exit the application.
 */
void
criawips_quit (void) {
#warning "criawips_quit(): FIXME: Ask open windows to save their content"

	/* FIXME: should be freed in the application destructor */
        g_object_unref (G_OBJECT (self->priv->gconf_client));

	/* get the application object and quit all window instances */
	if (gtk_main_level() > 0) {
		gtk_main_quit ();
	} else {
		exit(0);
	}
}

/*****************************************************************************
 * file selection handling stuff                                             *
 *****************************************************************************/

static void
cria_open_presentation_from_filename(const gchar*filename) {
	CriaPresentation	* pres;
	GtkWidget		* show;
	GError			* error = NULL;

	g_assert(filename != NULL);
	g_debug("Opening file '%s'", filename);

	pres = cria_presentation_new_from_file(filename, &error);

	if(error) {
		char		* error_message = g_strdup_printf("<span weight=\"bold\">%s</span>\n\n%s %s:\n<span style=\"italic\">%s</span>.",
								  _("The Presentation could not be opened"),
								  _("For some reason the presentation you wanted to open could not be opened."),
								  _("The reason was:"),
								  error->message);
		cria_application_show_error_dialog(NULL,
						   _("Error opening file"),
						   error_message);
		g_free(error_message);
		g_error_free(error);
		gtk_main_quit();
	} else {
		show = cria_slide_show_new(pres);
		g_signal_connect(G_OBJECT(show), "destroy",
				 G_CALLBACK(criawips_quit), NULL);
	}
}

/*---------------------------------------------------------------------------*/

static void
cria_application_class_init (CriaApplicationClass* cria_application_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_application_class);

	/* setting up signal system */
	/* We do not have a default handler. */
	cria_application_class->changed_default_font = NULL;

	cria_application_signals[CHANGED_DEFAULT_FONT_SIGNAL] = g_signal_new("default-font-changed",
									     CRIA_TYPE_APPLICATION,
									     G_SIGNAL_RUN_LAST,
									     G_STRUCT_OFFSET(CriaApplicationClass,
											     changed_default_font),
									     NULL,
									     NULL,
									     g_cclosure_marshal_VOID__VOID,
									     G_TYPE_NONE,
									     0);

	/* setting up property system */
	g_object_class->set_property = cria_application_set_property;
	g_object_class->get_property = cria_application_get_property;

	/* FIXME: We have runtime errors in the following function */
	g_object_class_install_property(g_object_class,
					PROP_DEFAULT_FONT,
					g_param_spec_boxed("default_font",
							   "Default_font",
							   "The default font for the slides",
							   PANGO_TYPE_FONT_DESCRIPTION, 
							   G_PARAM_READWRITE));

	g_object_class_install_property(g_object_class,
					PROP_GCONF_CLIENT,
					g_param_spec_object("gconf_client",
							    "GConf_client",
							    "The Gconfclient for the application",
							    GCONF_TYPE_CLIENT,
							    G_PARAM_READABLE));
}

/**
 * cria_application_register_window:
 * @window: a main window to be registeres by the application
 *
 * Registers a #CriaMainWindow to the application. Thus it's being propted to
 * close the file before quitting.
 */
void
cria_application_register_window(CriaMainWindow	* window) {
	CriaApplication	* self;

	g_debug("cria_application_register_window(): start");

	g_return_if_fail(window != NULL);
	g_assert(CRIA_IS_MAIN_WINDOW(window));

	self = cria_application_get_instance();

	self->priv->windows = g_list_prepend(
			self->priv->windows,
			g_object_ref(window));

	/*g_object_notify(self, "windows");*/

	g_debug("cria_application_register_window(): end");
}

/**
 * cria_application_get_instance:
 *
 * As #CriaApplication is a singleton, this method delivers the single
 * instance.
 *
 * Return value: the instance of the application
 */
CriaApplication*
cria_application_get_instance (void) {
	if (!self) {
		self = g_object_new(CRIA_TYPE_APPLICATION, NULL);
	}

	return self;
}

static void
cria_application_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaApplication	* self;

	self = CRIA_APPLICATION (object);

	switch (prop_id) {
	case PROP_DEFAULT_FONT:
		g_value_set_object(value, self->priv->default_font);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

static void
cria_application_init(CriaApplication* app) {
	GError * error = NULL;

	if (!self) {
		self = app;
	} else {
		g_critical("%s", "CriaApplication can only be instantiated once");
	}
	
	self->priv = g_new0(CriaApplicationPriv,1);
	self->priv->windows = NULL;

	/* Initialise the recent-files support */
	self->priv->recent_model = egg_recent_model_new(EGG_RECENT_MODEL_SORT_MRU);
	egg_recent_model_set_filter_groups(self->priv->recent_model, PACKAGE, NULL);
	egg_recent_model_set_limit(self->priv->recent_model, 5);

	/* Initialise the preferences */
	/* Initialise GConf */
	self->priv->gconf_client = gconf_client_get_default();

	g_assert(error == NULL);

	gconf_client_add_dir(self->priv->gconf_client,
			     PACKAGE_GCONF_PATH,
			     GCONF_CLIENT_PRELOAD_NONE,
			     &error);
	if(error) {
		gchar	* message = g_strdup_printf(_("Couldn't open preferences directory: %s"), error->message);
		cria_application_show_error_dialog(NULL, _("Error"), message);
		g_error_free(error);
		g_free(message);
		error = NULL;
	}

	/* Set the default font */
	notify_default_font_changed(NULL,
				    0,
				    NULL,
				    NULL);

	/* Set up notifications */
	gconf_client_notify_add(self->priv->gconf_client,
			        PACKAGE_GCONF_PATH "/default_fallback_font",
				notify_default_font_changed,
				NULL,
				NULL,
				&error);
	if(error) { 
		g_debug("CriaApplication::init(): Couldn't add gconf notification: '%s'", error->message);
		g_error_free(error);
		error = NULL;
	}
}

static void
cria_application_set_property (
		GObject		* object,
		guint		  prop_id,
		const	GValue	* value,
		GParamSpec	* param_spec)
{
	CriaApplication	* self;
	
	self = CRIA_APPLICATION (object);
	
	switch (prop_id) {
	case PROP_DEFAULT_FONT:
		cria_application_set_default_font_name(g_value_get_string(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

/**
 * cria_application_show_error_dialog:
 * @parent: a #GtkWindow or NULL
 * @title: the dialog title
 * @message: the dialog message
 *
 * Shows an error dialog. To be used when fatal things happen (e.g. loading the
 * ui definitions from the glade file failed). @parent is used to specify a
 * window that this dialog is transient to.
 */
void
cria_application_show_error_dialog (GtkWindow* parent, gchar const* title, gchar const* message) {
	GtkWidget	* dialog,
			* label;

	dialog = gtk_dialog_new_with_buttons(title,
                                             parent,
					     GTK_DIALOG_NO_SEPARATOR,
					     GTK_STOCK_CLOSE, GTK_RESPONSE_ACCEPT,
					     NULL);
	label = gtk_label_new("");
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_FILL);
	gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
	gtk_label_set_markup(GTK_LABEL(label), message);
	gtk_widget_show(label);

	gtk_container_add(GTK_CONTAINER(GTK_DIALOG(dialog)->vbox),
			  label);
	gtk_widget_show_all(dialog);
	
	gtk_dialog_run(GTK_DIALOG(dialog));
	gtk_object_destroy(GTK_OBJECT(dialog));
}

GType
cria_application_get_type (void) {
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaApplicationClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_application_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaApplication),
			0,
			(GInstanceInitFunc)cria_application_init,
			0
		};

		type = g_type_register_static (
				G_TYPE_OBJECT,
				"CriaApplication",
				&info,
				0);
	}

	return type;
}

EggRecentModel *
cria_application_get_recent_model(void) {
	return self->priv->recent_model;
}

void
cria_application_add_to_recent(const gchar* text_uri) {
	EggRecentModel	* recent_model;
	EggRecentItem	* recent_item;

	recent_model = cria_application_get_recent_model();

	recent_item = egg_recent_item_new_from_uri(text_uri);
	egg_recent_item_add_group(recent_item, PACKAGE);
	egg_recent_model_add_full(recent_model, recent_item);
	egg_recent_item_unref(recent_item);
}

PangoFontDescription *
cria_application_get_default_font(void) {
	return self->priv->default_font;
}

void
cria_application_set_default_font_name(const gchar* font) {

	GError* error = NULL;

	g_assert(font);

	gconf_client_set_string(self->priv->gconf_client,
				PACKAGE_GCONF_PATH "/default_fallback_font",
				font,
				&error);

	if(error) {
		gchar	* message = g_strdup_printf(_("Couldn't set default font: %s"), error->message);
		cria_application_show_error_dialog(NULL, _("Error"), message);
		g_error_free(error);
		g_free(message);
		error = NULL;
	}
}

static void
notify_default_font_changed(GConfClient *client,
			    guint cnxn_id,
			    GConfEntry *entry,
			    gpointer user_data)
{
	gchar  *font = NULL;
	GError *error = NULL;

	PangoFontDescription *old_default = NULL;
	
	g_debug("CriaApplication::notify_default_font_changed(): start");

	old_default = self->priv->default_font;

	font = gconf_client_get_string(self->priv->gconf_client,
				       PACKAGE_GCONF_PATH "/default_fallback_font",
				       &error);
	if(error) {  
		g_debug("CriaApplication::init(): Couldn't load default fallback font: '%s'", error->message);
		g_error_free(error);
		error = NULL;
	} else if(!font) {
		g_debug("CriaApplication::init(): Fallback font was null.");
	} else {
		self->priv->default_font = pango_font_description_from_string(font);
		if(old_default) {
			pango_font_description_free(old_default);
			old_default = NULL;
		}
		g_signal_emit(self, cria_application_signals[CHANGED_DEFAULT_FONT_SIGNAL], 0, NULL);
	}
	g_debug("CriaApplication::notify_default_font_changed(): end");
}
