/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *       Adrien Beaucreux     <informancer@afturgurluk.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 * Copyright (C) 2004 Adrien Beaucreux
 * Copyright (C) 2004 Keith Sharp
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string.h>
#include <stdint.h>

#include <glib.h>
#include <glib-object.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <libgnomeui/libgnomeui.h>

#include "debug.h"
#include "application.h"
#include "main-window.h"
#include "presentation.h"
#include "presentation-properties-dialog.h"
#include "preferences-dialog.h"
#include "slide-editor.h"
#include "slide-list.h"
#include "slide-show.h"
#include "title-bar.h"
#include "egg-recent.h"

enum {
	PROP_0,
	PROP_PRESENTATION
};

enum {
	SIGNAL,
	N_SIGNALS
};

struct _CriaMainWindowPriv {
	GtkWidget	* hpaned,
			* vbox,
			* slide_editor,
			* slide_list,
			* status_bar,
			* title_bar,
			* zoom_combo;
	GtkUIManager	* ui_manager;
	CriaPresentation* presentation;
	EggRecentViewGtk* recent_view;
};

static	void	cria_main_window_get_property	       (GObject		* object,
							guint		  prop_id,
							GValue		* value,
							GParamSpec	* param_spec);
static	void	cria_main_window_init		       (CriaMainWindow	* self);
static	void	cria_main_window_open		       (CriaMainWindow	* self,
							gpointer	  data);
static	void	cria_main_window_selected_slide_changed(CriaMainWindow	* self,
							GtkTreeSelection* selection);
static	void	cria_main_window_set_property	       (GObject		* object,
							guint		  prop_id,
							const	GValue	* value,
							GParamSpec	* param_spec);
static void	  cria_main_window_set_zoom_from_combo (CriaMainWindow  * template,
							GtkEntry	* entry);
static	void	cria_main_window_start_off	       (CriaMainWindow	* self);
static	void	cria_main_window_set_presentation_from_text_uri	(CriaMainWindow* self,
							gchar	* text_uri);
#if 0
static	void	cria_main_window_signal		       (CriaMainWindow	* template,
							const	gchar	* string);

static	guint	cria_main_window_signals[N_SIGNALS] = { 0 };
#endif

static void cb_action_about_info(GtkAction* action, CriaMainWindow* window);
static void cb_action_application_quit(GtkAction* action, CriaMainWindow* window);
static void cb_action_file_new(GtkAction* action, CriaMainWindow* window);
static void cb_action_file_open(GtkAction* action, CriaMainWindow* window);
static void cb_action_insert_slide(GtkAction* action, CriaMainWindow* window);
static void cb_action_main_window_close(GtkAction* action, CriaMainWindow* window);
static void cb_action_presentation_properties(GtkAction* action, CriaMainWindow* window);
static void cb_action_show_preferences(GtkAction* action, CriaMainWindow* window);
static void cb_action_view_first(GtkAction* action, CriaMainWindow* window);
static void cb_action_view_last(GtkAction* action, CriaMainWindow* window);
static void cb_action_view_next(GtkAction* action, CriaMainWindow* window);
static void cb_action_view_presentation(GtkAction* action, CriaMainWindow* window);
static void cb_action_view_previous(GtkAction* action, CriaMainWindow* window);

static void cb_recent_files_open(GtkWidget *widget, const EggRecentItem *item, CriaMainWindow* self);

static GtkActionEntry entries[] = {
	/* use this one when you add new items */
	{"FooBarDummy",		GTK_STOCK_MISSING_IMAGE,NULL},
	{"ApplicationQuit",	GTK_STOCK_QUIT,		N_("_Quit"),
	 NULL,			N_("Quit Criawips and close all Windows"),
	 G_CALLBACK(cb_action_application_quit)},
	{"FileMenu",		NULL, 			N_("_File")},
	{"FileNew",		GTK_STOCK_NEW,		N_("_New"),
	 "<control>N",		N_("Create a new Presentation"),
	 G_CALLBACK(cb_action_file_new)},
	{"FileOpen",		GTK_STOCK_OPEN,		N_("_Open"),
	 "<control>O",		N_("Open a Presentation"),
	 G_CALLBACK(cb_action_file_open)},
	{"EditMenu",            NULL,                   N_("_Edit")},
	{"PreferencesOpen",     GTK_STOCK_PREFERENCES,  N_("_Preferences"),
	 NULL,                  N_("Show Preferences"),
	 G_CALLBACK(cb_action_show_preferences)},
	{"InsertMenu",		NULL,			N_("_Insert")},
	{"InsertSlide",		GTK_STOCK_ADD,		N_("_Slide"),
	 NULL,			N_("Insert a new slide after this one"),
	 G_CALLBACK(cb_action_insert_slide)},
	{"HelpMenu",		NULL,			N_("_Help")},
	{"HelpInfo",		GNOME_STOCK_ABOUT,	N_("_About"),
	 NULL,			N_("Show Information about Criawips"),
	 G_CALLBACK(cb_action_about_info)},
	{"MainWindowClose",	GTK_STOCK_CLOSE,	N_("_Close"),
	 "<control>W",			N_("Close this window"),
	 G_CALLBACK(cb_action_main_window_close)},
	{"PresentationProperties", GTK_STOCK_PROPERTIES, N_("_Properties"),
	 NULL,			NULL,
	 G_CALLBACK(cb_action_presentation_properties)},
	{"ViewMenu",		NULL,			N_("_View")},
	{"ViewFirst",		GTK_STOCK_GOTO_FIRST,	N_("_First Slide"),
	 NULL,			N_("Display the first Slide of the Presentation"),
	 G_CALLBACK(cb_action_view_first)},
	{"ViewLast",		GTK_STOCK_GOTO_LAST,	N_("_Last Slide"),
	 NULL,			N_("Display the last Slide of the Presentation"),
	 G_CALLBACK(cb_action_view_last)},
	{"ViewNext",		GTK_STOCK_GO_FORWARD,	N_("_Next Slide"),
	 "Page_Down",		N_("Display the next Slide"),
	 G_CALLBACK(cb_action_view_next)},
	{"ViewPresentation",	GTK_STOCK_EXECUTE,	N_("_Start Presentation"),
	 NULL,			N_("Show this Presentation"),
	 G_CALLBACK(cb_action_view_presentation)},
	{"ViewPrevious",	GTK_STOCK_GO_BACK,	N_("_Previous Slide"),
	 "Page_Up",		N_("Display the previous Slide"),
	 G_CALLBACK(cb_action_view_previous)}
};

static void
cria_main_window_class_init(CriaMainWindowClass* cria_main_window_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_main_window_class);
#if 0
	/* setting up signal system */
	cria_main_window_class->signal = cria_main_window_signal;

	cria_main_window_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_MAIN_WINDOW,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaMainWindowClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_main_window_set_property;
	g_object_class->get_property = cria_main_window_get_property;

	g_object_class_install_property(g_object_class,
					PROP_PRESENTATION,
					g_param_spec_object("presentation",
							    "Presentation",
							    "This is the presentation that's currently assotiated with this main window",
							    CRIA_TYPE_PRESENTATION,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

static void
cria_main_window_close(CriaMainWindow* self) {
#warning "MainWindow::close(): FIXME: ask whether we need to save"
#warning "MainWindow::close(): FIXME: close the window only, don't quit the application"
	criawips_quit();
}

/*
 * cria_main_window_displayed_slide_changed:
 * @self: ...
 * @slide_view: ...
 *
 * ...
 */
static void
cria_main_window_displayed_slide_changed(CriaMainWindow* self, gpointer nulldata) {
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->presentation && CRIA_IS_PRESENTATION(self->priv->presentation));
	g_assert(self->priv->slide_editor != NULL && CRIA_IS_SLIDE_EDITOR(self->priv->slide_editor));

#warning "MainWindow::displayedSlideChanged(): FIXME: This one sucks really, move this to the place where the SlideEditor's slide is being changed"
	cria_title_bar_set_slide(CRIA_TITLE_BAR(self->priv->title_bar), cria_slide_editor_get_slide(CRIA_SLIDE_EDITOR(self->priv->slide_editor)));
}

CriaPresentation*
cria_main_window_get_presentation (
		CriaMainWindow	* self)
{
	g_return_val_if_fail (CRIA_IS_MAIN_WINDOW(self), NULL);
	
	return self->priv->presentation;
}

static void
cria_main_window_get_property (
		GObject		* object,
		guint		  prop_id,
		GValue		* value,
		GParamSpec	* param_spec)
{
	CriaMainWindow	* self;

	self = CRIA_MAIN_WINDOW (object);

	switch (prop_id)
	{
	case PROP_PRESENTATION:
		g_value_set_object (
				value,
				self->priv->presentation);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (
				object,
				prop_id,
				param_spec);
		break;
	}
}

GType
cria_main_window_get_type (void)
{
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaMainWindowClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_main_window_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaMainWindow),
			0,
			(GInstanceInitFunc)cria_main_window_init,
			0
		};

		type = g_type_register_static (
				GTK_TYPE_WINDOW,
				"CriaMainWindow",
				&info,
				0);
	}

	return type;
}

static void
cria_main_window_init(CriaMainWindow* self) {
	GtkWidget	* helper,
			* recent_widget,
			* scrolled,
			* text_pane;
	EggRecentModel	* recent_model;
	GError		* error = NULL;
	GtkActionGroup	* actions;
	g_debug("MainWindow::init(): start");
	
	/* creating the private data */
	self->priv = g_new0(CriaMainWindowPriv,1);

	/* set the empty presentation as a default */
	self->priv->presentation = cria_presentation_new_default();

	/* GtkAction UI API */
	self->priv->ui_manager = gtk_ui_manager_new();
	actions = gtk_action_group_new("Actions");
	gtk_action_group_set_translation_domain(actions, GETTEXT_PACKAGE);
	gtk_action_group_add_actions(actions, entries, G_N_ELEMENTS(entries), self);
	gtk_ui_manager_insert_action_group(self->priv->ui_manager, actions, 0);
	gtk_window_add_accel_group(GTK_WINDOW(self),
				   gtk_ui_manager_get_accel_group(self->priv->ui_manager));
	gtk_ui_manager_add_ui_from_file(self->priv->ui_manager, PACKAGE_DATA_DIR "/" PACKAGE "/data/criawips.ui", &error);

	if(error) {
#warning "MainWindow::init(): FIXME: Add some nice warning if this fails"
		g_critical("MainWindow::init(): loading of the ui failed: %s (is your installation broken?)", error->message);
		g_error_free(error);
	}
	
	self->priv->vbox = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(self->priv->vbox);
	gtk_container_add(GTK_CONTAINER(self), self->priv->vbox);

	gtk_box_pack_start(GTK_BOX(self->priv->vbox), gtk_ui_manager_get_widget(self->priv->ui_manager, "/menubar"), FALSE, FALSE, 0);
	helper = gtk_handle_box_new();
	gtk_box_pack_start(GTK_BOX(self->priv->vbox), helper, FALSE, FALSE, 0);
	gtk_container_add(GTK_CONTAINER(helper), gtk_ui_manager_get_widget(self->priv->ui_manager, "/toolbar1"));
	gtk_widget_show_all(helper);
	helper = NULL;
	//gtk_box_pack_start(GTK_BOX(self->priv->vbox), gtk_ui_manager_get_widget(self->priv->ui_manager, "/toolbar2"), FALSE, FALSE, 0);
	
	/* Recent files support using egg-recent */	
	recent_model = cria_application_get_recent_model();
	recent_widget = gtk_ui_manager_get_widget(self->priv->ui_manager, "/menubar/FileMenu/EggRecentDocuments");
	self->priv->recent_view = egg_recent_view_gtk_new(gtk_widget_get_parent(recent_widget), recent_widget);
	egg_recent_view_gtk_set_trailing_sep(self->priv->recent_view, TRUE);

	egg_recent_view_set_model(EGG_RECENT_VIEW(self->priv->recent_view), recent_model);
	g_signal_connect(G_OBJECT(self->priv->recent_view), "activate", G_CALLBACK(cb_recent_files_open), self);

	self->priv->hpaned = gtk_hpaned_new();
	gtk_widget_show(self->priv->hpaned);
	gtk_box_pack_start(GTK_BOX(self->priv->vbox), self->priv->hpaned, TRUE, TRUE, 0);

	/* the text pane over the slide list */
	text_pane = cria_text_pane_new(_("Slides"));
	gtk_widget_show(text_pane);
	
	/* the slide list */
	self->priv->slide_list = cria_slide_list_new(self->priv->presentation);
	gtk_widget_show(self->priv->slide_list);
	
	/* the scrolled window to put the slide list into */
	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_AUTOMATIC);
	gtk_widget_show(scrolled);
	gtk_container_add(GTK_CONTAINER(scrolled), self->priv->slide_list);

	/* the vbox containing the slide list and the text pane */
	helper = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(helper), text_pane, FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(helper), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(helper);
	gtk_paned_pack1(GTK_PANED(self->priv->hpaned), helper,
			TRUE, FALSE);
	helper = NULL;
	scrolled = NULL;

	/* the vpaned for the title bar and the editor */
	helper = gtk_vbox_new(FALSE, 0);
	gtk_widget_show(helper);
	gtk_paned_pack2(GTK_PANED(self->priv->hpaned), helper,
			TRUE, FALSE);
	
	/* the title bar */
	/* we could to this by getting the selection of the SlideList */
	self->priv->title_bar = cria_title_bar_new(cria_slide_list_get_selected(CRIA_SLIDE_LIST(self->priv->slide_list)));
	gtk_widget_show(self->priv->title_bar);
	gtk_box_pack_start(GTK_BOX(helper), self->priv->title_bar, FALSE, FALSE, 0);

	/* the zoom combo box */
	self->priv->zoom_combo = gtk_combo_box_entry_new_text();
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->priv->zoom_combo), "100%");
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->priv->zoom_combo), "75%");
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->priv->zoom_combo), "50%");
	gtk_combo_box_append_text(GTK_COMBO_BOX(self->priv->zoom_combo), "25%");
	gtk_entry_set_width_chars(GTK_ENTRY(gtk_bin_get_child(GTK_BIN(self->priv->zoom_combo))), 5);
	g_signal_connect_swapped(gtk_bin_get_child(GTK_BIN(self->priv->zoom_combo)), "changed",
				 G_CALLBACK(cria_main_window_set_zoom_from_combo), self);
	gtk_widget_show(self->priv->zoom_combo);
	cria_text_pane_pack_end(CRIA_TEXT_PANE(self->priv->title_bar), self->priv->zoom_combo);

	/* slide editor */
	self->priv->slide_editor = cria_slide_editor_new(cria_presentation_get_slide(self->priv->presentation, 0));
	gtk_widget_show(self->priv->slide_editor);
	gtk_box_pack_start_defaults(GTK_BOX(helper), self->priv->slide_editor);
	helper = NULL;

	/* status bar */
	self->priv->status_bar = gtk_statusbar_new();
	gtk_widget_show(self->priv->status_bar);
	gtk_box_pack_start(GTK_BOX(self->priv->vbox), self->priv->status_bar, FALSE, FALSE, 0);

#warning "MainWindow::init(): FIXME: get this value from GConf"
	gtk_paned_set_position(GTK_PANED(self->priv->hpaned), 150);

	/* connecting signals */
	/* window */
#warning "MainWindow::init(): FIXME: handle this properly"
	g_signal_connect(self, "delete-event",
			G_CALLBACK(criawips_quit), NULL);

	/* slide list */
#warning "MainWindow::init(): FIXME: make this one nicer"
	g_signal_connect_swapped(gtk_tree_view_get_selection(GTK_TREE_VIEW(self->priv->slide_list)), "changed",
				 G_CALLBACK(cria_main_window_selected_slide_changed), self);

	/* slide view */
	g_signal_connect_swapped(self->priv->slide_editor, "notify::slide",
				 G_CALLBACK(cria_main_window_displayed_slide_changed), self);

	/* update the last stuff */
	/* FIXME: set the value from the presentation */
	gtk_combo_box_set_active(GTK_COMBO_BOX(self->priv->zoom_combo), 2);

	g_debug("MainWindow::init(): end");
}

static void
cria_main_window_insert_slide(CriaMainWindow* self) {
	CriaSlide* slide;
	gint	   pos = 0;
	
	g_return_if_fail(CRIA_IS_MAIN_WINDOW(self));

	/* get the current slide */
	slide = cria_slide_editor_get_slide(CRIA_SLIDE_EDITOR(self->priv->slide_editor));
	pos = cria_presentation_get_slide_index(self->priv->presentation,
						slide) + 1;
	cria_slide_new_pos(self->priv->presentation, pos);
	cria_slide_list_refresh(CRIA_SLIDE_LIST(self->priv->slide_list));
}

/**
 * cria_main_window_new:
 * @filename: filename of the new window
 *
 * Creates a new main window displaying the presentation that @filename
 * points to. It automatically registers the window for the application
 * to make it managed by the application. @filename can be NULL to create
 * a window without a presentation.
 *
 * Returns a new main window
 */
GtkWidget*
cria_main_window_new(const gchar* filename) {
	CriaMainWindow	* self;
	CriaPresentation* presentation;

	g_debug("cria_main_window_new(): start");
	
	if (filename != NULL) {
		GError			* error = NULL;

		g_debug("cria_main_window_new(): got filename 0x%x(\"%s\")", (uintptr_t)filename, filename);
		
		presentation = cria_presentation_new_from_file(filename, &error);

		if (error) {
			/* NOTE: keep these strings in sync with the ones from application.c */
			char            * error_message = g_strdup_printf("<span weight=\"bold\">%s</span>\n\n%s %s:\n<span style=\"italic\">%s</span>.",
									  _("The Presentation could not be opened"),
									  _("For some reason the presentation you wanted to open could not be opened."),
									  _("The reason was:"),
									  error->message);

			cria_application_show_error_dialog(NULL,
					 		   _("Error opening file"),
							   error_message);
			g_free(error_message);
			g_error_free(error);
		}
	} else {
		presentation = cria_presentation_new_default();
	}

	g_debug("MainWindow::new(): creating the new main window with presentation 0x%x (%s)", (uintptr_t)presentation, G_OBJECT_TYPE_NAME(presentation));
	self = g_object_new(CRIA_TYPE_MAIN_WINDOW, "presentation", presentation, NULL);

	g_debug("cria_main_window_new(): in between");
	
	cria_application_register_window(self);

	g_debug("cria_main_window_new(): end");
	return GTK_WIDGET(self);
}

static void
cria_main_window_open(CriaMainWindow* self, gpointer d) {
	GtkFileFilter	* filter;
	GtkWidget	* dialog = gtk_file_chooser_dialog_new(_("Open File"),
							       NULL,
							       GTK_FILE_CHOOSER_ACTION_OPEN,
							       GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
							       GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
							       NULL);
	gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(dialog), TRUE);
	filter = gtk_file_filter_new();
	gtk_file_filter_add_pattern(filter, "*.criawips");
	gtk_file_filter_set_name(filter, _("Criawips Presentations"));
	gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(dialog),
				    filter);

	if(gtk_dialog_run(GTK_DIALOG(dialog)) == GTK_RESPONSE_ACCEPT) {
		gtk_widget_hide(dialog);

		while(gtk_events_pending()) {
			gtk_main_iteration();
		}
		
		GSList			* uris	   = gtk_file_chooser_get_uris(GTK_FILE_CHOOSER(dialog));
		/*g_object_unref(dialog); dialog = NULL;*/

		if(uris && g_slist_length(uris) == 1) {
			cria_main_window_set_presentation_from_text_uri(self, (gchar*)uris->data);
		} else {
			GSList	* it = NULL;
			for(it = uris; it; it = it->next) {
				GtkWidget *mainwin = cria_main_window_new(NULL);
				cria_main_window_set_presentation_from_text_uri(self, (gchar*)it->data);
				gtk_widget_show(mainwin);
			}
		}
	}

	gtk_object_destroy(GTK_OBJECT(dialog));
}

static void
cria_main_window_selected_slide_changed(CriaMainWindow* self, GtkTreeSelection* selection) {
	CriaSlide	* slide;

	g_return_if_fail(CRIA_IS_MAIN_WINDOW(self));

	slide = cria_slide_list_get_selected(CRIA_SLIDE_LIST(self->priv->slide_list));
	cria_slide_editor_set_slide(CRIA_SLIDE_EDITOR(self->priv->slide_editor), slide);
}

static void
cb_presentation_title_changed(CriaMainWindow* self) {
	const gchar	* title;
	
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->presentation != NULL && CRIA_IS_PRESENTATION(self->priv->presentation));
	
	g_debug("MainWindow::cbPresentationTitleChanged(): we have a presentation");
	/* setting the window's title */
	title = cria_presentation_get_title(self->priv->presentation);
	
	if(!(title && strlen(title) && strcmp(title, _("Untitled")))) {
		g_debug("MainWindow::cbPresentationTitleChanged(): we don't have a cool title, using the filename");
		title = cria_presentation_get_filename(self->priv->presentation);
	}
	
	g_debug("MainWindow::cbPresentationTitleChanged(): setting title to 0x%x (%s)", (uintptr_t)title, title);
	gtk_window_set_title(GTK_WINDOW(self), title);
}

/**
 * cria_main_window_set_presentation:
 * @self: ...
 * @presentation: The presentation to connect to this window. Must not be NULL
 *
 * Specify the presentation to be displayed and edited within this window.
 */
void
cria_main_window_set_presentation(CriaMainWindow* self, CriaPresentation* presentation) {
	g_return_if_fail(CRIA_IS_MAIN_WINDOW(self));
	g_return_if_fail(CRIA_IS_PRESENTATION(presentation));
	g_assert(CRIA_IS_PRESENTATION(self->priv->presentation));
#warning "MainWindow::setPresentation(): connect to notify::uri signal"
	
	g_debug("MainWindow::setPresentation(): start");
	
	if(self->priv->presentation != NULL) {
		g_signal_handlers_disconnect_by_func(self->priv->presentation, cb_presentation_title_changed, self);

		g_object_unref(self->priv->presentation);
		self->priv->presentation = NULL;
	}

	g_debug("MainWindow::setPresentation(): cleared the old value, now setting the new one");

	self->priv->presentation = g_object_ref(presentation);

	cria_slide_list_set_presentation(CRIA_SLIDE_LIST(self->priv->slide_list),
					 cria_main_window_get_presentation(self));
	cria_slide_editor_set_slide(CRIA_SLIDE_EDITOR(self->priv->slide_editor),
				    cria_presentation_get_slide(cria_main_window_get_presentation(self), 0));

	/* set the window title */
	g_signal_connect_swapped(self->priv->presentation, "notify::title",
				 G_CALLBACK(cb_presentation_title_changed), self);
	cb_presentation_title_changed(self);

	g_object_notify (G_OBJECT (self), "presentation");

	g_debug("MainWindow::setPresentation(): end");
}

static void
cria_main_window_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaMainWindow	* self;

	g_debug("MainWindow::setProperty(): start");
	
	self = CRIA_MAIN_WINDOW(object);
	
	switch (prop_id) {
	case PROP_PRESENTATION:
		cria_main_window_set_presentation(self, g_value_get_object (value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object,
						  prop_id,
						  param_spec);
		break;
	}

	g_debug("MainWindow::setProperty(): end");
}

static void
cria_main_window_set_zoom_from_combo(CriaMainWindow* self, GtkEntry* entry) {
	gdouble       zoom;

	zoom = strtod(gtk_entry_get_text(entry), NULL) / 100;
	
	if(zoom >= 0.01) {
		cria_slide_editor_set_zoom(CRIA_SLIDE_EDITOR(self->priv->slide_editor), zoom);
	}
}

void
cria_main_window_show_about(CriaMainWindow* self) {
#warning "MainWindow::showAbout(): FIXME: this must go into application or an own class"
#warning "MainWindow::showAbout(): FIXME: create authors[] from $(top_srcdir)/AUTHORS"
	const gchar	* authors[] = {
		"Adrien Beaucreux (GConf and Preferences)",
		"Keith Sharp (Recent Files)",
		"Sven Herzberg (Project Leader)",
		NULL
	};
#warning "MainWindow::showAbout(): FIXME: add a criawips logo to the dialog"
	GdkPixbuf	* pixbuf = NULL;

	const gchar	* documenters[] = {NULL};

	GtkWidget	* dialog = gnome_about_new(PACKAGE,
						   VERSION,
						   _("Copyright (C) 2004 - The criawips development team"),
						   _("Criawips is a presentation application for the GNOME "
						     "desktop environment which should integrate smoothly into "
						     "the desktop"),
						   authors,
						   documenters,
						   /* TRANSLATORS: please translate 'translator-credits' with your name and the name of other
						    * translators that have been involved in this translation */
						   (strcmp("translator-credits", _("translator-credits")) == 0) ? NULL : _("translator-credits"),
						   pixbuf);

	gtk_window_set_transient_for(GTK_WINDOW(dialog),
				     GTK_WINDOW(self));

	gtk_widget_show(dialog);

	if(pixbuf) {
		g_object_unref(pixbuf);
	}
}

static void
cria_main_window_start_off (CriaMainWindow* self) {
	GtkWidget	* slide_show;

	g_debug("MainWindow::startOff(): start");
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(self->priv != NULL);
	g_assert(self->priv->presentation != NULL && CRIA_IS_PRESENTATION(self->priv->presentation));

	slide_show = cria_slide_show_new(self->priv->presentation);
	gtk_widget_show(slide_show);

	g_debug("MainWindow::startOff(): end");
}

static void
cria_main_window_set_presentation_from_text_uri(CriaMainWindow* self, gchar* text_uri) {
	GError		* error = NULL;
	CriaPresentation* presentation;

	presentation = cria_presentation_new_from_text_uri(text_uri, &error);

	if(error) {
		/* NOTE: keep the string in sync with the ones from application.c */
		char            * error_message = g_strdup_printf("<span weight=\"bold\">%s</span>\n\n%s %s:\n<span style=\"italic\">%s</span>.",
								  _("The Presentation could not be opened"),
								  _("For some reason the presentation you wanted to open could not be opened."),
								  _("The reason was:"),
								  error->message);

		cria_application_show_error_dialog(GTK_WINDOW(self),
						   _("Error opening file"),
						   error_message);
		g_debug("MainWindow::setPresentationFromTextUri(): uri was %s", text_uri);
				
		g_free(error_message);
		g_error_free(error);
	} else {
		cria_application_add_to_recent(text_uri);
	}
	
	cria_main_window_set_presentation(self, presentation);
}

/* GtkAction callbacks:
 *
 * data is always set to the main window, so just check the parameters and
 * call the correct method
 */

static void
cb_action_application_quit(GtkAction* action, CriaMainWindow* window) {
	criawips_quit();
}

static void
cb_action_about_info(GtkAction* action, CriaMainWindow* window) {
	g_assert(window != NULL && CRIA_IS_MAIN_WINDOW(window));

	cria_main_window_show_about(window);
}

static void
cb_action_file_new(GtkAction* action, CriaMainWindow* window) {
	g_assert(action != NULL && GTK_IS_ACTION(action));
	g_assert(window != NULL && CRIA_IS_MAIN_WINDOW(window));

	cria_main_window_set_presentation(window, cria_presentation_new_default());
}

static void
cb_action_file_open(GtkAction* action, CriaMainWindow* window) {
	g_assert(action != NULL && GTK_IS_ACTION(action));
	g_assert(window != NULL && CRIA_IS_MAIN_WINDOW(window));

	cria_main_window_open(window, action);
}

static void
cb_action_insert_slide(GtkAction* action, CriaMainWindow* window) {
	g_return_if_fail(CRIA_IS_MAIN_WINDOW(window));

	cria_main_window_insert_slide(window);
}

static void
cb_action_main_window_close(GtkAction* action, CriaMainWindow* window) {
	g_assert(action != NULL && GTK_IS_ACTION(action));
	g_assert(window != NULL && CRIA_IS_MAIN_WINDOW(window));

	cria_main_window_close(window);
}

static void
cb_action_presentation_properties(GtkAction* action, CriaMainWindow* window) {
	GtkWidget * properties = cria_presentation_properties_dialog_new(window->priv->presentation);
	gtk_widget_show(properties);
}

static void
cb_action_show_preferences(GtkAction* action, CriaMainWindow* window) {
	gtk_widget_show(cria_preferences_dialog_get_instance());
}

static void
cb_action_view_first(GtkAction* action, CriaMainWindow* self) {
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(self->priv != NULL);
	
	cria_slide_list_select_first(CRIA_SLIDE_LIST(self->priv->slide_list));
}
static void
cb_action_view_last(GtkAction* action, CriaMainWindow* self) {
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(self->priv != NULL);
	
	cria_slide_list_select_last(CRIA_SLIDE_LIST(self->priv->slide_list));
}
static void
cb_action_view_next(GtkAction* action, CriaMainWindow* self) {
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(self->priv != NULL);
	
	cria_slide_list_select_next(CRIA_SLIDE_LIST(self->priv->slide_list));
}
static void
cb_action_view_presentation(GtkAction* action, CriaMainWindow* self) {
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	
	cria_main_window_start_off(self);
}
static void
cb_action_view_previous(GtkAction* action, CriaMainWindow* self) {
	g_assert(self != NULL && CRIA_IS_MAIN_WINDOW(self));
	g_assert(self->priv != NULL);
	
	cria_slide_list_select_previous(CRIA_SLIDE_LIST(self->priv->slide_list));
}

/* Callback to load recent file */
static void
cb_recent_files_open(GtkWidget *widget, const EggRecentItem *item, CriaMainWindow* self) {
	gchar	* text_uri;
	
	text_uri = egg_recent_item_get_uri(item);
	cria_main_window_set_presentation_from_text_uri(self, text_uri);
}
