/* this file is part of criawips a gnome presentation application
 *
 * AUTHORS
 *       Sven Herzberg        <herzi@gnome-de.org>
 *
 * Copyright (C) 2004 Sven Herzberg
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "slide.h"

#include <inttypes.h>
#include <string.h>

#include <glib-object.h>
#include <glib.h>
#include <gdk/gdk.h>
#include <libgnome/gnome-i18n.h>

#include "debug.h"

struct _CriaSlidePrivate {
	GHashTable	* blocks;
	CriaSlide	* master_slide;
	CriaTheme	* theme;
	gchar		* comment;
	CriaImage	* background;
	CriaPresentation* presentation;
	GOColor		* background_color;
};

enum {
	PROP_0,
	PROP_BACKGROUND,
	PROP_COMMENT,
	PROP_TITLE,
	PROP_MASTER_SLIDE,
	PROP_THEME
};

enum {
	SIGNAL,
	N_SIGNALS
};

static	void	  cria_slide_get_property      (GObject		* object,
						guint		  prop_id,
						GValue		* value,
						GParamSpec	* param_spec);
static	void	  cria_slide_init	       (CriaSlide	* self);
static	void	  cria_slide_set_property      (GObject		* object,
						guint		  prop_id,
						const	GValue	* value,
						GParamSpec	* param_spec);
#if 0
static	guint	cria_slide_signals[N_SIGNALS] = { 0 };

static	void	cria_slide_signal	       (CriaSlide	* template,
						const	gchar	* string);
#endif

/**
 * cria_slide_add_block:
 * @self: ...
 * @block: ...
 *
 * Adds a #Block to this slide
 */
void
cria_slide_add_block(CriaSlide* self, CriaBlock* block) {
	g_return_if_fail(CRIA_IS_SLIDE(self));
	g_return_if_fail(CRIA_IS_BLOCK(block));
	g_return_if_fail(cria_block_get_name(block) != NULL);
        g_return_if_fail(g_hash_table_lookup(self->priv->blocks, (cria_block_get_name(block))) == NULL);
	/* FIXME: check that the block is defined in the layout
	 * this shouldn't be neccessary with the new rendering
	 * framework*/

	g_hash_table_insert(self->priv->blocks,
			    g_strdup(cria_block_get_name(block)),
			    block);
}

/*
 * cria_slide_class_init:
 * @cria_slide_class: ...
 *
 * Initializes @cria_slide_class.
 */
static void
cria_slide_class_init(CriaSlideClass* cria_slide_class) {
	GObjectClass	* g_object_class;

	g_object_class = G_OBJECT_CLASS(cria_slide_class);
#if 0
	/* setting up signal system */
	cria_slide_class->signal = cria_slide_signal;

	cria_slide_signals[SIGNAL] = g_signal_new (
			"signal",
			CRIA_TYPE_SLIDE,
			G_SIGNAL_RUN_LAST,
			G_STRUCT_OFFSET (
				CriaSlideClass,
				signal),
			NULL,
			NULL,
			g_cclosure_marshal_VOID__STRING,
			G_TYPE_NONE,
			0);
#endif
	/* setting up property system */
	g_object_class->set_property = cria_slide_set_property;
	g_object_class->get_property = cria_slide_get_property;

	g_object_class_install_property(g_object_class,
					PROP_BACKGROUND,
					g_param_spec_object("background",
							    "Slide Background",
							    "The Background image for this slide",
							    CRIA_TYPE_IMAGE,
							    G_PARAM_READWRITE));
	g_object_class_install_property(g_object_class,
					PROP_COMMENT,
					g_param_spec_string("comment",
							    "Comment",
							    "Some descriptive comment for the slide, typically used for "
							    "master slides",
							    "",
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_MASTER_SLIDE,
					g_param_spec_object("master-slide",
							    "Master Slide",
							    "A Slide that specifies drawing details",
							    CRIA_TYPE_SLIDE,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_TITLE,
					g_param_spec_string("title",
							    "Title",
							    "The title of that slide",
							    _("untitled"),
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
	g_object_class_install_property(g_object_class,
					PROP_THEME,
					g_param_spec_object("theme",
							    "Theme",
							    "The theme associated with this slide",
							    CRIA_TYPE_THEME,
							    G_PARAM_READWRITE | G_PARAM_CONSTRUCT));
}

/**
 * cria_slide_get_background:
 * @self: the slide to get the background from
 *
 * Get the background from a slide.
 *
 * Returns the background specified for this slide
 */
CriaImage *
cria_slide_get_background(CriaSlide* self) {
	g_assert(self != NULL && CRIA_IS_SLIDE(self));

	if(!self->priv->background && self->priv->master_slide && cria_slide_get_background(self->priv->master_slide)) {
		return cria_slide_get_background(self->priv->master_slide);
	}
		
	return self->priv->background;
}

/**
 * cria_slide_get_background_color:
 * @self: a #CriaSlide
 *
 * Get the background color of a slide. Free it after use.
 *
 * Returns the background color of a slide, NULL is not defined.
 */
GOColor*
cria_slide_get_background_color(CriaSlide* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);

	if(!self->priv->background_color && self->priv->master_slide && cria_slide_get_background_color(self->priv->master_slide)) {
		return cria_slide_get_background_color(self->priv->master_slide);
	}

	return self->priv->background_color;
}

/**
 * cria_slide_get_block:
 * @self: ...
 * @block: ...
 *
 * Get a block...
 *
 * Returns ...
 */
CriaBlock*
cria_slide_get_block(CriaSlide* self, const gchar* block) {
	CriaBlock * cblock;
	
	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);

	/* search for the block in here */
	cblock = cria_slide_get_block_no_recurse(self, block);

	if(cblock == NULL && CRIA_IS_SLIDE(self->priv->master_slide)) {
		cblock = cria_slide_get_block(self->priv->master_slide, block);
	}

	g_debug("Slide::getBlock(): self (0x%x) returns \"%s\" 0x%x now", (uintptr_t)self, block, (uintptr_t)cblock);
	return cblock;
}

CriaBlock*
cria_slide_get_block_no_recurse(CriaSlide* self, const gchar* block) {
	CriaBlock	* cblock = CRIA_BLOCK(g_hash_table_lookup(self->priv->blocks, block));
	g_debug("Slide::getBlockNoRecurse(): \"%s\" is 0x%x", block, (uintptr_t)cblock);
	return cblock;
}

struct vector {
	gchar	**vector;
	guint	  length;
};

static void
populate_array_with_keys(gpointer key, gpointer value, gpointer user_data) {
	struct vector* v = (struct vector*)user_data;
	gchar	* skey  = (gchar*)key;
	guint	  i;

	if(!v) {
		return;
	}

	for (i = 0; v->vector[i]; i++) {
		if (!strcmp(v->vector[i], skey)) {
			/* we already contain that key */
			return;
		}
		/* stepping to the first NULL */
	}

	if(i == (v->length - 1)) {
		int j;
		
		v->length += 10;
		v->vector = g_renew(gchar*,v->vector,v->length);

		for(j = i; j < v->length; j++) {
			v->vector[j] = NULL;
		}
	}
	
	v->vector[i] = g_strdup(skey);
}

static void
cria_slide_get_block_namesv(CriaSlide* self, struct vector* v) {
	if(CRIA_IS_SLIDE(self->priv->master_slide)) {
		cria_slide_get_block_namesv(self->priv->master_slide, v);
	}

	g_hash_table_foreach(self->priv->blocks,
			     populate_array_with_keys,
			     v);
}

/**
 * cria_slide_get_block_names:
 * @self: the slide to get the block names from
 *
 * ...
 *
 * Returns ...
 */
gchar**
cria_slide_get_block_names(CriaSlide* self) {
	gchar	**names = NULL;
	struct vector* v = g_new0(struct vector, 1);
	
	/* we will need at least that much space */
	v->length = g_hash_table_size(self->priv->blocks) + 1;
	v->vector = g_new0(gchar*,v->length);

	cria_slide_get_block_namesv(self, v);

	names = v->vector;
	g_free(v);

	return names;
}

/**
 * cria_slide_get_display_size:
 * @self: ...
 * @screen: ...
 *
 * ...
 *
 * Returns a #GoPoint defining the size of the screen in pixels. Don't forget
 * to g_free it
 */
GoPoint*
cria_slide_get_display_size(CriaSlide* self, GdkScreen* screen) {
	GoPoint* retval;

	/* Require this one once we store the aspect ratio in the presentation
	 * and we have a cool way to access it from a slide */
/*	g_assert(self != NULL && CRIA_IS_SLIDE(self));
*/	g_assert(screen != NULL && GDK_IS_SCREEN(screen));

	retval = g_new0(GoPoint,1);
	retval->x = gdk_screen_get_width(screen);
	retval->y = gdk_screen_get_height(screen);

	g_debug("Slide::getDisplaySize(): got resolution %llix%lli (%f)", retval->x, retval->y, 1.0 * retval->x / retval->y);

#warning "Slide::getDisplaySize(): FIXME: get the ratio from the presentation"
#define RATIO_X 4
#define RATIO_Y 3
	if ((1.0 * retval->x / retval->y) >= (1.0 * RATIO_X / RATIO_Y)) {
		/* screen is too wide */
		retval->y -= (retval->y % RATIO_Y);
		retval->x  = retval->y / RATIO_Y * RATIO_X;
		g_debug("Slide::getDisplaySize(): The screen is too wide, display resolution is %llix%lli", retval->x, retval->y);
	} else {
		/* screen is too high */
		retval->x -= (retval->x % RATIO_X);
		retval->y  = retval->x / RATIO_X * RATIO_Y;
		g_debug("Slide::getDisplaySize(): the screen is too high, display resolution is %llix%lli", retval->x, retval->y);
	}
	
	return retval;
#undef RATIO_X
#undef RATIO_Y
}

/**
 * cria_slide_get_master_slide:
 * @self: The slide to the get master slide from
 *
 * The master slide defines several rendering details that the slide doesn't
 * need to contain (because it's shared across all similar looking slides).
 * This method returns the master slide of a slide or NULL if the given slide
 * is a master slide.
 *
 * Returns the master slide of a slide
 */
#warning "Slide::getMasterSlide(): FIXME: Think about this: can we have a tree of master slides so NULL may even be returned when the given slide is a master slide?"
CriaSlide*
cria_slide_get_master_slide(CriaSlide* self) {
	g_assert(self != NULL && CRIA_IS_SLIDE(self));
	
	return self->priv->master_slide;
}

static void
cria_slide_get_property(GObject* object, guint prop_id, GValue* value, GParamSpec* param_spec) {
	CriaSlide	* self;

	self = CRIA_SLIDE(object);

	switch(prop_id) {
	case PROP_BACKGROUND:
		g_value_set_object(value, self->priv->background);
		break;
	case PROP_COMMENT:
		g_value_set_string(value, self->priv->comment);
		break;
	case PROP_MASTER_SLIDE:
		g_value_set_object(value, self->priv->master_slide);
		break;
	case PROP_TITLE:
		g_value_set_string(value, self->title);
		break;
	case PROP_THEME:
		g_value_set_object(value, self->priv->theme);
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

/**
 * cria_slide_get_size:
 * @self: a #CriaSlide
 *
 * ...
 *
 * Returns a #GoPoint defining the size of the slide in master coordinates
 * (576dpi). Don't forget to g_free it.
 */
GoPoint*
cria_slide_get_size(CriaSlide* self) {
#warning "Slide::getSize(): FIXME return value from presentation"
	GoPoint* size = g_new0(GoPoint,1);
	size->x = 5760;
	size->y = 4320;
	return size;
}

CriaTheme*
cria_slide_get_theme(CriaSlide* self) {
	g_return_val_if_fail(CRIA_IS_SLIDE(self), NULL);
	
	return self->priv->theme;
}

const char*
cria_slide_get_title(CriaSlide* self) {
	g_return_val_if_fail (CRIA_IS_SLIDE(self), NULL);
	
	return self->title;
}

GType
cria_slide_get_type(void) {
	static GType	type = 0;

	if (!type) {
		const GTypeInfo info = {
			sizeof (CriaSlideClass),
			NULL,	/* base initializer */
			NULL,	/* base finalizer */
			(GClassInitFunc)cria_slide_class_init,
			NULL,	/* class finalizer */
			NULL,	/* class data */
			sizeof (CriaSlide),
			0,
			(GInstanceInitFunc)cria_slide_init,
			0
		};

		type = g_type_register_static(G_TYPE_OBJECT,
					      "CriaSlide",
					      &info,
					      0);
	}

	return type;
}

static void
cria_slide_init(CriaSlide* self) {
	g_debug("Slide::init(): initializing Slide at 0x%x", (uintptr_t)self);
	self->priv = g_new0(CriaSlidePrivate, 1);
	
	self->priv->blocks = g_hash_table_new_full(g_str_hash,
						   g_str_equal,
						   g_free,
						   g_object_unref);
}

static void
cria_void(void) {
	// dummy for g_object_weak_ref
}

/**
 * cria_slide_new:
 * @presentation: a #CriaPresentation
 *
 * Creates a new empty slide in @presentation at the end.
 *
 * Returns the new #CriaSlide.
 */
CriaSlide*
cria_slide_new(CriaPresentation* presentation) {
	CriaSlide* self = g_object_new(CRIA_TYPE_SLIDE, NULL);

	if(presentation) {
		g_object_weak_ref(G_OBJECT(presentation), (GWeakNotify)cria_void, &(self->priv->presentation));
		cria_presentation_append_slide(presentation, self);
	}
	
	return self;
}

/**
 * cria_slide_new_pos:
 * @presentation: a #CriaPresentation
 * @pos: a position
 *
 * Creates a new empty slide in @presentation at @pos.
 *
 * Returns the new #CriaSlide.
 */
CriaSlide*
cria_slide_new_pos(CriaPresentation* presentation, gint pos) {
	CriaSlide* self = g_object_new(CRIA_TYPE_SLIDE, NULL);

	if(presentation) {
		g_object_weak_ref(G_OBJECT(presentation), (GWeakNotify)cria_void, &(self->priv->presentation));
		cria_presentation_insert_slide(presentation, self, pos);
	}

	return self;
}

/**
 * cria_slide_set_background:
 * @self: the slide to set the background for
 * @background: the background to be set
 *
 * Specify a background for a slide
 */
void
cria_slide_set_background(CriaSlide* self, CriaImage* background) {
	g_assert(self != NULL && CRIA_IS_SLIDE(self));
	g_assert(background != NULL && CRIA_IS_IMAGE(background));

	if(self->priv->background == background)

	if(self->priv->background) {
		g_object_unref(self->priv->background);
	}

	self->priv->background = g_object_ref(background);

	g_object_notify(G_OBJECT(self), "background");
}

/**
 * cria_slide_set_background_color:
 * @self: a #CriaSlide
 * @background: the color for the slide
 *
 * Set the background color of a slide. Pass NULL to unset the background.
 */
void
cria_slide_set_background_color(CriaSlide* self, const GOColor* color) {
	g_return_if_fail(CRIA_IS_SLIDE(self));

	if(self->priv->background_color) {
		g_free(self->priv->background_color);
	}

	if(color) {
		self->priv->background_color= g_new0(GOColor,1);
		*(self->priv->background_color) = *color;
	} else {
		self->priv->background_color = NULL;
	}

	// g_object_notify(G_OBJECT(self), "background-color");
}

/**
 * cria_slide_set_comment:
 * @self: The slide to set a comment for
 * @comment: The comment, a plain NUL terminated UTF8 string
 *
 * Set a comment for this slide. This is usually used for master slides
 * to set some descriptive text for a layout.
 */
void
cria_slide_set_comment(CriaSlide* self, const gchar* comment) {
	g_assert(self != NULL && CRIA_IS_SLIDE(self));
	g_assert(comment != NULL);

	g_debug("Slide::setComment(): start");

	if(comment != NULL && self->priv->comment != NULL && !strcmp(comment, self->priv->comment)) {
		return;
	}

	if(self->priv->comment != NULL) {
		g_free(self->priv->comment);
	}

	self->priv->comment = g_strdup(comment);

	g_object_notify(G_OBJECT(self), "comment");

	g_debug("Slide::setComment(): end");
}

/**
 * cria_slide_set_master_slide:
 * @self: the slide to set the master slide for
 * @master_slide: the slide to be set as the master slide
 *
 * Defines a slide containing common rendering details shared across several
 * slides.
 */
void
cria_slide_set_master_slide(CriaSlide* self, CriaSlide* master_slide) {
	g_assert(self != NULL && CRIA_IS_SLIDE(self));
	g_assert(master_slide == NULL || CRIA_IS_SLIDE(master_slide));

	if(self->priv->master_slide == master_slide) {
		return;
	}

	if(self->priv->master_slide != NULL) {
		g_object_unref(self->priv->master_slide);
	}

	if(master_slide) {
		self->priv->master_slide = g_object_ref(master_slide);
	} else {
		self->priv->master_slide = master_slide;
	}

	g_object_notify(G_OBJECT(self), "master_slide");
}

/*
 * cria_slide_set_property:
 * @object: ...
 * @prop_id: ...
 * @value: ...
 * @param_spec: ...
 *
 * ...
 */
static void
cria_slide_set_property(GObject* object, guint prop_id, const GValue* value, GParamSpec* param_spec) {
	CriaSlide	* self = CRIA_SLIDE(object);
	
	switch(prop_id) {
	case PROP_BACKGROUND:
		cria_slide_set_background(self, g_value_get_object(value));
		break;
	case PROP_COMMENT:
		cria_slide_set_comment(self, g_value_get_string(value));
		break;
	case PROP_MASTER_SLIDE:
		cria_slide_set_master_slide(self, g_value_get_object(value));
		break;
	case PROP_TITLE:
		cria_slide_set_title(self, g_value_get_string(value));
		break;
	case PROP_THEME:
		cria_slide_set_theme(self, g_value_get_object(value));
		break;
	default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, param_spec);
		break;
	}
}

/**
 * cria_slide_set_theme:
 * @self: ...
 * @theme: ...
 *
 * ...
 */
void
cria_slide_set_theme(CriaSlide* self, CriaTheme* theme) {
	g_return_if_fail(CRIA_IS_SLIDE(self));

	if (self->priv->theme != NULL) {
		g_object_unref(self->priv->theme);
	}

	if (theme) {
		self->priv->theme = g_object_ref(theme);
	} else {
		self->priv->theme = theme;
	}

	g_object_notify(G_OBJECT(self), "theme");
}

/**
 * cria_slide_set_title:
 * @self: ...
 * @title: ...
 *
 * ...
 */
void
cria_slide_set_title(CriaSlide* self, const gchar* title) {
	g_return_if_fail (CRIA_IS_SLIDE(self));

	if (self->title != NULL) {
		g_free (self->title);
	}

	g_debug("cria_slide_set_title(%i): \"%s\"", __LINE__, title);
	self->title = g_strdup(title);

	g_object_notify(G_OBJECT(self), "title");
}

