#!/usr/bin/env python
# -*- coding: utf-8 -*-

#     Copyright 2010 Sergiy S. Kolesnikov
#
#     This file is part of csv2itdb.
#
#     csv2itdb is free software: you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation, either version 3 of the License, or
#     (at your option) any later version.
#
#     csv2itdb is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.
#
#     You should have received a copy of the GNU General Public License
#     along with csv2itdb.  If not, see <http://www.gnu.org/licenses/>.

# Project homepage: https://savannah.nongnu.org/projects/csv2itdb/

import os, sys, optparse, codecs, csv, time

# Default program settings.
settings = {
    'inputFilePath':None,
    'outputFilePath':None,
    'itestVersion':'1.42',
    'dbVersion':'1.4',
    'inputEncoding':'utf-8',
    'outputEncoding':'utf-8',
    'correctAnswerNum':1
    }

programName = 'csv2itdb'
programVersion = '1.01'
programUsage = 'python csv2itdb.py [options] <file.csv>'
programDescription=''' %(programName)s (v%(programVersion)s) is a CSV to iTestDB
converter. It converts a test database in a specific CSV format to an iTest
database. Generated data base format version is %(dbVersion)s. For more
information on the accepted CSV format and supported character encodings see the
attached documentation.
''' % {'programName':programName,
       'programVersion':programVersion, 
       'dbVersion':settings['dbVersion']}

# First portion of the generated iTest database.
itdbHeader = u'''[ITEST_VERSION]
%(itestVersion)s
[ITEST_DB_VERSION]
%(dbVersion)s
[DB_NAME]
%(dbName)s
[DB_DATE]
%(date)s
[DB_DATE_ULSD]
true
[DB_COMMENTS]
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0//EN" "http://www.w3.org/TR/REC-html40/strict.dtd"><html><head><meta name="qrichtext" content="1" /><style type="text/css">p, li { white-space: pre-wrap; }</style></head><body style=" font-family:'Sans Serif'; font-size:11pt; font-weight:400; font-style:normal;"><p style=" margin-top:0px; margin-bottom:0px; margin-left:0px; margin-right:0px; -qt-block-indent:0; text-indent:0px;">%(comments)s</p></body></html>
[DB_QNUM]
%(questionsNumber)i
[DB_SNUM]
0
[DB_CNUM]
0
[DB_FLAGS]
--------------------
[DB_F0]

[DB_F1]

[DB_F2]

[DB_F3]

[DB_F4]

[DB_F5]

[DB_F6]

[DB_F7]

[DB_F8]

[DB_F9]

[DB_F10]

[DB_F11]

[DB_F12]

[DB_F13]

[DB_F14]

[DB_F15]

[DB_F16]

[DB_F17]

[DB_F18]

[DB_F19]

[DB_FLAGS_END]'''

class Answers(object):
    ''' Encapsulates answers for a question.  Calling unicode() on an instance
    of the class returns a preformatted string suitable for inserting into a
    [Q_ANS] database entry.
    
    Constructor parameter 'answers' - a list of strings.  One string represents
    one answer.'''

    def __init__(self, answers):
        self.answers = answers

    def getAnswersNum(self):
        return len(self.answers)

    def __str__(self):
        return unicode(self).encode(settings['outputEncoding'])
    
    def __unicode__(self):
        answersString = u''
        for a in self.answers:
            answersString += a + '\n'
        return answersString[0:-1] # trim the last newline

class Question(object):
    '''Encapsulates a question with answers and all the metadata.  Calling
    unicode() on an instance of the class returns a preformatted string suitable
    for inserting into iTest database.  Correct answer number is determined by
    csv2itdb.settings['correctAnswerNum'].'''

    def __init__(self, name=u'', question=u'', explanation=u'', answers=Answers([u''])):
        if name == u'':
            self.name = question
        else:
            self.name = name
        self.question = question
        self.explanation = explanation
        self.answers = answers

    def __str__(self):
        return unicode(self).encode(settings['outputEncoding'])

    def __unicode__(self):
        return u'''[Q_NAME]
%(name)s
[Q_FLAG]
-1
[Q_GRP]

[Q_DIF]
0
[Q_TEXT]
<!DOCTYPE HTML PUBLIC "-//W3C//DTD HTML 4.0//EN" "http://www.w3.org/TR/REC-html40/strict.dtd"><html><head><meta name="qrichtext" content="1" /><style type="text/css">p, li { white-space: pre-wrap; }</style></head><body style=" font-family:'Sans Serif'; font-size:11pt; font-weight:400; font-style:normal;"><p style=" margin-top:0px; margin-bottom:0px; margin-left:0px; margin-right:0px; -qt-block-indent:0; text-indent:0px;">%(question)s</p></body></html>
[Q_ANS]
0
%(correctAnswerNum)i
%(answersNum)i
%(answers)s
[Q_EXPL]
%(explanation)s
[Q_ICCNT]
0
0
[Q_HID]
false
[Q_SVG]
0''' % {'name':self.name,
            'question':self.question,
            'correctAnswerNum':settings['correctAnswerNum'],
            'answersNum':self.answers.getAnswersNum(),
            'answers':unicode(self.answers),
            'explanation':self.explanation}

class CSVUTF8Recoder(object):
    """
    Iterator that reads an encoded stream and reencodes the input to UTF-8.
    """

    def __init__(self, f, encoding):
        self.reader = codecs.getreader(encoding)(f)

    def __iter__(self):
        return self

    def next(self):
        return self.reader.next().encode("utf-8")

class CSVUnicodeReader(object):
    """
    A CSV reader which will iterate over lines in the CSV file "f",
    which is encoded in the given encoding.
    """

    def __init__(self, f, encoding="utf-8", dialect=csv.excel, **kwds):
        f = CSVUTF8Recoder(f, encoding)
        self.reader = csv.reader(f, dialect=dialect, **kwds)

    def next(self):
        row = self.reader.next()
        return [unicode(s, "utf-8") for s in row]

    def __iter__(self):
        return self

def _printErrorMsg(msg):
    '''Prints a standard preformatted error message.'''

    print(programName + ': error: ' + msg)

def lineNumbers(filePath):
    '''Counts line numbers in the text file filePath.'''

    file = open(filePath, 'rU')
    ln = 0
    for line in file:
        ln += 1
    file.close()
    return ln

def main(csvFilePath, outFilePath=None):
    '''Reads csvFilePath and converts the data into iTest database format.  The
    output is written to outFilePath [default: stdout].'''

    csvFile = open(csvFilePath, 'rb')
    csvReader = CSVUnicodeReader(csvFile, settings['inputEncoding'])
    header = itdbHeader % {'itestVersion':settings['itestVersion'],
                           'dbVersion':settings['dbVersion'],
                           'dbName':unicode(os.path.basename(csvFilePath)
                                            ,sys.getfilesystemencoding()),
                           'date':time.strftime('%Y.%m.%d-%H:%M'
                                                 ,time.localtime()),
                           'comments':u'Generated with csv2itdb.py',
                           'questionsNumber':lineNumbers(csvFilePath)}
    if outFilePath == None:
        outFile = sys.stdout
    else:
        outFile = open(outFilePath, 'wb')
    outFile.write(header.encode(settings['outputEncoding']))
    outFile.write(u'\n'.encode(settings['outputEncoding']))
    for row in csvReader:
        name = row[0]
        question = row[1]
        explanation = row[2]
        answers = Answers(row[3:])
        outFile.write(unicode(Question(name, question, explanation, answers))
                         .encode(settings['outputEncoding']))
        outFile.write(u'\n'.encode(settings['outputEncoding']))
    outFile.close()
    csvFile.close()
            
#Parameter parsing and sanity checks
if __name__ == '__main__':
    oparser = optparse.OptionParser(usage=programUsage, 
                                    description=programDescription,
                                    version=programVersion, prog=programName)
    oparser.add_option( '--out', dest='outputFilePath', metavar='OUT',
                        help='output to OUT file [default: stdout]',
                        default=None)
    oparser.add_option( '--encoding', dest='inputEncoding', metavar='ENC',
                        help='use ENC encoding for reading the input file [default: %default]',
                        default='UTF-8')
    try:
        options, operands = oparser.parse_args()

        if len(operands) == 0: 
            raise optparse.OptParseError('a CSV file must be specified')
        if len(operands) > 1: 
            raise optparse.OptParseError('too many arguments')
        if not os.path.isfile(operands[0]):
            raise optparse.OptParseError('input file %s does not exist' 
                                       % (operands[0],))
        if options.outputFilePath != None and os.path.isfile(options.outputFilePath):
            raise optparse.OptParseError('output file %s exists already'
                                           % (options.outputFilePath,))
        settings['inputFilePath'] = operands[0]
        settings['outputFilePath'] = options.outputFilePath
        settings['inputEncoding'] = options.inputEncoding
    except optparse.OptParseError, e:
        oparser.print_usage()
        _printErrorMsg(e.msg)
        sys.exit(1)
    try:
        main(settings['inputFilePath'], settings['outputFilePath'])
    except UnicodeDecodeError:
        e,p,t = sys.exc_info()
        msg = str(p) + "\n\nCheck if the encoding of the CSV file is %s, or use corresponding option to set the encoding of the CSV file." % settings['inputEncoding']
        _printErrorMsg(msg)
