# output.py
# Copyright 2002 Alex Mercader <alex.mercader@iinet.net.au>
#
# This file is part of Curphoo.
#
# Curphoo is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Curphoo is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Curphoo; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

import curses
import time
import sys
import os
import re

def wordwrap(win, text, indent=2):
	x = win.getyx()[1]
	mx = win.getmaxyx()[1]
	text = text.strip()
	if text.find('\n') != -1:
		if text.count('\n') == 1 and len(text) < (mx-x):
			text = text.replace('\n', ' ')
		wrap = text
	else:
		wrap = ''
		r = curses.COLS - x
		while len(text) > r:
			i = text[:r].rfind(' ')
			if i == -1:
				if text[:r-1].strip():
					wrap = "%s%s\n%s" % (wrap,text[:r-1],' '*indent)
				text = text[r-1:].lstrip()
			else:
				if text[:i].strip():
					wrap = "%s%s\n%s" % (wrap,text[:i],' '*indent)
				text = text[i:].lstrip()
			r = curses.COLS - indent
		else:
			wrap = "%s%s" % (wrap, text.strip())
	return wrap.rstrip()

def wordwrap_log(text, indent=2):
	x = 0
	mx = 80
	text = text.strip()
	if text.find('\n') != -1:
		if text.count('\n') == 1 and len(text) < (mx-x):
			text = text.replace('\n', ' ')
		wrap = text
	else:
		wrap = ''
		r = mx - x
		while len(text) > r:
			i = text[:r].rfind(' ')
			if i == -1:
				if text[:r-1].strip():
					wrap = "%s%s\n%s" % (wrap,text[:r-1],' '*indent)
				text = text[r-1:].lstrip()
			else:
				if text[:i].strip():
					wrap = "%s%s\n%s" % (wrap,text[:i],' '*indent)
				text = text[i:].lstrip()
			r = mx - indent
		else:
			wrap = "%s%s" % (wrap, text.strip())
	return wrap.rstrip()

def cmpfunc(x, y):
	res = 0
	a = x.lower()
	b = y.lower()
	if a < b: res = -1
	elif a > b: res = 1
	return res

def list_users(win, l, header=''):
	win.addstr('*** ', curses.color_pair(1))
	win.addstr('%s (%s): ' % (header, len(l)))
	l.sort(cmpfunc)
	c = []
	for e in l:
		c.append('[%s]' % e)
	take_nl(win)
	win.addstr(wordwrap(win, ' '.join(c), 0))
	take_nl(win)

def log_notice(win, yes=1):
	if yes:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('Logging started.')
	else:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('Logging stopped.')
	take_nl(win)

def pm_notice(win, user, yes=1):
	if yes:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('%s will be allowed to pm.' % user)
	else:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('%s will not be allowed to pm.' % user)
	take_nl(win)

def histr_notice(win, hitext, yes=1):
	if yes:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('added %s to list of strings to highlight.' % hitext)
	else:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('comments containing the string %s will be highlighted.' % hitext)
		win.addstr('removed %s from list of strings to highlight.' % hitext)
	take_nl(win)

def hi_notice(win, user, yes=1):
	if yes:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('added %s to list of users to highlight.' % user)
	else:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('removed %s from list of users to highlight.' % user)
	take_nl(win)

def ignore_notice(win, user, yes=1):
	if yes:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('%s will be ignored.' % user)
	else:
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('%s will not be ignored.' % user)
	take_nl(win)

def shcmd(win, user, s):
	win.addstr('<%s> ' % user, curses.A_DIM)
	win.addstr(wordwrap(win, s), curses.A_DIM)
	take_nl(win)

def help(win, s):
	win.addstr('*** HELP: ', curses.color_pair(1))
	win.addstr(wordwrap(win, s), curses.A_DIM)
	take_nl(win)

def blah(win, s):
	win.addstr('*** BLAH: ', curses.color_pair(1))
	win.addstr(wordwrap(win, s), curses.A_DIM)
	take_nl(win)

def pm_user(win, me, user, s):
	win.addstr('<%s '  % me, curses.color_pair(4))
	win.addstr('(%s)'  % user, curses.A_DIM)
	win.addstr('> ', curses.color_pair(4))
	win.addstr(wordwrap(win, '%s' % s))
	take_nl(win)

def take_nl(win):
	if win.getyx()[1]: win.addstr('\n')

def status(win, sess):
	if sess.afk:
		s = '[%s] %s@%s:%s' % (sess.time, sess.user, sess.server[:3], sess.room)
		s = '%s%s' % (s, ' '*(curses.COLS - (len(' -AWAY-') + len(s))))
		s = '%s%s' % (s, '-AWAY-')
	else:
		s = '[%s] %s@%s:%s C:%s I:%s B:%s' % (
			sess.time,
			sess.user,
			sess.server[:3],
			sess.room,
			len(sess.users),
			len(sess.ignores),
			len(sess.buddies))
	s = s[:curses.COLS-1]
	win.erase()
	win.addstr(s)
	win.noutrefresh()
	os.environ['TERM'] in ("rxvt" , "xterm", "xterm-color") and sys.stderr.write("\033]0;%s\a" % s)

def message(win, m, me, sess):
	t = m['type']
	if t == 'comment':
		if m['user'] == me:
			win.addstr('<%s> ' % m['user'], curses.color_pair(4))
			win.addstr(wordwrap(win, m['text']))
		else:
			highlight_on = 0
			if m['user'].lower() in sess.high or \
					[e for e in sess.highstr if re.search(r"%s" % e, m['text'], re.I)]:
				win.addstr('<%s> ' % m['user'], curses.color_pair(1)|curses.A_BOLD)
				win.addstr(wordwrap(win, m['text']), curses.color_pair(1)|curses.A_BOLD)
			else:
				win.addstr('<%s> ' % m['user'], curses.color_pair(1))
				win.addstr(wordwrap(win, m['text']))
		take_nl(win)
	elif t == 'emote':
		win.addstr('* ', curses.color_pair(1))
		win.addstr(wordwrap(win, '%s %s' %(m['user'], m['text'])))
		take_nl(win)
	elif t == 'away':
		win.addstr(wordwrap(win, '* %s %s' % (m['user'], m['text'])), curses.A_DIM)
		take_nl(win)
	elif t == 'enter':
		win.addstr('::: %s has joined the room :::' % m['user'], curses.color_pair(2))
		take_nl(win)
		r = win.getyx()[0]
		win.move(r-1, 0)
	elif t == 'exit':
		win.addstr('::: %s has left the room :::' % m['user'], curses.color_pair(2))
		take_nl(win)
		r = win.getyx()[0]
		win.move(r-1, 0)
	elif t == 'pm':
		win.addstr('<%s ' % m['user'], curses.color_pair(1))
		win.addstr('(%s)' % sess.user, curses.A_DIM)
		win.addstr('> ', curses.color_pair(1))
		win.addstr(wordwrap(win, m['text']))
		take_nl(win)
	elif t == 'logout':
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('Logout successful.')
		take_nl(win)
	elif t == 'login':
		win.addstr('*** ', curses.color_pair(1))
		win.addstr('Login successful.')
		take_nl(win)
	elif t == 'join':
		win.addstr('*** ', curses.color_pair(1))
		text = wordwrap(win, ' %s (%s)' % (m['room'], m['topic']))
		win.addstr(text)
		take_nl(win)
		list_users(win, m['users'], 'Users')

def log_time(f):
	if f:
		f.write(time.strftime('---[ %x %X ]---\n'))

def log(m, f):
	if (not f) or (m['type'] == 'crap'): return
	t = m['type']
	try:
		if t == 'comment':
			s = '<%s> %s' % (m['user'], m['text'])
			s = '%s\n' % wordwrap_log(s)
			f.write(s)
		elif t == 'emote':
			s = '* %s %s' % (m['user'], m['text'])
			s = '%s\n' % wordwrap_log(s)
			f.write(s)
		elif t == 'away':
			s = '* %s %s' % (m['user'], m['text'])
			s = '%s\n' % wordwrap_log(s)
			f.write(s)
		elif t == 'enter':
			f.write('*** %s has joined the room.\n' % m['user'])
		elif t == 'exit':
			f.write('*** %s has left the room.\n' % m['user'])
		elif t == 'pm':
			s = '*%s* %s' % (m['user'], m['text'])
			s = '%s\n' % wordwrap_log(s)
			f.write(s)
		elif t == 'logout':
			f.write('*** Logout successful.\n')
			f.write(time.strftime('---End Capture %x %X---\n'))
		elif t == 'login':
			f.write(time.strftime('---Start Capture %x %X---\n'))
			f.write('*** Login successful.\n')
		elif t == 'join':
			s = '%s\n' % wordwrap_log('*** %s (%s)' % (m['room'], m['topic']))
			f.write(s)
			s = '%s\n' % wordwrap_log('*** Users: %s' % ', '.join(m['users']))
			f.write(s)
	except IOError:
		pass

