/***************************************************************************
                          spielfeld.cpp  -  description
                             -------------------
    begin                : Sat Oct 2 1999
    copyright            : (C) 1999 by immi
    email                : cuyo@pcpool.mathematik.uni-freiburg.de
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <qpainter.h>
#include <qarray.h>
#include <qpoint.h>
#include <cstdio>

#include "nachbariterator.h"
#include "cuyo.h"
#include "spielfeld.h"
#include "knoten.h"
#include "aufnahme.h"
#include "sound.h"

/* Wird nur included fr die Konstanten taste_* */
#include "prefs.h"


/***** Konstanten, die was einstellen *****/


/* Wie weit unten mssen Graue sein, damit ein neues Fall kommt? */
#define neues_fall_platz 5


/* Wenn grade neue Graue auf einen maximal hohen Turm gefallen sind,
   soll man auf jeden Fall noch ein bisschen Zeit haben, ihn wieder
   zu verkleinern. Deshalb tauchen Graue ein Stck weiter unten als
   der Hetzrand auf:
 */
/* Wo relativ zum Hetzrand... */
/* ... tauchen neue Steine auf? */
#define hetzrand_dy_auftauch 8
/* ... ist die maximal erlaubte Turmhhe? */
#define hetzrand_dy_erlaubt 0

/* Wie oft wird versucht, die "+"-Blops am Spielanfang unverbunden
   zu machen? */
#define startlevel_rand_durchgaenge 10


/* Wie lange wird ein Message angezeigt? */
#define message_gesamtzeit 45
#define message_blink 4

/***** Rckgabe-Konstanten u. . *****/

	
/* Werte fr mModus */

/* In diesem Modus ist kein Spielfeld zu sehen (Beim Warten auf
   Leertaste am Anfang vom Spiel luft das Spiel eigentlich schon,
   aber spielSchritt() wird noch nicht aufgerufen.) */
#define modus_keinspiel 0
/* Bedeutet auch, dass kein Spiel luft. Einziger Unterschied: Das
   Spielfeld ist trotzdem zu sehen, damit man sehen kann, in welcher
   Situation der Fehler passiert ist. */
#define modus_fehler 1

#define modus_testflopp 30  // nur ein bergangsmodus: Es muss noch getestet
                            // werden, ob was platzt
#define modus_flopp 31
#define modus_rutschnach 40
#define modus_neue_graue 41 // Da kommen grad neue Graue vom Himmel. So lange
                            // das noch passiert, kann kein neues Fall kommen.
                            // Irgend wann sind die Grauen weit genug unten;
                            // Dann wird ein neues Fall losgeschickt und der
                            // Modus gewechselt. Und zwar nach rutschnach,
                            // falls noch Graue unterwegs sind und nach
                            // testflopp, falls sie schon gelandet sind (z. B.
                            // auf hohen Trmen.)
               
#define modus_fallplatz 90 // da war grad was am runterfallen, whrend das
                     // Spiel beendet wurde; da lassen wir es halt platzen

// das Spiel ist zu Ende (alle Schlussanimationen sind fertig abgelaufen,
// wir warten nur noch darauf, dass stopLevel() aufgerufen wird):
#define modus_warte_auf_stop 91

/* Modus-Diagramm:

         ,-- warte_auf_stop <----,
         |        ^              |
         |        |          fallplatz <-,
         |        |                      |
         |        +-----------------> testflopp --> flopp
         |        |                     |    ^        |
         |        |                     |    |        |
         |        |                     |    |        |
         v        |                     |    |        |
   keinspiel --> neue_graue <-----------    |        |
                       |                     |        |
                       `-------------> rutschnach <---+



   - testflopp wird schon am Anfang von Spielschritt ausgewertet. (d. h. der
     bergang von da woandershin dauert keinen Zeitschritt)
   - Nur in keinspiel ist kein Spielfeld sichtbar
	 - Mit Escape kann von jedem Modus zu keinspiel gewechselt werden
*/

				
/* Werte fr mRueberReihenModus */
#define rrmodus_nix 0
#define rrmodus_gib_rueber 10
#define rrmodus_gib_runter 11
#define rrmodus_bekomm_hoch 20
#define rrmodus_bekomm_rueber 21




/* Rckgabewerte fr rutschNach() */
#define rutschnach_nix 0
#define rutschnach_wenig 1 // Wenn nur im unteren Bereich noch was nachrutscht,
// so dass das neue Fall kommen kann
#define rutschnach_viel 3
// Brauche: rutschnach_viel | rutschnach_wenig = rutschnach_viel


/* Rckgabewerte von sigWillReihe bzw. gebReihe */
#define bewege_reihe_nein 0
/* "Moment, ich bin noch mit der vorigen Reihe beschftigt. Vielleicht
    gleich." */
#define bewege_reihe_moment 1
#define bewege_reihe_ja 2




Spielfeld::Spielfeld(bool re, QWidget *parent, const char *name )
  : QWidget(parent,name), mBild(grx * gric, gry * gric),
    mDaten(re), mFall(this, re)
{
  resize(grx * gric, gry * gric);
  setPalette( QPalette( white ) );

  /* fr das Reihenrberschieben... */
  mRechterSpieler = re;

  mModus = modus_keinspiel;


}

Spielfeld::~Spielfeld(){
}


/** Sollte einmal direkt nach dem Start von cuyo aufgerufen werden;
    initialisiert den aktuellen Level und benutzt das Startbild als
    Titelbild. */
void Spielfeld::erzeugTitelbild() {
  startLevel();

  try {
    /* Init-Events verschicken. (Was man nicht alles fr ein bldes
       Titelbild tut...) */
    Blop::sendeFehlendeInitEvents();

    /* Die Blops sich malen lassen */
    Blop::beginGleichzeitig();
    
    /* Alle Grafiken lschen. Sollte nicht ntig sein, aber fhlt sich
       besser an. */
    Blop::lazyInitStapel();
    
    animiere();
    Blop::endGleichzeitig();
    mTitelbild.resize(grx * gric, gry * gric);
    {
      QPainter p(&mTitelbild);
      malUpdate(p, true); // true = nicht drauf schauen, ob die Zeit knapp ist
    }

  } catch (Fehler fe) {
    /* Wenn's einen Fehler gegeben hat, sollten wir das Spiel
       sicherheitshalber wieder stoppen... */
    stopLevel(false);
    throw fe;
  }
  
  stopLevel(false);
}



/**  */
void Spielfeld::paintEvent(QPaintEvent * e) {

  QRect r = e->rect();

  {
    QPainter p(this);

    if (mModus == modus_keinspiel) {
      /* Kein Spiel */
      
      if (mTitelbild.isNull()) {
	p.setBrush(QColor(150, 150, 150));
	p.setPen(NoPen);
	p.drawRect(r);
      } else {
	p.drawPixmap(0, 0, mTitelbild);
	/* Damit die Schrift das Titelbild nicht strt, nach unten
	   verschieben */
      }
      /* Schrift drbermalen. In der Mitte, falls es kein Titelbild gibt. */
      malSchrift(p, mTitelbild.isNull());

    } else if (Cuyo::gCuyo->spielPause()) {
      /* Spiel steht auf Pause */

      p.setBrush(QColor(150, 150, 150));
      p.setPen(NoPen);
      p.drawRect(r);
      if (Cuyo::gCuyo->mPauseBild) {
	Cuyo::gCuyo->mPauseBild->malBild(p,
		 (gric*grx - Cuyo::gCuyo->mPauseBild->getBreite())/2,
		 (gric*gry - Cuyo::gCuyo->mPauseBild->getHoehe())/2);
      }

      /* Schrift drbermalen. In der Mitte, falls es kein Pausebild gibt. */
      malSchrift(p, !Cuyo::gCuyo->mPauseBild);

    } else {
      /* Whrend des Spiels... */
      /* Achtung! malUpdateIntern() (und damit auch malUpdate()) sollte
         nicht aufgerufen werden. */
         
      p.drawPixmap(0, 0, mBild);
      
      /* Keine Schrift drbermalen; die Schrift sollte schon auf mBild
         gemalt sein. Auer, wir sind im da-ist-ein-Fehler-passiert-Modus.
	 (Siehe setText() fr Details.) */
      if (mModus == modus_fehler)
        malSchrift(p, true);

    }

  } // QPainter p(this)
}





/** Danach muss noch einmal animiere() aufgerufen werden, damit alle
    Blops wissen, wie sie aussehen, und damit die Grafik gemalt wird. */
void Spielfeld::startLevel() {
  /* Startmodus: Graue drfen kommen. Da noch keine da sind, kommt ein neues
     Fall */
  mModus = modus_neue_graue;
  mKettenreaktion = false;
	
  mRueberReihenModus = rrmodus_nix;
  mWillHartnaeckigReihe = false;
  mHochVerschiebung = 0; // Bild unverschoben
  mGrauAnz = 0; // Anzahl der Grauen, die auf ihr Kommen warten

  mMessageText = "";
  mMessageZeit = 0;
  
  /* Whrend der Spielfeld-Erzeugung werden ein Haufen Blops erzeugt und
     wieder vernichtet. Die sollen nicht alle gleich ein Init-Event
     bekommen. */
  //Blop::setInitEventsAutomatisch(false);

  /* Spielfeld leeren. */
  mDaten.init();
		
  /* Kein Fallendes */
  mFall.zerstoere();
		
  /* Anfangsgraskonfiguration schreiben */

  /* Etwas heuristisch dafr sorgen, dass keine gleichen Blops
     nebeneinander sind. Hier sind die Variablen dafr. */
  /* pr = PlusRand */
#define prmax (grx * gry * startlevel_rand_durchgaenge)
  /* Welche Blops sollen noch nachbarentfremdet werden? */
  int prx[prmax], pry[prmax], pranz = 0;


  /* Zeilen durchgehen... */
  for (int i = 0; i < ld->mAnfangsZeilen->getLaenge(); i++) {
    int y = gry - ld->mAnfangsZeilen->getLaenge() + i;
    __String zeile = ld->mAnfangsZeilen->getDatum(i,type_WortDatum)->getWort();
    
    CASSERT(zeile.length() == grx);
    for (int x = 0; x < grx; x++) {
      char c = zeile.data()[x];

      if (c == '+') {
	mDaten.getFeld(x, y) = Blop(Aufnahme::rnd(ld->mAnzFarben));
	/* "+"-Blop: Ein paar mal ins Nachbarentfremdungsarray
	   speichern... */
	for (int i = 0; i < startlevel_rand_durchgaenge; i++) {
	  CASSERT(pranz < prmax);
	  prx[pranz] = x;
	  pry[pranz] = y;
	  pranz++;
	}

      } else if (c == '-') {
	mDaten.getFeld(x, y) = Blop(blopart_grau);

      } else if (c >= '0' && c <= '9' && c - '0' < ld->mAnzFarben) {
        mDaten.getFeld(x, y) = Blop(c - '0');

      } else if (c >= 'A' && c <= 'Z'  ||  c >= 'a' && c <= 'z') {
        int n =  c >= 'a'  ?  c - 'a' + 26  :  c - 'A';
	mDaten.getFeld(x, y) = Blop(blopart_gras);
	mDaten.getFeld(x, y).setVersion(n);
	
      } else if (c != '.') {
        throw Fehler(_("Wrong character \"%c\" in startdist"), c);
      }
    }
  }

#undef prmax

  /* Hier kommt das eigentliche Nachbarentfremden */
  while (pranz > 0) {
    /* Zuflliges Element aus dem pr-Array rausnehmen */
    int prnr = Aufnahme::rnd(pranz);
    int x = prx[prnr];
    int y = pry[prnr];
    pranz--;
    prx[prnr] = prx[pranz];
    pry[prnr] = pry[pranz];

    /* Welche Farbe ist wie oft bei den Nachbarn vertreten? */
    int besetzt[max_farben_zahl];
    for (int i = 0; i < ld->mAnzFarben; i++)
      besetzt[i] = 0;
    /* !! Sehr provisorisch: Nur Nachbarschaftsverhltnisse von
       !! Farbe 0 bercksichtigen!!! */
    for (NachbarIterator ni(ld->mSorten[0], x, y); ni; ++ni) {
      if (mDaten.koordOK(ni.mX, ni.mY)) {
	int f = mDaten.getFeld(ni.mX, ni.mY).getFarbe();
	if (f != keine_farbe) besetzt[f]++;
      }
    }
			
    /* Wie wenig gleiche Nachbarn knnen wir erreichen? Und mit welchen
	 Farben? */
    int best = 0x7fff;
    int bestanz = 0;
    int bestli[max_farben_zahl];
    for (int j = 0; j < ld->mAnzFarben; j++) {
      if (besetzt[j] < best) {
	best = besetzt[j];
	bestanz = 0;
      }
      if (besetzt[j] == best)
	bestli[bestanz++] = j;
    }
				
    /* OK, jetzt eine dieser Mglichkeiten auswhlen. */
    mDaten.getFeld(x, y) = Blop(bestli[Aufnahme::rnd(bestanz)]);
  }


  /* Ok, jetzt drfen init-Events wieder automatisch verschickt werden. */
  //Blop::setInitEventsAutomatisch(true);

  /* Und dann verschicken wir mal noch schnell die fehlenden Init-Events. */
  //mDaten.initEvents();

  
  /* Levelzeit... */
  mZeit = 0;
  mHetzrandYPix = 0;
  /* Nicht "setHetzrandYPix(0)" aufrufen; sonst wird, falls mHetzrandYPix
     noch gar nicht initialisiert war, evtl. ein riesiger Bereich in die
     updateRegion geschrieben, was offenbar QT verwirrt. */
}



/** Spiel abbrechen (sofort, ohne Animation; oder die
    Animation ist schon vorbei).
    Wenn malen = true ist, Bildschirm sofort updaten.
    Wenn wegen_fehler = true ist, wird das Spiel trotzdem weiter
    angezeigt. */
void Spielfeld::stopLevel(bool malen, bool wegen_fehler /*= false*/) {
  if (mModus != modus_keinspiel)
    mModus = wegen_fehler ? modus_fehler : modus_keinspiel;
  if (malen)
    repaint(false);
}


/** Einmal pro Schritt aufrufen; kmmert sich ggf. um blinkendes
    Message */
void Spielfeld::blinkeMessage() {
  /* Ggf. Message-Text blinken lassen */
  if (mMessageZeit > 0) {
    mMessageZeit--;
    
    if (mMessageZeit % message_blink == 0) {
      /* Text entfernen / setzen; das "false" bedeutet, dass
         kein update-Event geschickt werden soll. */
      if ((mMessageZeit / message_blink) % 2 == 0)
        setText("", false);
      else
        setText(mMessageText, false);
    }
  }
}


/** Sorgt dafr, dass bei Spielende bei geeigneter Gelegenheit auch in diesem
    Spielfeld das Spiel beendet wird */
void Spielfeld::testeSpielende() {
  if (mModus != modus_testflopp) return;

  if (!Cuyo::gCuyo->spielLaeuft()) {
    /* Evtl. Fall platzen lassen */
    if (mFall.existiert()) {
      mFall.lassPlatzen();
      mModus = modus_fallplatz;
    } else
      mModus = modus_warte_auf_stop;
  }
}


/** Zusammenhangskomponenten bestimmen und ggf. Explosionen auslsen */
void Spielfeld::testeFlopp() {
  if (mModus != modus_testflopp) return;

  /* Hat jemand einen Platztest bestellt? (Allerdings mit dem Platzen noch
     warten, falls das Fall grade am zerfallen ist.)...
     nderung: Es wird nicht mehr geschaut, ob jemand einen Platztest
     bestellt hat. Etwas ineffektiver. */
  if (!mFall.istEinzel()) {

    /* Platzt was? (calcFlopp sendet ggf. auch die Grauen an den Mitspieler) */
    if (calcFlopp()) {
      mModus = modus_flopp;
      gSound->playSample(mRechterSpieler, sample_explodier);
    }
  }
}


/** Um Fall kmmern. (Aber nicht darum, neues Fall zu erzeugen). */
void Spielfeld::fallSchritt() {
  /* Um Fall kmmern. Macht auch ggf. notwendig gewordene
     Verwandlungen. */
  mFall.spielSchritt();
  /* Fr den KIPlayer: Jetzt ist das Fall nicht mehr neu. */
  mFallIstNeu = false;
};


/** Ein Schritt vom Spiel.
    Animationen und das Grafik-Update werden *nicht* gemacht.
    Dazu muss animiere() aufgerufen werden.
    (Weil alle Animationen innerhalb einer eigenen Gleichzeit
    stattfinden sollen.)
    spielSchritt() sollte innerhalb einer Gleichzeit aufgerufen
    werden fr evtl. auftretende Events. */
void Spielfeld::spielSchritt() {


  /* Immernoch in Modus testflopp? Dann wollte wohl nix explodieren. Also
     wieder in einen gravitations-Modus wechseln. */
     
  if (mModus == modus_testflopp) {
    /* Keine Sofort-Explosion, also keine Kettenreaktion */
    mKettenreaktion = false;
			
    /* Wenn ein Fall existiert, drfen grad keine Grauen nachkommen. Sonst
       schon. */
    if (mFall.existiert())
      mModus = modus_rutschnach;
    else
      mModus = modus_neue_graue;			
  }  // Ende von mModus == modus_testflopp

	
  switch (mModus) {
  					
  case modus_flopp: // etwas ist am explodieren
  	
    /* Einfach nur warten, bis die Blops fertig explodiert sind... */	 	
    if (!mDaten.getWasAmPlatzen()) {
      /* Ja, fertig explodiert. */

      /* Jetzt haben wir es uns verdient, vielleicht eine Reihe zu
         bekommen. */
      bekommVielleichtReihe();
    
      /* Wenn wir wieder in Modus testflopp kommen (und sofort eine
	 Explosion auftaucht), ist es eine Kettenreaktion. */
      mModus = modus_rutschnach;
      mKettenreaktion = true;
    }
    break;
  		
  case modus_rutschnach: {
    /* Steine runterrutschen lassen. */
    bool passiert_was = rutschNach(false) != rutschnach_nix;
  		
    /* Ist jetzt (in diesem Schritt) wirklich was passiert? */
    if (!passiert_was) {
      /* Nein. Das Nachrutschen ist also fertig. */
  			
      /* Jetzt: Nchste Explosion? */
      mModus = modus_testflopp;
    }		// if (es ist nix mehr nachgerutscht)
    break;
  }
  case modus_neue_graue: {
    /* Graue Steine runterrutschen lassen. */
    int rn = rutschNach(true);
  		
    /* Die Grauen sind weit genug unten, um ein neues Fall loszuschicken */
    if (rn != rutschnach_viel) {

      /* Aber nur, wenn das Spiel noch luft... */
      if (Cuyo::gCuyo->spielLaeuft()) {
  		
	if (mFall.erzeug()) {
        
          /* Fr den KIPlayer abpeichern, dass ein neues Fall kommt. */
          mFallIstNeu = true;
          
	  /* OK, neues Fall unterwegs. Sind berhaupt noch Graue
             unterwegs? Wenn ja, dann weiter im Gravitationsmodus;
             sonst auf Explosionen testen. */
	  if (rn == rutschnach_wenig)
	    mModus = modus_rutschnach;
	  else
	    mModus = modus_testflopp;
	} else {
	  /* Kein Platz mehr fr neuen Stein => tot */
	  emit sigTot();
	  CASSERT(!Cuyo::gCuyo->spielLaeuft());
	  /* Allerdings knnen wenigstens die Grauen noch fertig runterfallen.
	     Also sind wir noch nicht bereit zum stoppen. */
	}
      } else {
	/* Spiel luft eigentlich gar nicht mehr. Wenn die Grauen ganz auf dem
	   Boden angekommen sind, dann sind wir auch sterbebereit */
	if (rn == rutschnach_nix)
	  mModus = modus_warte_auf_stop;
      }
    }
    
    break;
  }
  	  			  	
  case modus_fallplatz: // Steine, die am runterfallen waren, zerplatzen 
    // in der Luft, weil das Spiel zu Ende ist
  	
    /* fertig explodiert? */
    if (!mFall.getAmPlatzen()) {
      /* Wir sind jetzt mit der Animation fertig und warten auf spielStop() */
      mFall.zerstoere();
      mModus = modus_warte_auf_stop;
    }
  		
    break;
  		
  case modus_warte_auf_stop:
    /* Wir warten nur darauf, dass wir ein stopLevel() bekommen */
    break;
  default:
    CASSERT(false);
  } // switch mModus


  /* ggf. Spiel fr dieses Spielfeld beenden */
  testeSpielende();


  /* Jetzt knnte man meinen, man will malUpdate() aufrufen, damit
     die Grafik upgedatet wird. Das wird aber im spteren animiere()-
     Aufruf getan. */
     
}   // spielSchritt()






/** Fhrt alle Animationen durch, und bei der Gelegenheit auch
    die Grafikupdates (auch die, die evtl. frher beim
    spielSchritt()-Aufruf angefallen sind). Sollte innerhalb einer
    Gleichzeit aufgerufen werden. */
void Spielfeld::animiere() {
  //CASSERT(gGleichZeit);
  /* Die ganzen festliegenden Blops animieren */
  mDaten.animiere();
  /* Fallendes animieren */
  mFall.animiere();

  /* Jetzt alle Grafik-Updates, die so angefallen sind, auf einmal machen */
  {
    QPainter p(this);
    malUpdate(p);
  }
  
}




/** Liefert die Koordinaten eines Felds in Pixeln zurck (ungespiegelt) */
QPoint Spielfeld::getFeldKoord(int x, int y) const {
  return QPoint(x * gric, y * gric - mHochVerschiebung -
		(ld->mSechseck && (x & 1)) * gric / 2);
}




/** Fhrt eine der nachfolgenden Tasten-Routinen aus.
    (t = taste_*). */
void Spielfeld::taste(int t) {
  switch (t) {
    case taste_links: tasteLinks(); break;
    case taste_rechts: tasteRechts(); break;
    case taste_dreh: tasteDreh(); break;
    case taste_fall: tasteFall(); break;    
    default: CASSERT(false);
  }
}


/** Bewegt das Fall eins nach links */
void Spielfeld::tasteLinks() {
  mFall.tasteLinks();
  gSound->playSample(mRechterSpieler, sample_links);
}

/** Bewegt das Fall eins nach rechts */
void Spielfeld::tasteRechts() {
  mFall.tasteRechts();
  gSound->playSample(mRechterSpieler, sample_rechts);
}

/** Dreht das Fall */
void Spielfeld::tasteDreh() {
  mFall.tasteDreh();
  gSound->playSample(mRechterSpieler, sample_dreh);
}

/** ndert die Fallgeschwindigkeit vom Fall */
void Spielfeld::tasteFall() {
  mFall.tasteFall();
  gSound->playSample(mRechterSpieler, sample_fall);
}




/** berechnet, ob und welche Blops platzen mssen. Auerdem
    werden den Blops die Kettengren mitgeteilt, und Graue
    und Punkte verteilt */
bool Spielfeld::calcFlopp() {
  int x, y;
  bool wasPassiert = false;
  /* Die Punkte werden nachher nur gesendet, wenn wirklich was
     explodiert ist. */
  int punkte = mKettenreaktion ? punkte_fuer_kettenreaktion : 0;
  
  /* Event an Blops senden, damit sie ihre Verbindungsspecials
     ausfhren knnen */
  mDaten.sendeConnectEvent();
	
  /* Anzahl der Grauen fr den anderen Spieler, wenn etwas platzt:
     Mindestens 1. Die minimal mgliche Blop-Zahl, die platzt,
     wird spter abgezogen. Und zwar von denen die grte.
     Ggf. wird was fr Kettenreaktionen addiert */
  int grz = 1;
  if (mKettenreaktion) grz += graue_bei_kettenreaktion;
  /* Der grte PlatzAnzahl-Wert unter den Sorten, die jetzt platzen */
  int maxPlatzAnzahl = 0;

  /* Array fr Zusammenhangskomponentensuche */
  int flopp[grx][gry];
	
  /* Erst mal muss nix platzen */
  for (x = 0; x < grx; x++)
    for (y = 0; y < gry; y++)
      flopp[x][y] = 0;
			
  /* Spielfeld absuchen */
  for (x = 0; x < grx; x++)
    for (y = 0; y < gry; y++)
      /* Normaler farbiger Blop, der noch nicht am platzen ist? */
      if (mDaten.getFeld(x, y).getArt() == blopart_farbe &&
	  !mDaten.getFeld(x, y).getAmPlatzen()) {
	int PlatzAnzahl = ld->mSorten[mDaten.getFeld(x,y).getFarbe()]
	  ->getPlatzAnzahl();
				/* Dann Kettengre berechnen,... */
	int anz = calcFloppRec(flopp, x, y,
			       mDaten.getFeld(x, y).getFarbe(),
			       -1);

        /* ... maxPlatzAnzahl fr grz aktualisieren ... */
	if ((anz >= PlatzAnzahl) && (PlatzAnzahl > maxPlatzAnzahl))
	  maxPlatzAnzahl = PlatzAnzahl;

	/* ... und allen Blops in der Kette mitteilen. */
	int neue_pt = calcFloppRec(flopp, x, y,
				   mDaten.getFeld(x, y).getFarbe(),
				   1, /* Gre mitteilen... */
				   anz >= PlatzAnzahl, /* Platzen? */
				   mKettenreaktion || !ld->mGrasBeiKettenreaktion, /* Auch Gras? */
				   anz);
	CASSERT(anz >= PlatzAnzahl  ||  neue_pt == 0);
				
	/* Wenn's Punkte gab, ist wohl auch was geplatzt */
	if (neue_pt > 0) {
	  punkte += neue_pt;
	  wasPassiert = true;
	  grz += anz;
	}	
      }

  /* Jetzt kennen wir maxPlatzAnzahl endgltig, jetzt lassens wir's wirken */
  grz -= maxPlatzAnzahl;

  /* Ist noch was explodiert? */
  if (wasPassiert) {
    /* Dann sende graue */
    emit sigSendeGraue(grz);
    /* und bekomme Punkte */			
    emit sigBekommPunkte(mRechterSpieler, punkte);
  }
	
  return wasPassiert;
} // calcFlopp()

/** sucht die Zusammenhangskomponente von x, y mit Farbe n.
    Setzt flopp[][] an den entspr. Stellen auf w.
    - Aufruf mit w = -1 um rauszufinden, wie viele gleichfarbige
      beisammen sind; liefert Anzahl der gleichfarbigen zurck;
    - Danach Aufruf mit w = 1 und richtigem anz-Wert, um den Blops ihre neue
      Kettengre mitzuteilen.
      Wenn platzen = true bergeben wird,
      - platzen die Blops (auch angrenzende
        Graue, und bei auch_gras = true auch angrenzendes Gras).
      - Liefert Anzahl der Punkte zurck, die's dafr gibt.
    @return Je nach w und platzen: Gre der Zshgskomp. oder Anz. d. Punkte oder 0 */
int Spielfeld::calcFloppRec(int flopp[grx][gry], int x, int y,
			    int n, int w, bool platzen /*= false*/,
			    bool auch_gras /*= false*/, int anz /*= 0*/) {
  CASSERT(x >= 0 && x < grx && y >= 0 && y < gry);
			    
  if (flopp[x][y] == w) // Hatten wir dieses Feld schon?
    return 0;
			
  int a = mDaten.getFeldArt(x, y);
  Blop & b = mDaten.getFeld(x, y);

  switch (a) {
  case blopart_keins:
    return 0;		
  case blopart_farbe: {
    /* Falsche Farbe? Ignorieren. */
    if (b.getFarbe() != n)
      return 0;
      
    int ret = 0;
    flopp[x][y] = w;
    if (w == 1) {
      b.setKettenGroesse(anz);
      if (platzen) {
	b.lassPlatzen();
	if (b.getVariable(spezvar_goalblob))
	  ret = punkte_fuer_gras;
	else
  	  ret = punkte_fuer_normales;
      }
    } else {
      /* Normalerweise 1... nur z. B. beim Go-Level nicht immer. */
      ret = b.getKettenBeitrag();
    }
    
    /* Rekursiv weiteraufrufen: */
    for (NachbarIterator i(ld->mSorten[n], x, y); i; ++i)
      if (i.mX >= 0 && i.mX < grx && i.mY >= 0 && i.mY < gry) {
        /* Nicht koordOk() verwenden, weil das die Rberreihe erlaubt, wenn
	   sie existiert (bei y == gry) */
	   
	/* Noch checken, ob einer der Blops ein "inhibit" fr diese Verbindung
	   gesetzt hat. */
	if ((b.getVariable(spezvar_inhibit) & i.mDir) == 0 &&
  	    (mDaten.getFeld(i.mX, i.mY).getVariable(spezvar_inhibit) & i.mDirOpp) == 0)
          ret += calcFloppRec(flopp, i.mX, i.mY, n, w, platzen, auch_gras, anz);
      }
      
    return ret;
  }
  case blopart_grau:
    if (w == -1 || !platzen) {
      return 0;
    } else {
      flopp[x][y] = w;
      b.lassPlatzen();
      return punkte_fuer_graues;
    }
  case blopart_gras:
    if (w == -1 || !platzen || !auch_gras) {
      return 0;
    } else {
      flopp[x][y] = w;
      b.lassPlatzen();
      return punkte_fuer_gras;
    }	
  }  // switch (a)
  CASSERT(0);
  return 0;  // Um keine Warnung zu bekommen
}




/** Ruft malUpdateIntern() auf und malt das interne Bild dann
    auf p.
    Bei force = true wird nicht darauf geschaut, ob die Zeit gerade
    knapp ist. */
void Spielfeld::malUpdate(QPainter & p, bool force /* = false */) {
  if (!force && !Cuyo::gCuyo->getInTime()) {
    /* Hilfe, die Zeit wird knapp. Grafik ausfallen lassen. */
    return;
  }

  malUpdateIntern();
  /* Jetzt auf den Bildschirm klatschen (oder was auch immer p ist): */
  p.drawPixmap(0, 0, mBild);
}



/** Hier findet die Grafik statt: Alles was in den mUpdateXXX-
    Variablen angegeben ist (auch in denen von den Blops selbst),
    wird auf den internen Puffer neu gezeichnet. */
void Spielfeld::malUpdateIntern() {


  for (int x = 0; x < grx; x++)
    for (int y = 0; y <= gry; y++)
      if (mDaten.koordOK(x, y) && mDaten.getFeld(x, y).takeUpdaten()) {
	QPoint pos = getFeldKoord(x, y);
	setUpdateRect(pos.x(), pos.y());
      }

  /* Erst mal alles offscreen malen */
  { 
    QPainter p(&mBild);
  
    /* Hintergrund updaten */

    // Gut, dass mUpdateRegion schon gespiegelt ist...
    p.setClipRegion(mUpdateRegion);
    
    p.setBrush(ld->mHintergrundFarbe);
    p.setPen(NoPen);
    p.drawRect(0, 0, width(), height());
    if (ld->mMitHintergrundbildchen) {
      ld->mHintergrundBild.malBild(p, 0,
			       ld->mSpiegeln ? 0 :
			       height() - ld->mHintergrundBild.getHoehe());
    }

    /* Clipping funktioniert fr die Bildchen eh nicht
       (wg. bitBlt statt drawPixmap). Also ausschalten. */
    p.setClipping(false);
    
    /* Blops updaten */
    int y0 = mHetzrandYPix / gric; // oberste sichtbare Reihe
    for (int x = 0; x < grx; x++)
      for (int y = y0; y < mDaten.getGrY(); y++) // Evtl. auch Rberreihe
	if (mUpdateDaten[x][y]) {
	  mUpdateDaten[x][y] = false;
	  
	  QPoint pos = getFeldKoord(x, y);
	  mDaten.getFeld(x, y).malen(p, pos.x(), pos.y());
          
          
  	  /* Beim Malen des Blops knnte das Fall oder der Hetzrand
	     bermalt worden sein. Ggf. neu malen: */
	  setUpdateRect(pos.x(), pos.y(), gric, gric, ebene_fall);
	}

    /* Fallendes updaten */
    if (mUpdateFall) {
      mUpdateFall = false;
      mFall.malen(p);

      /* Beim Malen des Falls knnte der Hetzrand
	 bermalt worden sein. Ggf. neu malen: */
      setUpdateFall(ebene_hetzrand);
    }
    
    /* Hetzrand updaten */
    if (mUpdateHetzrand) {
      mUpdateHetzrand = false;

      /* Scheint etwas schneller zu gehen, wenn man hier nicht clippt. */
      //p.setClipRegion(mUpdateRegion);

      /* Der nichtbild-Hetzrand */
      p.setBrush(ld->hetzrandFarbe);
      p.setPen(NoPen);
      p.drawRect(spiegelRect(QRect(0, 0, width(), mHetzrandYPix)));
      
      /* Der Bild-Hetzrand */
      if (ld->mMitHetzbildchen) {
	ld->mHetzBild.malBild(p, 0,
			  ld->mSpiegeln ?
			  height() - mHetzrandYPix - ld->mHetzrandUeberlapp :
			  mHetzrandYPix + ld->mHetzrandUeberlapp -
                          ld->mHetzBild.getHoehe()
			  );
      }

      //p.setClipping(false);
    }	
    
    malSchrift(p);
    
    
    /* Update-Region lschen. Darf erst hier am Ende passieren, weil
       sie zwischendrin nochmal vergrert werden knnte.
       Anscheinend hat Windows Probleme, wenn man "QRegion()" benutzt.
       Workaround... */
    mUpdateRegion = QRegion(QRect(0,0,0,0));

  } // QPainter p(&mBild)
  /* Jetzt enhlt mBild den richtigen Bildschirminhalt... */  
} // malUpdateIntern






/** Graue von anderem Spieler bekommen; wird ignoriert, falls dieser
    Spieler grad nicht spielt */
void Spielfeld::empfangeGraue(int g){
  if (mModus == modus_keinspiel)
    return;
  mGrauAnz += g;
}




/** liefert die Hhe vom hchsten Trmchen... unter der Annahme, dass
		er schon komplett zusammengesackt ist. */
int Spielfeld::getHoehe(){
  int h = 0;
  for (int x = 0; x < grx; x++) {
    int nh = 0;
    for (int y = 0; y < gry; y++)
      nh += mDaten.getFeldArt(x, y) != blopart_keins;
    if (nh > h) h = nh;
  }
	
  return h;		
}










/** sollte nur aufgerufen werden, wenn Cuyo::gCuyo->spielLaeuft()
    false liefert; liefert true, wenn alle Spiel-Stop-
    Animationen fertig sind; liefert brigens auch true,
    wenn dieser Spieler gar nicht mitspielt */
bool Spielfeld::bereitZumStoppen(){
  if (mModus == modus_keinspiel || mModus == modus_warte_auf_stop)
    return true;
  else
    return false;
}




/** ndert die Hhe vom Hetzrand auf y (in Pixeln). */
void Spielfeld::setHetzrandYPix(int y) {
  if (y == mHetzrandYPix)
    return;
	
  int vy = mHetzrandYPix;
  mHetzrandYPix = y;
	
  /* Wichtig: setUpdateRect darf erst aufgerufen werden,
     _nachdem_ mHetzrandYPix  gendert wurde (weil sonst
     der Hetzrand evtl. noch gar nicht in dem
     bergebenen Rechteck drin liegt und deshalb kein
     frisch-malen-Flag kriegt. */
  if (ld->mMitHetzbildchen)
    setUpdateRect(0, vy - ld->mHetzBild.getHoehe() + ld->mHetzrandUeberlapp,
		  gric * grx, ld->mHetzBild.getHoehe() + y - vy);
  else
    setUpdateRect(0, vy, gric * grx, y - vy);
}

/** Zeigt t gro an. (Oder weniger gro.) Auf Wunsch wird
    ein Update-Event gesendet. (Sonst ist der Aufrufer selbst
    dafr verantwortlich, dass der Text auch wirklich mal
    angezeigt wird.) */
void Spielfeld::setText(__String t, bool update_event /*= true*/,
                        bool kleine_schrift /*= false*/) {
  if (mText != t || mTextKlein != kleine_schrift) {
    mText = t;
    mTextKlein = kleine_schrift;
    
    /* Alles zum updaten markieren. (Ist nur whrend des Spiels ntig;
       aber wann anders strt's nicht. */
    setUpdateRect(0, 0, grx * gric, gry * gric);
    
    if (update_event) {
      /* Wenn ein Update-Event geschickt werden soll whrend das Spiel
         luft, dann wird das update-Event allein nix bringen, weil
	 die Grafik auf dem Offscreen-Buffer noch alt ist. Also muss
	 die in dem Fall auch upgedatet werden. */
	 
      if (mModus != modus_keinspiel)
        malUpdateIntern();
	
      update();
    }
  }
}

/** Fr whrend des Spiels: Setzt einen Text, der ein paar mal
    aufblinkt. */
void Spielfeld::setMessage(__String mess) {
  mMessageText = mess;
  mMessageZeit = message_gesamtzeit;
}


/** Wenn der Level gespiegelt ist, wird auch r gespiegelt */
QRect Spielfeld::spiegelRect(const QRect & r){
  if (ld->mSpiegeln) {
    // bottom = top + height - 1 ... schade eigentlich.
    return QRect(r.left(), gry * gric - (r.bottom() + 1), r.width(), r.height());
  } else
    return r;
}

/** Setzt das Rechteck r auf upzudaten. Der ebene-Parameter sollte
    (im Moment) nur von malUpdate() benutzt werden. */
void Spielfeld::setUpdateRect(QRect r, int ebene /*= ebene_hintergrund*/) {

  /* Erst mal die Update-Regionen updaten. */
  QRegion nreg(spiegelRect(r));
  if (mUpdateRegion.isNull())
    mUpdateRegion = nreg;
  else {
    mUpdateRegion = mUpdateRegion.unite(nreg);
    /* mUpdateRegion += nreg;
      wre schner; geht aber mit QT 1 nicht. */
  }
  
  /* Und jetzt noch die speziellen update-Variablen setzen */
	
  /* Fr die Daten und das Fallende das Rechteck vorher verschieben... */
  QRect r2 = r;
  r2.moveBy(0, mHochVerschiebung);
  r2 = r2.intersect(QRect(0, 0, grx * gric, gry * gric));
	
  if (ebene <= ebene_blops) {
    /* Blops */
    for (int x = 0; x < grx; x++)
      for (int y = 0; y <= gry; y++) {
	QPoint pos = getFeldKoord(x, y);
	if (r.intersects(QRect(pos.x(), pos.y(), gric, gric)))
	  mUpdateDaten[x][y] = true;
      }
  }
	
  if (ebene <= ebene_fall) {
    /* Fallendes */
    if (mFall.existiert() && r2.intersects(mFall.getRect()))
      mUpdateFall = true;
  }

  if (ebene <= ebene_hetzrand) {
    /* Hetzrand */
    if (r.top() < mHetzrandYPix + ld->mHetzrandUeberlapp)
      mUpdateHetzrand = true;
  }
}

/** Setzt das Rechteck x, y, w, h auf upzudaten. */
void Spielfeld::setUpdateRect(int x, int y, int w /*= gric*/, int h /*= gric*/,
			      int ebene /*= ebene_hintergrund*/) {
  setUpdateRect(QRect(x, y, w, h), ebene);
}

/** setzt den Bereich vom Fallenden auf upzudaten */
void Spielfeld::setUpdateFall(int ebene /*= ebene_hintergrund*/) {
  if (mFall.existiert())
    setUpdateRect(mFall.getRect(), ebene);
}

/** Setzt alles auer den Hetzrand auf upzudaten. */
void Spielfeld::setUpdateFastAlles() {
  setUpdateRect(0, mHetzrandYPix, grx * gric, gry * gric - mHetzrandYPix);
}

/** Lsst den Hetzrand schnell runterkommen (fr die
		Zeitbonus-Animation). Liefert true, wenn fertig. */
bool Spielfeld::bonusSchritt() {

  CASSERT(mModus != modus_keinspiel);
	
  /* Wenn der Hetzrand schon weiter unten ist, als wir ihn runterkommen
     lassen wrden, gar nix tun. */
  int unten = gric * gry - ld->mHetzrandUeberlapp;
  if (mHetzrandYPix > unten)
    return true;

  /* Hetzrand runterkommen lassen */
  int ny = mHetzrandYPix + bonus_geschwindigkeit;
  if (ny > unten) ny = unten;

  setHetzrandYPix(ny);
	
  /* Ausgeben... */
  {
    QPainter p(this);
    malUpdate(p);
  } // QPainter p(this)*/

  /* Hetzrand unten angekommen? */
  return ny == unten;
}

/** Malt die Schrift auf den Bildschirm; genauer: auf p.
    In der richtigen Farbe. */
void Spielfeld::malSchrift(QPainter & p, bool mitte /*= true*/) {

  /* Wenn eh kein Text da ist, dann lieber gar nixh mit Schriften
     machen; wer wei, ob das langsam ist. */
  if (mText == "")
    return;
    

  int align = WordBreak | AlignHCenter |
	       (mitte ? AlignVCenter : AlignBottom);
  
  if (mTextKlein) {
    p.setPen(black);
    p.setFont(QFont("helvtica", 12, QFont::Normal, false));
    p.drawText(rect(), align, mText);
  } else {
    p.setFont(QFont("helvtica", 18, QFont::Bold, false));
    QRect r = rect();

    p.setPen(ld->mSchriftFarbe[schrift_hell]);
    r.moveTopLeft(QPoint(-1, 0));
    p.drawText(r, align, mText);
    r.moveTopLeft(QPoint(0, -1));
    p.drawText(r, align, mText);
    
    p.setPen(ld->mSchriftFarbe[schrift_dunkel]);
    r.moveTopLeft(QPoint(1, 0));
    p.drawText(r, align, mText);
    r.moveTopLeft(QPoint(0, 1));
    p.drawText(r, align, mText);
    
    p.setPen(ld->mSchriftFarbe[schrift_normal]);
    r.moveTopLeft(QPoint(0, 0));
    p.drawText(r, align, mText);
  }
}

/** Liefert true, wenn grade ein Fallendes unterwegs ist.
    Wird vom KIPlayer bentigt */
int Spielfeld::getFallModus() const {
  //return mModus == modus_fall;
  if (mFall.existiert())
    if (mFallIstNeu)
      return fallmodus_neu;
    else
      return fallmodus_unterwegs;
  else
    return fallmodus_keins;
}

/** Liefert true, wenn noch Gras da ist (d. h. wenn nicht gewonnen.) */
bool Spielfeld::istGrasDa() const {
  /* Jedes Mal alles durchsuchen: ineffektiv. Aber wenigstens von unten
     nach oben, weil das Gras ja eher unten liegt... */
  for (int y = mDaten.getGrY() - 1; y >= 0; y--)
    for (int x = 0; x < grx; x++)
      if (mDaten.getFeldArt(x, y) == blopart_gras ||
          mDaten.getFeld(x, y).getVariable(spezvar_goalblob))
        return true;
  return false;
}


/** Liefert einen Pointer auf das Blopgitter zurck. */
/*const*/ BlopGitter * Spielfeld::getDatenPtr() /*const*/ {
  return &mDaten;
}

/** Liefert einen Pointer auf die fallenden Blops zurck.
    Wird von KIPlayer einmal am Anfang aufgerufen. */
const Blop * Spielfeld::getFallPtr() {
  return mFall.getBlopPtr();
}

/** Liefert die Pos. zurck, an der neue Dinge oben
    auftauchen. */
int Spielfeld::getHetzrandYAuftauch() const{
  /* Siehe def. der Konstante... */
  return (mHetzrandYPix + hetzrand_dy_auftauch) / gric;
}

/** Liefert die Pos. zurck, bis wohin noch Dinge liegen
    drfen, ohne dass man tot ist. */
int Spielfeld::getHetzrandYErlaubt() const{
  /* Siehe def. der Konstante... */
  return (mHetzrandYPix + hetzrand_dy_erlaubt) / gric;
}

/** Bewegt den Hetzrand eins nach unten. Testet auch, ob dabei
    was berdeckt wird. */
void Spielfeld::bewegeHetzrand(){
  /* Hat der Hetzrand Steine berdeckt? (Bruchte eigentlich
     nur getestet werden, wenn sich
     getHetzrandYErlaubt() seit dem letzten Check erhht hat...) */
  int hye = getHetzrandYErlaubt();
  if (hye > 0) {
    for (int x = 0; x < grx; x++)
      if (mDaten.getFeldArt(x, hye - 1) != blopart_keins) {
	/* Hetzrand berdeckt Stein => tot */
	emit sigTot();
	CASSERT(!Cuyo::gCuyo->spielLaeuft());
	/* Evtl. muss auch bei uns noch eine Animation fertig
	   ablaufen; deshalb noch nicht gleich den mModus auf
	   modus_warte_auf_stop setzen, sondern
	   einfach die Dinge laufen lassen */
	break;
      }
  }
  /* Levelzeit hochzhlen, Hetzrand kommen lassen...; das darf
     erst geschehen,  nachdem getestet wurde, ob was berdeckt
     wurde, weil vielleicht vorher grade
     was in der Hhe aufgetaucht ist, wo es jetzt berdeckt wrde */
  mZeit++;
  setHetzrandYPix(mZeit / ld->hetzrandZeit);
}
/** Sendet sich selbst (ggf.) zufllige Graue */
void Spielfeld::zufallsGraue(){
  /* Ggf. sich selbst Graue schicken */
  if (ld->mZufallsGraue != zufallsgraue_keine)
    if (Aufnahme::rnd(ld->mZufallsGraue) == 0)
      empfangeGraue(1);
}

/** Lsst in der Luft hngende Blops ein Stck runterfallen.
    Liefert zurck, ob sich nichts bewegt hat, nur unten oder auch oben.
    Bei auchGraue = true, kommen auch Graue, die ganz ber
    dem Spielfeld hngen. */
int Spielfeld::rutschNach(bool auchGraue){
  int ret = rutschnach_nix;
		
  /* Anzahl der Spalten, in denen Platz fr ein neues Graues ist */
  int grauplatz = 0;

  /* Zeile, in der neue Graue auftauchen */
  int hya = getHetzrandYAuftauch();
			
  /* Steine nachrutschen lassen */
  for (int x = 0; x < grx; x++) {
    int y = gry - 1;
			
    /* Zunchst mal die Steine, die schon im Spielfeld sind, runterrutschen
       lassen; die knnen brigens auch noch leicht ber dem Hetzrand sein
       (also sicherheitshalber bis ganz oben gehen) */
    while (y >= 0 && mDaten.getFeldArt(x, y) != blopart_keins)
      y--; // suche unterstes leeres

    if (y >= 0) {	// wenn es leere Felder gab, dann nachrutschen
      for (; y > 0; y--) {
	if (mDaten.getFeldArt(x, y - 1) != blopart_keins) {
	  mDaten.getFeld(x, y) = mDaten.getFeld(x, y - 1);
	  mDaten.getFeld(x, y - 1) = Blop(blopart_keins);
	  if (y > hya + neues_fall_platz)
	    ret |= rutschnach_wenig;
	  else
	    ret |= rutschnach_viel;
	}
      }
				
    } // Ende von da ist noch Platz in der Spalte
			
    /* Ist in dieser Spalte Platz fr ein Graues? */
    if (mDaten.getFeld(x, hya).getArt() == blopart_keins)
      grauplatz++;
  } // for x

  /* Drfen neue Graue kommen? */
  if (auchGraue) {
    for (int x = 0; x < grx; x++) {
      /* Ist in Grauauftauchhhe der Spalte Platz? (Achtung:
	 Spalte voll ist nicht der einzige Grund fr keinen
	 Grauplatz. Der Hetzrand knnte auch
	 grade ein Feld runtergekommen sein.) */
      if (mDaten.getFeldArt(x, hya) == blopart_keins) {
	CASSERT(grauplatz > 0);
	if (Aufnahme::rnd(grauplatz) < mGrauAnz) {
	  mDaten.getFeld(x, hya) = Blop(blopart_grau);
	  ret |= rutschnach_viel;
	  mGrauAnz--;
	}
	grauplatz--;
      }
    }
  }
	
  return ret;
}
/** Liefert die Pos. vom Hetzrand in Pixeln zurck. Wird
    vom Fall gebraucht. */
int Spielfeld::getHetzrandYPix() const {
  return mHetzrandYPix;
}




/** Kmmert sich um hin- und hergeben von Reihen. */
void Spielfeld::rueberReihenSchritt() {
  switch (mRueberReihenModus) {
  case rrmodus_nix:
    break;
			
  case rrmodus_gib_rueber:
    /* Nichts tun. Nur warten, bis uns der andere die Reihe ganz
       weggenommen hat. */
    if (mRestRueberReihe == 0)
      mRueberReihenModus = rrmodus_gib_runter;

    /*
    if (mRechterSpieler) {
      if (mDaten.getFeldArt(0, gry) == blopart_keins)
	mRueberReihenModus = rrmodus_gib_runter;
    } else {
      if (mDaten.getFeldArt(grx - 1, gry) == blopart_keins)
	mRueberReihenModus = rrmodus_gib_runter;
	}*/
  		
    break;
  case rrmodus_gib_runter:
    /* alles zum updaten markieren */
    setUpdateFastAlles();
  		
    /* Bild verschieben */
    mHochVerschiebung -= reihe_rueber_senkrecht_pixel;
  	
    /* Weit genug verschoben? */
    if (mHochVerschiebung <= 0) {
      mHochVerschiebung = 0;
  	  	
      /* Rueberreihe vernichten */
      mDaten.setRueberReihe(false);
  	  	
      /* und fertig */
      mRueberReihenModus = rrmodus_nix;
    }
    break;
  case rrmodus_bekomm_hoch:
	
    /* alles zum updaten markieren */
    setUpdateFastAlles();
  			 	
    /* Bild verschieben */
    mHochVerschiebung += reihe_rueber_senkrecht_pixel;
  	
    /* Weit genug verschoben? */
    if (mHochVerschiebung >= gric) {
      mHochVerschiebung = gric;
  	  	  	  	
      mRueberReihenModus = rrmodus_bekomm_rueber;
      mRestRueberReihe = grx;
    }
    break;
			
  case rrmodus_bekomm_rueber:
    /* Neuen Stein vom anderen Spieler bestellen */
    Blop neuer;
    emit sigWillStein(neuer);
    
    /* Von welcher Seite kommt die Reihe? */
    if (mRechterSpieler) {
      /* von links: Reihe nach rechts schieben */
      for (int x = grx - 1; x > 0; x--)
        mDaten.getFeld(x, gry) = mDaten.getFeld(x - 1, gry);
      mDaten.getFeld(0, gry) = neuer;

      /* Seitenwechsel-Event */
      //Blop::beginGleichzeitig();
      mDaten.getFeld(0, gry).execEvent(event_changeside);
      //Blop::endGleichzeitig();
      		
    } else {
      /* von rechts: Reihe nach links schieben */
      for (int x = 0; x < grx - 1; x++)
        mDaten.getFeld(x, gry) = mDaten.getFeld(x + 1, gry);
      mDaten.getFeld(grx - 1, gry) = neuer;

      /* Seitenwechsel-Event */
      //Blop::beginGleichzeitig();
      mDaten.getFeld(grx - 1, gry).execEvent(event_changeside);
      //Blop::endGleichzeitig();
    }

    mRestRueberReihe--;

    /* Fertig? */
    if (mRestRueberReihe == 0) {
      mRueberReihenModus = rrmodus_nix;

      Blop::gGlobalBlop.execEvent(event_row_up1 + mRechterSpieler);

      /* Im Datenarray alles nach oben schieben. */
      for (int x = 0; x < grx; x++)
	for (int y = 0; y < gry; y++)
	  mDaten.getFeld(x, y) = mDaten.getFeld(x, y + 1);
      /* In der Rberreihe sind noch alte Restblops; aber die Rberreihe
         wird jetzt eh deaktiviert */
      mDaten.setRueberReihe(false);
      /* Optisch soll sich nix ndern; jetzt ist das Spielfeld wieder in
	 Normalposition */
      mHochVerschiebung = 0;

      /* Vielleicht sollten wir gleich noch eine Reihe bekommen?
         Mal nachschauen. Allerdings ist der andere Spieler noch mit
	 der letzten Reihe beschftigt. Deshalb warten wir hartnckig
	 so lange, bis der sich richtig entschieden hat, ob er uns eine
	 Reihe gibt oder nicht. */
      mWillHartnaeckigReihe = true;
    }
    break;
  }    // switch (mRueberReihenModus)
  
  /* Wir haben irgend wann beschlossen, so lange beim anderen Spieler
     nach einer Reihe zu fragen, bis der wei, ob er uns eine gibt: */
  if (mWillHartnaeckigReihe) {
    /* bekommVielleichtReihe() liefert true, wenn es sich lohnt, im
       nchsten Schritt nochmal nachzufragen. */
    if (!bekommVielleichtReihe())
      mWillHartnaeckigReihe = false;
  }
}

/** liefert (in ret) zurck, ob wir dem anderen
    Spieler eine Reihe geben (er hat Hhe h); eine der Konstanten
    bewege_reihe_xxx */
void Spielfeld::gebReihe(int h, int & ret){
  ret = bewege_reihe_nein;

  /* Wenn wir nicht mitspielen, geben wir natrlich auch keine Reihe her */	
  if (mModus == modus_keinspiel)
    return;
	
  /* Wenn das Spiel zu Ende gehen soll, keine Reihe mehr rbergeben */
  if (!Cuyo::gCuyo->spielLaeuft())
    return;
		
  /* Wenn wir schon irgendwie mit einer Reihe beschftigt sind, soll der
     andere sich erst noch ein bisschen gedulden. */
  if (mRueberReihenModus != rrmodus_nix) {
    ret = bewege_reihe_moment;
    return;
  }
		
  /* Sind wir hinreichend hher als der andere? Oder testen wir im
     Debug-Modus das Reihen hin und her schieben? */
  if (getHoehe() >= h + 2 || Cuyo::gCuyo->mRueberReihenTest) {
		
    /* Ist unsere unterste Reihe (komplett) verfgbar? */
    ret = bewege_reihe_ja;
    for (int x = 0; x < grx; x++)
      if (mDaten.getFeldArt(x, gry - 1) == blopart_keins ||
	  mDaten.getFeld(x, gry - 1).getAmPlatzen())
	ret = bewege_reihe_nein;
  }
	
  /* Wenn wir dem anderen Spieler eine Reihe geben wollen, dann
     machen wir uns auch mal dafr bereit */
  if (ret == bewege_reihe_ja) {
    mRueberReihenModus = rrmodus_gib_rueber;
    mRestRueberReihe = grx;
    
    Blop::gGlobalBlop.execEvent(event_row_down1 + mRechterSpieler);

    /* Im Datenarray alles nach unten schieben. */
    mDaten.setRueberReihe(true);
    for (int x = 0; x < grx; x++) {
      for (int y = gry; y > 0; y--)
	mDaten.getFeld(x, y) = mDaten.getFeld(x, y - 1);
      /* Oben eine Reihe leer einfgen. */
      mDaten.getFeld(x, 0) = Blop(blopart_keins);
    }
    /* Optisch soll sich noch nix ndern... */
    mHochVerschiebung = gric;
    
    /* Hier fehlt noch ein Signal an die Steine der neuen untersten Reihe,
       um ihnen mitzuteilen, dass ihre Zshgskomponente kleiner geworden ist. */

    /* brigens: Das Reihe-Rbergeben fngt (fr uns)
       noch nicht gleich an; erst, wenn der andere
       sein Zeug hochgeschoben hat. */
  }
}


/** gibt einen Stein an den anderen Spieler
    rber; in s wird die Steinfarbe zurckgeliefert */
void Spielfeld::gebStein(Blop & s){
  CASSERT(mRueberReihenModus == rrmodus_gib_rueber);
  CASSERT(mRestRueberReihe > 0);
	
  if (mRechterSpieler) {
    /* nach links... */
    s = mDaten.getFeld(0, gry);
    for (int x = 0; x < grx - 1; x++)
      mDaten.getFeld(x, gry) = mDaten.getFeld(x + 1, gry);
    mDaten.getFeld(grx - 1, gry) = Blop(blopart_keins);
  } else {
    /* nach rechts... */
    s = mDaten.getFeld(grx - 1, gry);
    for (int x = grx - 1; x > 0; x--)
      mDaten.getFeld(x, gry) = mDaten.getFeld(x - 1, gry);
    mDaten.getFeld(0, gry) = Blop(blopart_keins);
  }

  mRestRueberReihe--;
}


/** Prft, ob ein Reihenbekommen sinnvoll wre und initiiert es ggf.
    (Unterhlt sich auch mit dem anderen Spieler). Liefert true,
    wenn es jetzt grad nicht mglich war, eine Reihe zu bekommen,
    aber nur weil der andere Spieler noch damit beschftigt war,
    von einer vorigen Reihe sein Spielfeld runterzuschieben. Unter
    manchen Umstnden wird dann spter nochmal probiert, eine
    Reihe zu bekommen. */
bool Spielfeld::bekommVielleichtReihe() {
  /* Knnen wir eine Reihe vertragen? */
  if (mRueberReihenModus != rrmodus_nix)
    return false;
    
  /* Dann mal beim anderen Spieler anfragen, ob wir eine bekommen */
  int erg;
  emit sigWillReihe(getHoehe(), erg);
  if (erg == bewege_reihe_ja) {
    mDaten.setRueberReihe(true);
    /* Die Rberreihe lschen. (Da kann noch Mll von vorher drin sein. */
    for (int x = 0; x < grx; x++)
      mDaten.getFeld(x, gry) = Blop(blopart_keins);
    mRueberReihenModus = rrmodus_bekomm_hoch;
  }
  /* Wenn der andere Spieler grad noch mit einer alten Reihe beschftigt
     ist, liefern wir true zurck, um es vielleicht etwas spter nochmal
     zu probieren. */
  return erg == bewege_reihe_moment;
}



/** Liefert die (rberreihenbedingte) Hochverschiebung
    des gesamten Spielfelds. Wird vom Fall bentigt (um
    seine Koordinaten in Feldern zu berechnen). */
int Spielfeld::getHochVerschiebung(){
  return mHochVerschiebung;
}




