/*
 * Copyright (C) 1999-2012. Christian Heller.
 *
 * This file is part of the Cybernetics Oriented Interpreter (CYBOI).
 *
 * CYBOI is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * CYBOI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CYBOI. If not, see <http://www.gnu.org/licenses/>.
 *
 * Cybernetics Oriented Programming (CYBOP) <http://www.cybop.org/>
 * Christian Heller <christian.heller@tuxtax.de>
 *
 * @version CYBOP 0.12.0 2012-08-22
 * @author Christian Heller <christian.heller@tuxtax.de>
 */

#ifndef CHECK_EVENTS_X_WINDOW_SYSTEM_SENSOR_SOURCE
#define CHECK_EVENTS_X_WINDOW_SYSTEM_SENSOR_SOURCE

#ifdef GNU_LINUX_OPERATING_SYSTEM

#include <X11/Xlib.h>
//?? #include <X11/Xutil.h>
#include <pthread.h>
#include <signal.h>

#include "../../../../constant/model/cyboi/state/boolean_state_cyboi_model.c"
#include "../../../../constant/model/cyboi/state/integer_state_cyboi_model.c"
#include "../../../../constant/model/cyboi/state/pointer_state_cyboi_model.c"
#include "../../../../constant/type/cyboi/state_cyboi_type.c"

/**
 * Checks for x window system events.
 *
 * As an exception to other procedures in CYBOI, parametres are NOT handed over
 * as void* to this procedure, in order to avoid type casts and to gain faster
 * processing results.
 *
 * Another exception is that this procedure is actually a function, since it
 * returns a value, as opposed to other procedures in CYBOI which return
 * nothing (void).
 *
 * @param mt the mutex
 * @param d the display
 */
int sense_x_window_system_check_events(pthread_mutex_t* mt, struct _XDisplay* d) {

    // The mutex and display were already checked
    // for not being null in the calling function,
    // and are therefore not tested here again.

    // The number of events.
    int n = *NUMBER_0_INTEGER_STATE_CYBOI_MODEL;

    // Lock x window system mutex.
    pthread_mutex_lock(mt);

    // Check the number of events in the event queue.
    // XEventsQueued always returns immediately without
    // input/ output if there are events already in the queue.
    //
    // There are three possible modes:
    // 1 QueuedAlready: XEventsQueued returns the number of events
    //   already in the event queue (and never performs a system call).
    //   XEventsQueued with mode QueuedAlready is identical
    //   to the XQLength function.
    // 2 QueuedAfterFlush: XEventsQueued returns the number of events
    //   already in the queue if the number is nonzero. If there are no
    //   events in the queue, XEventsQueued flushes the output buffer,
    //   attempts to read more events out of the applications connection,
    //   and returns the number read.
    // 3 QueuedAfterReading: XEventsQueued returns the number of events
    //   already in the queue if the number is nonzero. If there are no
    //   events in the queue, XEventsQueued attempts to read more events
    //   out of the applications connection WITHOUT flushing the output
    //   buffer and returns the number read.
    //
    // The decision fell on mode number 3, because:
    // - mode number 1 did not display the x window initially, since
    //   probably no expose events are placed in the queue at startup
    // - mode number 2 is undesirable, since it would flush the output
    //   buffer and might thus cause this sense-thread to conflict
    //   with the send_x_window_system procedure of the main thread
    n = XEventsQueued(d, QueuedAfterReading);

    // Unlock x window system mutex.
    pthread_mutex_unlock(mt);

    return n;
}

/* GNU_LINUX_OPERATING_SYSTEM */
#endif

/* CHECK_EVENTS_X_WINDOW_SYSTEM_SENSOR_SOURCE */
#endif
