/*
 * Copyright (C) 1999-2022. Christian Heller.
 *
 * This file is part of the Cybernetics Oriented Interpreter (CYBOI).
 *
 * CYBOI is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * CYBOI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CYBOI. If not, see <http://www.gnu.org/licenses/>.
 *
 * Cybernetics Oriented Programming (CYBOP) <http://www.cybop.org/>
 * CYBOP Developers <cybop-developers@nongnu.org>
 *
 * @version CYBOP 0.23.0 2022-09-04
 * @author Christian Heller <christian.heller@cybop.org>
 */

#ifndef TERMINAL_AWAKENER_SOURCE
#define TERMINAL_AWAKENER_SOURCE

#include <sys/ioctl.h> // ioctl
#include <errno.h> // errno

#include "../../constant/model/character_code/ascii/ascii_character_code_model.c"
#include "../../constant/model/cyboi/log/level_log_cyboi_model.c"
#include "../../constant/model/cyboi/state/integer_state_cyboi_model.c"
#include "../../logger/logger.c"

/**
 * Let the system send an input to itself over terminal.
 *
 * @param p0 the device file descriptor
 */
void awake_terminal(void* p0) {

    log_message_terminated((void*) DEBUG_LEVEL_LOG_CYBOI_MODEL, (void*) L"Awake terminal.");
    fwprintf(stdout, L"Debug: Awake terminal. p0: %i\n", p0);

    //
    // Initialise error number.
    //
    // It is a global variable and other operations
    // may have set some value that is not wanted here.
    //
    // CAUTION! Initialise the error number BEFORE calling
    // the function that might cause an error.
    //
    errno = *NUMBER_0_INTEGER_STATE_CYBOI_MODEL;

    //
    // Write (push back) character to terminal input buffer,
    // so that it can be detected by the sensing thread function.
    //
    // The TIOCSTI command is known to at least Linux and BSD.
    // The characters handed over are limited to 4096 on Linux.
    //
    // Example:
    // char* text = "text";
    // ioctl(STDIN_FILENO, TIOCSTI, text);
    //
    // CAUTION! Calling the function "write" does NOT work here.
    // It writes the characters to the terminal screen,
    // no matter if sent to STDIN_FILENO or STDOUT_FILENO.
    // But it is NOT recognised by the sensing thread.
    //
    int r = ioctl(p0, TIOCSTI, LINE_FEED_ASCII_CHARACTER_CODE_MODEL);

    //
    // The meaning of the returned value depends upon the command used.
    //
    // Linux:
    // - success: zero or non-negative value
    // - error: -1 and errno set appropriately
    // - special case: sometimes used as output parameter
    //
    if (r >= *NUMBER_0_INTEGER_STATE_CYBOI_MODEL) {

        log_message_terminated((void*) DEBUG_LEVEL_LOG_CYBOI_MODEL, (void*) L"Awake terminal. Success.");
        fwprintf(stdout, L"Debug: Awake terminal. Success. r: %i\n", r);

    } else {

        log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not awake terminal. An error occured.");
        fwprintf(stdout, L"Error: Could not awake terminal. An error occured. %i\n", r);
        log_errno((void*) &errno);
    }
}

/* TERMINAL_AWAKENER_SOURCE */
#endif
