/*
 * Copyright (C) 1999-2023. Christian Heller.
 *
 * This file is part of the Cybernetics Oriented Interpreter (CYBOI).
 *
 * CYBOI is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published
 * by the Free Software Foundation, either version 3 of the License,
 * or (at your option) any later version.
 *
 * CYBOI is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with CYBOI. If not, see <http://www.gnu.org/licenses/>.
 *
 * Cybernetics Oriented Programming (CYBOP) <http://www.cybop.org/>
 * CYBOP Developers <cybop-developers@nongnu.org>
 *
 * @version CYBOP 0.26.0 2023-04-04
 * @author Christian Heller <christian.heller@cybop.org>
 */

#ifndef MUTEX_DEALLOCATOR_SOURCE
#define MUTEX_DEALLOCATOR_SOURCE

//
// System interface
//

#include <stdio.h> // stdout
#include <threads.h> // mtx_t, mtx_destroy
#include <wchar.h> // fwprintf

//
// Library interface
//

#include "constant.h"
#include "knowledge.h"
#include "logger.h"

/**
 * Deallocates the mutex.
 *
 * @param p0 the mutex (pointer reference)
 */
void deallocate_mutex(void* p0) {

    if (p0 != *NULL_POINTER_STATE_CYBOI_MODEL) {

        void** m = (void**) p0;

        log_message_terminated((void*) DEBUG_LEVEL_LOG_CYBOI_MODEL, (void*) L"Deallocate mutex.");
        //?? fwprintf(stdout, L"Debug: Deallocate mutex. p0: %i\n", p0);
        //?? fwprintf(stdout, L"Debug: Deallocate mutex. *p0: %i\n", *((int*) p0));

        // Cast mutex to correct type.
        mtx_t* t = (mtx_t*) *m;

        // Finalise mutex.
        mtx_destroy(t);

        //
        // Deallocate mutex.
        //
        // CAUTION! The second argument "count" is NULL,
        // since it is only needed for looping elements of type PART,
        // in order to decrement the rubbish (garbage) collection counter.
        //
        deallocate_array(p0, *NULL_POINTER_STATE_CYBOI_MODEL, (void*) PRIMITIVE_STATE_CYBOI_MODEL_COUNT, (void*) MUTEX_THREAD_STATE_CYBOI_TYPE);

    } else {

        log_message_terminated((void*) ERROR_LEVEL_LOG_CYBOI_MODEL, (void*) L"Could not deallocate mutex. The mutex is null.");
        fwprintf(stdout, L"Error: Could not deallocate mutex. The mutex is null. p0: %i\n", p0);
    }
}

/* MUTEX_DEALLOCATOR_SOURCE */
#endif
