;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2021 Jan (janneke) Nieuwenhuizen <janneke@gnu.org>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.

(define-module (main)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (ice-9 rdelim)
  #:use-module (oop goops)
  #:use-module (dzn runtime)
  #:use-module (dzn pump)
  #:use-module (dzn pump)
  #:use-module (async_calling_context)
  #:duplicates (merge-generics)
  #:export (main))

(define relaxed? #f)

(define (drop-prefix string prefix)
  (if (string-prefix? prefix string)
    (substring string (string-length prefix))
    string))

(define (consume-synchronous-out-events prefix event event-alist)
  (let ((match (string-append prefix event)))
    (let loop ((s (read-line)))
      (and s
        (not (eof-object? s))
        (not (equal? s match))
        (loop (read-line)))))
  (let loop ((s (read-line)))
    (let ((event (and s
            (not (eof-object? s))
            (assoc-ref event-alist (string->symbol s)))))
      (if (not event) (and s (not (eof-object? s)) (last (string-split s #\.)))
        (begin (event)
          (loop (read-line)))))))

(define (log-in prefix event event-alist)
  (stderr "<external>.~a~a -> sut.~a~a\n" prefix event prefix event)
  (when (not relaxed?)
    (consume-synchronous-out-events prefix event event-alist)
    (stderr "<external>.~a~a -> sut.~a~a\n" prefix "return" prefix "return"))
  #f)

(define (log-out prefix event event-alist)
  (stderr "<external>.~a~a <- sut.~a~a\n" prefix event prefix event)
  #f)

(define (type-helper value type)
  (cond ((eq? type 'int) (string->number value))
    ((eq? type 'bool) (string=? value "true"))
    (else (string->symbol value))))

(define (log-valued prefix event event-alist)
  (stderr "<external>.~a~a -> sut.~a~a\n" prefix event prefix event)
  (if (not relaxed?)
    (let ((s (consume-synchronous-out-events prefix event event-alist)))
      (if s
        (begin (stderr "<external>.~a~a <- sut.~a~a\n" prefix s prefix s)
          s)
        0))
    0))

(define %calling-context (make <calling_context>))

(define (fill-event-alist o)
  (let* ((dzn-i 0)
      (flush? (find (cut string=? <> "--flush") (command-line)))
      (c (make <dzn:component> #:locator (.locator o) #:flushes? flush?))
      (e `(
          (p.hello . ,(lambda _ (apply (.hello (.in (.p o))) (list %calling-context 0))))
          (p.bye . ,(lambda _ (apply (.bye (.in (.p o))) (list))))
          (a.ack . ,(lambda _ (apply (.ack (.out (.a o))) (list %calling-context 0)))))))
    (slot-set! (slot-ref (slot-ref o 'p) 'in) 'name "p")
    (slot-set! (slot-ref (slot-ref o 'p) 'out) 'name "p")
    (when flush?
      (slot-set! (slot-ref (slot-ref o 'p) 'out) 'self c)
      (slot-set! (slot-ref (slot-ref o 'p) 'out) 'name "<external>.p"))
    (set! (.world (.out (.p o)))
      (lambda _ (log-out "p." "world" e)))
    e))

(define (main . args)
  (let* ((print-illegal (lambda () (stderr "illegal\n") (exit 0)))
      (locator (make <dzn:locator>))
      (runtime (make <dzn:runtime> #:illegal print-illegal))
      (pump (make <dzn:pump>))
      (locator (dzn:set! locator runtime))
      (locator (dzn:set! locator pump))
      (sut (make <async_calling_context> #:locator locator #:name "sut"))
      (event-alist (fill-event-alist sut)))
    (define (next-event)
      (let ((s (read-line)))
        (when (getenv "PUMP_DEBUG") (format (current-error-port) "next-event s=~a\n" s))
        (if (eof-object? s) s
          (assoc-ref event-alist (string->symbol s)))))
    (let loop ()
      (when (getenv "PUMP_DEBUG") (format (current-error-port) "main loop\n"))
      (let ((event (next-event)))
        (unless (eof-object? event)
          (when event
            (dzn:pump pump event next-event))
          (loop))))
    (dzn:finalize pump)))
;;; version: 2.12.0.rc1.3-4fcb88
