;;; Dezyne --- Dezyne command line tools
;;;
;;; Copyright © 2021, 2022, 2023, 2024 Janneke Nieuwenhuizen <janneke@gnu.org>
;;; Copyright © 2022, 2023, 2024 Rutger van Beusekom <rutger@dezyne.org>
;;; Copyright © 2023 Paul Hoogendijk <paul@dezyne.org>
;;;
;;; This file is part of Dezyne.
;;;
;;; Dezyne is free software: you can redistribute it and/or modify it
;;; under the terms of the GNU Affero General Public License as
;;; published by the Free Software Foundation, either version 3 of the
;;; License, or (at your option) any later version.
;;;
;;; Dezyne is distributed in the hope that it will be useful, but
;;; WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;; Affero General Public License for more details.
;;;
;;; You should have received a copy of the GNU Affero General Public
;;; License along with Dezyne.  If not, see <http://www.gnu.org/licenses/>.
;;;
;;; Commentary:
;;;
;;; Code:

(define-module (dzn code scmackerel makreel)
  #:use-module (srfi srfi-1)
  #:use-module (srfi srfi-26)
  #:use-module (srfi srfi-71)
  #:use-module (srfi srfi-9 gnu)

  #:use-module (ice-9 match)
  #:use-module (ice-9 rdelim)

  #:use-module (scmackerel processes)
  #:use-module (scmackerel records)

  #:use-module (dzn ast goops)
  #:use-module (dzn ast)
  #:use-module (dzn ast lookup)
  #:use-module (dzn code)
  #:use-module (dzn code goops)
  #:use-module (dzn code scmackerel code)
  #:use-module (dzn code language dzn)
  #:use-module (dzn code language makreel)
  #:use-module (dzn config)
  #:use-module (dzn misc)

  #:declarative? #f

  #:export (root->scmackerel
            scmackerel:display))

(define-record-type* <sm:tag>
  sm:tag make-sm:tag
  sm:tag?
  (component? sm:tag-component?)
  (line sm:tag-line)
  (column sm:tag-colomn)
  (statement sm:tag-statement))

(define (print-sm:tag tag port)
  "Write a concise representation of TAG to PORT."
  (match tag
    (($ <sm:tag> component? line column statement)
     (if (and (not component?) (%no-tags-interface?))
         (simple-format port "~a" statement)
         (simple-format port "tag (~a, ~a) . ~a" line column statement)))))

(set-record-type-printer! <sm:tag> print-sm:tag)

(define (dummy-tag-process)
  ;; FIXME: Cater for sm:tag being used as a stealth ACT, possibly
  ;; failing to declare TAG as ACT.
  (sm:process
    (name "dzn_dummy_tag_process")
    (statement
     (sm:invoke %tag-action (list 0 0)))))

;;;
;;; Helpers.
;;;
(define (makreel:caption str)
  (let* ((width (floor/ (- 80 (+ 2 (string-length str))) 2))
         (pad (make-string width #\%)))
    (string-append pad " " str " " pad)))

(define (makreel:generated-comment o)
  (sm:comment* (simple-format #f "%% Generated by dzn code from ~a\n"
                              (ast:source-file o))))

(define (makreel:version-comment)
  (sm:comment* (simple-format #f "%% version ~a\n" %package-version)))

(define (file-comments file-name)
  (let ((dir (string-append %template-dir "/makreel")))
    (list
     (sm:comment* (makreel:caption file-name))
     (sm:comment*
      (with-input-from-file (string-append dir "/" file-name)
        read-string)))))

(define* (model-prefix str #:optional model)
  (match model
    (#f
     (string-append (%model-name) str))
    (($ <port>) ;; HMM?
     (port-prefix str model))
    (_
     (string-append (makreel:full-name model) str))))

(define* (port-prefix str #:optional port)
  ;; XXX FIXME: port-constructor?
  (string-append (or (and=> port .name) (%port-name)) "port_" str))

(define-method (variable-prefix str (o <variable>))
  (string-append (.name o) str))

(define-method (variable-prefix str (o <formal>))
  (string-append (.name o) str))

(define-method (statement->process-name (o <behavior>))
  (model-prefix "behavior"))

(define-method (statement->process-name (o <ast>))
  (model-prefix (makreel:process-identifier o)))

(define-method (makreel:type->string (o <type>))
  (match o
    (($ <bool>) "Bool")
    (($ <int>) "Int")
    (($ <subint>) "Int")
    (($ <enum>) (makreel:enum-type o))
    (($ <void>) "Void")))

(define-method (makreel:reply-type (o <type>))
  (match o
    (($ <bool>) %sm:bool)
    (($ <int>) %sm:int)
    (($ <subint>) %sm:int)
    (($ <enum>) (enum->scmackerel o))))

(define-method (enum->scmackerel (o <enum>))
  (sm:type (name (makreel:enum-type o))
           (entities (map
                      (cute string-append (makreel:full-name o) <>)
                      (ast:field* o)))))

(define-method (makreel:reply-type (o <action>))
  (makreel:reply-type (ast:type o)))

(define-method (makreel:return-variable (o <ast>))
  (let* ((type (ast:type o))
         (variable (match type
                     (($ <bool>)
                      (make <variable>
                        #:name "return_value"
                        #:type.name (.name ast:bool)
                        #:expression (make <literal> #:value "return_value")))
                     (($ <enum>)
                      (make <variable>
                        #:name "return_value"
                        #:type.name (.name type)
                        #:expression (make <literal> #:value "return_value")))
                     (($ <subint>)
                      (make <variable>
                        #:name "return_value"
                        #:type.name (.name ast:int)
                        #:expression (make <literal> #:value "return_value")))
                     (_ (pke "XXX TODO makreel:return-variable for" type)))))
    (clone variable #:parent (or (.parent type) (.parent o)))))

(define-method (makreel:process-formals (o <ast>))
  (let* ((parameters (makreel:process-parameters o))
         (continuations calls (makreel:call-continuation* o))
         (return? (member o continuations ast:eq?))
         (return? (list-index (cute ast:eq? <> o) continuations))
         (call (and=> return? (cute list-ref calls <>)))
         (return? (and call
                       (not (is-a? (ast:type call) <void>))))
         (parameters (if (not return?) parameters
                         (append parameters
                                 (list (makreel:return-variable call)))))
         (parameters (if (not (and (is-a? o <action-reply>)
                                   (.variable.name o)))
                         parameters
                         (filter (compose not
                                          (cute equal? (.variable.name o) <>)
                                          .name)
                                 parameters)))
         (model (ast:parent o <model>)))
    (map (match-lambda
           ((and ($ <variable>) variable)
            (sm:formal (type (makreel:type->string (.type variable)))
                       (name (.name variable))))
           ((and ($ <formal>) variable)
            (sm:formal (type (makreel:type->string (.type variable)))
                       (name (.name variable))))
           ((and ($ <shared-variable>) variable)
            (sm:formal (type (makreel:type->string (.type variable)))
                       (name (makreel:full-name variable))))
           ((and ($ <bool>) type)
            (sm:formal (type (makreel:type->string type))
                       (name "return_value")))
           ((and ($ <enum>) type)
            (sm:formal (type (makreel:type->string type))
                       (name "return_value")))
           ((and ($ <subint>) type)
            (sm:formal (type (makreel:type->string type))
                       (name "return_value")))
           ((and ($ <stack>) stack)
            (sm:formal (type (model-prefix "stack" model))
                       (name "s"))))
         parameters)))

(define-method (makreel:shared-process-arguments (o <action>))
  (let* ((port (.port o))
         (behavior (ast:parent o <behavior>))
         (shared (filter (compose (cute ast:eq? <> port) .port)
                         (ast:shared* behavior)))
         (interface (.type port)))
    (map (lambda (s)
           (simple-format #f "~a=~a (~a)"
                          ;; FIXME ticked sufiix
                          (port-prefix (.name s) port)
                          ;; FIXME ticked sufiix
                          (model-prefix (.name s) interface)
                          (port-prefix "s" port)))
         shared)))

(define-method (makreel:shared-process-arguments (o <action-reply>))
  (makreel:shared-process-arguments (.action o)))

(define-method (makreel:enum-type (o <enum>))
  (string-append (makreel:full-name o) "enum"))

(define-method (makreel:return->value (o <return>))
  (let* ((model (ast:parent o <model>))
         (expression (.expression o))
         (type (ast:type expression)))
    (match type
      (($ <void>)
       (model-prefix "void_return (void)" model))
      (($ <bool>)
       (let ((expression (makreel:ast->expression expression)))
         (model-prefix
          (simple-format #f "bool_return (~a)" expression)
          model)))
      (($ <enum>)
       (let ((expression (makreel:ast->expression expression)))
         (model-prefix
          (simple-format #f "~a_return (~a)"
                         (makreel:type->string type)
                         expression)
          model)))
      ((or ($ <int>) ($ <subint>))
       (let ((expression (makreel:ast->expression expression)))
         (model-prefix
          (simple-format #f "int_return (~a)" expression)
          model)))
      (_ (pke "XXX TODO makreel:return->value for type"  type)
         "TODO"))))

(define-method (makreel:type->range-expression (o <subint>) name)
  (let* ((range (.range o))
         (from (.from range))
         (to (.to range))
         (i (sm:literal* name))
         (left (sm:less-equal* from i))
         (right (sm:less-equal* i to))
         (and (sm:and* left right))
         (expression (sm:group* and)))
    expression))

(define-method (makreel:type->out-of-range-processes (o <type>) name)
  '())

(define-method (makreel:type->out-of-range-processes (o <subint>) name)
  (if (equal? name "return_value") '()
      (list
       (sm:if* (sm:not* (makreel:type->range-expression o name))
               (sm:sequence* %range-error-action %sm:delta)
               "tau"))))

(define (variable->current? o)
  (let ((select (variable-prefix "select" o))
        (state (variable-prefix "state" o)))
    (sm:group* (sm:or* (sm:not* (sm:in* select "select (context)"))
                       (sm:equal* (simple-format #f "~a (current)" state)
                                  (simple-format #f "~a (state (context))" state))))))

(define (merge-scmackerels o result)
  (sm:mcrl2 (inherit result)
            (equations
             (append (sm:mcrl2-equations result)
                     (sm:mcrl2-equations o)))
            (types
             (append (sm:mcrl2-types result)
                     (sm:mcrl2-types o)))
            (processes
             (append (sm:mcrl2-processes result)
                     (sm:mcrl2-processes o)))))


;;;
;;; Actions and types.
;;;
(define %port-name (make-parameter #f))

(define %void (sm:type (name "Void") (entities '("void"))))

(define %defer-context-type
  (sm:type (name "DeferContext")
           (entities
            (list
             (sm:entity (name "defer_context")
                        (formals
                         (list (sm:formal (type "State") (name "state"))
                               (sm:formal (type "Set (Selection)") (name "select"))
                               (sm:formal (type "Locals") (name"locals")))))))))

(define %declarative-illegal-action (sm:action (prefix "declarative_illegal")))
(define %illegal-action (sm:action (prefix "illegal")))

(define (%compliant-action o)
  (sm:action (prefix (if o (port-prefix "compliance" o) "compliance"))))
(define %constrained-legal-action (sm:action (prefix "constrained_legal")))
(define %constrained-illegal-action (sm:action (prefix "constrained_illegal")))
(define %defer-end-action (sm:action (prefix "defer_end")))
(define %defer-qin-action (sm:action (prefix "defer_qin")
                                     (event
                                      (sm:event (type %defer-context-type)
                                                (constructor "defer_context")
                                                (arguments '("state_vector"))))))
(define %queue-empty-action (sm:action (prefix "queue_empty")))
(define %queue-not-empty-action (sm:action (prefix "queue_not_empty")))
(define %queue-full-action (sm:action (prefix "queue_full")))
(define %missing-reply-action (sm:action (prefix "missing_reply")))
(define %non-compliant-action (sm:action (prefix "non_compliance")))
(define %second-reply-action (sm:action (prefix "second_reply")))
(define %range-error-action (sm:action (prefix "range_error")))
(define %recurse-action (sm:action (prefix "recurse")))
(define %return-action (sm:action (prefix "return")))
(define %tau-void-action (sm:action (prefix "tau_void")))
(define %tag-action (sm:action (prefix "tag")
                               (event (sm:event (type (list %sm:nat %sm:nat))))))

(define %no-port-predicate (sm:predicate (type "no_port")
                                         (name "no_port_is")))

(define %actions
  (let ((cache '()))
    (lambda (o)
      (or (assq-ref cache o)
          (let ((result
                 (match o
                   (($ <interface>)
                    (sm:type (name (model-prefix "actions" o))
                             (entities
                              (append ;; TODO split actions from triggers
                               (map
                                (cute string-append (model-prefix "in'") <>)
                                (map .name (ast:in-event* o)))
                               (map
                                (cute string-append (model-prefix "out'") <>)
                                (map .name (ast:out-event* o))))))))))
            (set! cache (acons o result cache))
            result)))))

(define %modeling
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:type (name (model-prefix "modeling" o))
                          (entities
                           (map model-prefix '("optional" "inevitable"))))))))
         (set! cache (acons o result cache))
         result)))))

(define %nil-predicate
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:predicate (type (model-prefix "nil" o))
                               (name (model-prefix "nil_is" o))))
                (($ <port>)
                 (sm:predicate (name (port-prefix "nil" o))
                               (type (port-prefix "nil_is" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %replies
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (let* ((in-types (map ast:type (ast:in-event* o)))
                        (reply-enums (filter (is? <enum>) in-types))
                        (reply-enums (delete-duplicates reply-enums ast:eq?))
                        (void? (find (is? <void>) in-types))
                        (bool? (find (is? <bool>) in-types))
                        (int? (find (is? <int>) in-types)))
                   (sm:type (name (model-prefix "replies" o))
                            (entities
                             `(,(%nil-predicate o)
                               ,@(if (not void?) '()
                                     (list
                                      (sm:entity
                                       (name (model-prefix "Void" o))
                                       (formals
                                        (list (sm:formal (type %void)
                                                         (name "value")))))))
                               ,@(if (not bool?) '()
                                     (list
                                      (sm:entity
                                       (name (model-prefix "Bool" o))
                                       (formals
                                        (list (sm:formal (type %sm:bool)
                                                         (name "value")))))))
                               ,@(if (not int?) '()
                                     (list
                                      (sm:entity
                                       (name (model-prefix "Int" o))
                                       (formals
                                        (list (sm:formal (type %sm:int)
                                                         (name "value")))))))
                               ,@(map
                                  (lambda (e)
                                    (sm:entity (name (makreel:full-name e))
                                               (formals
                                                (list
                                                 (sm:formal (type (makreel:enum-type e))
                                                            (name "value"))))))
                                  reply-enums)))))))))
         (set! cache (acons o result cache))
         result)))))

(define %return-type
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       ;; XXX c&p from %replies
       (let* ((functions (ast:function* (.behavior o)))
              (return-types (map ast:type functions)))
         (and
          (pair? return-types)
          (let* ((return-enums (filter (is? <enum>) return-types))
                 (return-enums (delete-duplicates return-enums ast:eq?))
                 (void? (find (is? <void>) return-types))
                 (bool? (find (is? <bool>) return-types))
                 (int? (find (is? <int>) return-types))
                 (result
                  (sm:type (name (model-prefix "return_type" o))
                           (entities
                            `(,@(if (not void?) '()
                                    (list
                                     (sm:entity
                                      (name (model-prefix "void_return" o))
                                      (formals
                                       (list
                                        (sm:formal (type %void)
                                                   (name "void_value")))))))
                              ,@(if (not bool?) '()
                                    (list
                                     (sm:entity
                                      (name (model-prefix "bool_return" o))
                                      (formals
                                       (list
                                        (sm:formal (type %sm:bool)
                                                   (name "bool_value")))))))
                              ,@(if (not int?) '()
                                    (list
                                     (sm:entity
                                      (name (model-prefix "int_return" o))
                                      (formals
                                       (list
                                        (sm:formal (type %sm:int)
                                                   (name "int_value")))))))
                              ,@(map
                                 (lambda (e)
                                   (let ((enum (makreel:enum-type e)))
                                     (sm:entity
                                      (name (model-prefix
                                             (string-append enum "_return")
                                             o))
                                      (formals
                                       (list (sm:formal
                                              (type enum)
                                              (name (string-append enum "_value"))))))))
                                 return-enums))))))
            (set! cache (acons o result cache))
            result)))))))

(define %stack-type
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let* ((returns (makreel:call-continuation* o))
              (returns (delete-duplicates returns ast:eq?))
              (result
               (sm:type (name (model-prefix "stack" o))
                        (entities
                         (cons*
                          (sm:entity (name (model-prefix "stack_empty" o)))
                          (map
                           (lambda (r)
                             (let* ((process (statement->process-name r))
                                    (locals (makreel:locals r)))
                               (sm:entity
                                (name (string-append process "_frame"))
                                (formals
                                 `(,(sm:formal (type (model-prefix "returns" o))
                                               (name "return"))
                                   ,@(map
                                      (lambda (l)
                                        (let* ((type (.type l))
                                               (type (makreel:type->string type))
                                               (name (variable-prefix "local" l)))
                                          (sm:formal (type type)
                                                     (name name))))
                                      locals)
                                   ,(sm:formal (type (model-prefix "stack" o))
                                               (name "stack")))))))
                           returns))))))
         (set! cache (acons o result cache))
         result)))))

(define %return-predicate
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (let ((process (statement->process-name o)))
                (sm:predicate (type (string-append process "'return"))
                              (name (string-append "is'" process "'return"))))))
         (set! cache (acons o result cache))
         result)))))

(define %returns-type
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let* ((returns (makreel:call-continuation* o))
              (returns (delete-duplicates returns ast:eq?)))
         (and (pair? returns)
              (let ((result
                     (sm:type (name (model-prefix "returns" o))
                              (entities (map %return-predicate returns)))))
                (set! cache (acons o result cache))
                result)))))))

(define %reply-values
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <component>)
                 (let* ((ports (ast:provides-port* o))
                        (formals
                         (map
                          (lambda (p)
                            (let* ((interface (.type p))
                                   (type (model-prefix "replies" interface))
                                   (name (port-prefix "port" p)))
                              (sm:formal (type type) (name name))))
                          ports)))
                   (sm:type (name "reply_values")
                            (entities (list
                                       (sm:entity
                                        (name "replies")
                                        (formals formals))))))))))
         (set! cache (acons o result cache))
         result)))))

(define %events
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:type (name (model-prefix "events" o))
                          (entities
                           (list (sm:entity (name (model-prefix "action" o))
                                            (formals
                                             (list (sm:formal (type (%actions o))
                                                              (name "value")))))
                                 (sm:entity (name (model-prefix "reply" o))
                                            (formals
                                             (list (sm:formal (type (%replies o))
                                                              (name "value"))))))))))))
         (set! cache (acons o result cache))
         result)))))

(define %state
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (let ((members (ast:variable* o)))
                   (sm:type (name (model-prefix "State" o))
                            (entities
                             (if (null? members) (list (model-prefix "Void" o))
                                 (let ((formals
                                        (map
                                         (lambda (v)
                                           (let* ((type (.type v))
                                                  (type (makreel:type->string type))
                                                  (name (.name v))
                                                  (name (model-prefix name o)))
                                             (sm:formal (type type) (name name))))
                                         members)))
                                   (list
                                    (sm:entity (name (model-prefix "variables" o))
                                               (formals formals)))))))))
                (($ <component>)
                 ;; FIXME C&P
                 (let* ((members (ast:member* o))
                        (members (filter (negate (is? <shared-variable>)) members))
                        (defers (makreel:defer* o))
                        (arguments? (or (pair? members)
                                        (pair? defers))))
                   (sm:type (name "State")
                            (entities
                             (if (not arguments?) (list (model-prefix "Void" o))
                                 (let ((formals
                                        (map
                                         (lambda (v)
                                           (let* ((type (.type v))
                                                  (type (makreel:type->string type))
                                                  (name (variable-prefix "state" v)))
                                             (sm:formal (type type) (name name))))
                                         members)))
                                   (list (sm:entity (name "state_vector")
                                                    (formals
                                                     (and (pair? formals)
                                                          formals)))))))))))))
         (set! cache (acons o result cache))
         result)))))

(define %locals-predicate
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (let ((id (makreel:process-identifier o))
                    (locals (makreel:locals o)))
                (sm:predicate
                 (type
                  (if (null? locals) (simple-format #f "locals~a" id)
                      (let ((formals
                             (map
                              (lambda (l)
                                (let* ((type (.type l))
                                       (type (makreel:type->string type))
                                       (name (variable-prefix "locals" l)))
                                  (sm:formal (type type)
                                             (name name))))
                              locals)))
                        (sm:entity
                         (name (simple-format #f "locals~a" id))
                         (formals formals)))))
                 (name (simple-format #f "islocals~a" id))))))
         (set! cache (acons o result cache))
         result)))))

(define %locals
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (let* ((defers (makreel:defer* o))
                     (statements (map .statement defers)))
                (sm:type (name "Locals")
                         (entities
                          (cons* "empty"
                                 (map %locals-predicate statements)))))))
         (set! cache (acons o result cache))
         result)))))

(define %defer-qout-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (sm:action (prefix "defer_qout")
                         (event
                          (sm:event (type (%locals o))
                                    (arguments '("state_vector")))))))
         (set! cache (acons o result cache))
         result)))))

(define %defer-skip-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (let* ((members (filter (negate (is? <shared-variable>)) (ast:variable* o)))
                     (names (map makreel:ast->expression members)))
                (sm:action (prefix "defer_skip")
                           (event
                            (sm:event (type (%state o))
                                      (constructor "state_vector")
                                      (arguments (and (pair? members) names))))))))
         (set! cache (acons o result cache))
         result)))))

(define %in-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "in" o))
                            (event
                             (sm:event (constructor (model-prefix "action" o))
                                       (type (%events o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%in-action (.type o)))
                  (prefix (port-prefix "in" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %out-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "out" o))
                            (event
                             (sm:event (constructor (model-prefix "action" o))
                                       (type (%events o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%out-action (.type o)))
                  (prefix (port-prefix "out" o)))))))
         (set! cache (acons o result cache))
         result)))))

;;TODO consider fixing duplication with %out-action
(define %qin-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <port>)
                 (sm:action
                  (inherit (%out-action (.type o)))
                  (prefix (port-prefix "qin" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %qout-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <port>)
                 (sm:action
                  (inherit (%out-action (.type o)))
                  (prefix (port-prefix "qout" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %internal-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "internal" o))
                            (event
                             (sm:event (constructor (model-prefix "modeling" o))
                                       (type (%modeling o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%internal-action (.type o)))
                  (prefix (port-prefix "internal" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %flush-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result (sm:action (prefix (model-prefix "flush" o)))))
         (set! cache (acons o result cache))
         result)))))

(define %blocking-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result (sm:action (prefix (model-prefix "blocking" o)))))
         (set! cache (acons o result cache))
         result)))))

(define %state-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "state" o))
                            (event
                             (sm:event (constructor (model-prefix "variables" o))
                                       (type (%state o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%state-action (.type o)))
                  (prefix (port-prefix "state" o))))
                (($ <component>)
                 (sm:action (prefix (model-prefix "state" o))
                            (event
                             (sm:event (constructor (model-prefix "variables" o))
                                       (type (%state o)))))))))
         (set! cache (acons o result cache))
         result)))))

(define-method (sum-state-action (port <port>))
  (sm:sum (type (%state (.type port)))
          (var (port-prefix "s" port))
          (statement
           (sm:invoke (%state-action port) var))))

(define-method (sum-state-action (port <port>) continuation)
  (sm:sum (type (%state (.type port)))
          (var (port-prefix "s" port))
          (statement
           (sm:sequence*
            (sm:invoke (%state-action port) var)
            continuation))))

(define %switch-context-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "switch_context" o))))
                (($ <port>)
                 (sm:action
                  (inherit (%switch-context-action (.type o)))
                  (prefix (port-prefix "switch_context" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %end-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result (sm:action (prefix (model-prefix "end" o)))))
         (set! cache (acons o result cache))
         result)))))

(define %end-reordered-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result (sm:action (prefix (model-prefix "end'reordered" o)))))
         (set! cache (acons o result cache))
         result)))))

(define %reorder-end-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result (sm:action (prefix (model-prefix "reorder_end" o)))))
         (set! cache (acons o result cache))
         result)))))

(define %reply-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "reply" o))
                            (event
                             (sm:event (constructor (model-prefix "reply" o))
                                       (type (%replies o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%reply-action (.type o)))
                  (prefix (port-prefix "reply" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %reply-reordered-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result (sm:action
                      (inherit (%reply-action o))
                      (prefix
                       (model-prefix "reply'reordered" o)))))
         (set! cache (acons o result cache))
         result)))))

(define %reply-nil-predicate
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let* ((interface (or (as o <interface>)
                             (.type o)))
              (result (model-prefix "nil" interface)))
         (set! cache (acons o result cache))
         result)))))

(define %port-queue-full-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <port>)
                 (sm:action (prefix (model-prefix "queue_full" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %port-predicate
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                ((? string?)
                 o)
                (($ <interface>)
                 #f)
                (($ <port>)
                 (sm:predicate (type (port-prefix "port" o))
                               (name (port-prefix "is" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %tau-event-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "tau_event" o))
                            (event (sm:event (type (%events o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%tau-event-action (.type o)))
                  (prefix (port-prefix "tau_event" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %tau-modeling-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "tau_modeling" o))
                            (event (sm:event (type (%modeling o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%tau-modeling-action (.type o)))
                  (prefix (port-prefix "tau_modeling" o)))))))
         (set! cache (acons o result cache))
         result)))))

(define %tau-reply-action
  (let ((cache '()))
    (lambda (o)
      (or
       (assq-ref cache o)
       (let ((result
              (match o
                (($ <interface>)
                 (sm:action (prefix (model-prefix "tau_reply" o))
                            (event (sm:event (type (%replies o))))))
                (($ <port>)
                 (sm:action
                  (inherit (%tau-reply-action (.type o)))
                  (prefix (port-prefix "tau_reply" o)))))))
         (set! cache (acons o result cache))
         result)))))


;;;
;;; makreel:ast->expression.
;;;

(define-method (makreel:ast->expression (o <top>))
  (ast->expression o))

(define-method (makreel:ast->expression (o <not>))
  (sm:not* (makreel:ast->expression (.expression o))))

(define-method (makreel:ast->expression (o <binary>))
  (sm:expression
   (operator (operator->string o))
   (operands (list (makreel:ast->expression (.left o))
                   (makreel:ast->expression (.right o))))))

(define-method (makreel:ast->expression (o <group>))
  (sm:group* (makreel:ast->expression (.expression o))))

(define-method (makreel:ast->expression (o <shared-var>))
  (string-append (.port.name o) "port_" (.name o)))

(define-method (makreel:ast->expression (o <shared-variable>))
  (string-append (.port.name o) "port_" (.name o)))

(define-method (makreel:ast->expression (o <shared-field-test>))
  (let* ((variable (.variable o))
         (type (.type variable))
         (type-name (make <scope.name> #:ids (ast:full-name type)))
         (enum-literal (make <enum-literal>
                         #:type.name type-name
                         #:field (.field o)))
         (enum-literal (clone enum-literal #:parent o))
         (name (.name variable))
         (port-name (.port.name o))
         (var (make <shared-var> #:name name #:port.name port-name))
         (expression (make <equal>
                       #:left var
                       #:right enum-literal)))
    (makreel:ast->expression expression)))


;;;
;;; Ast->action.
;;;
(define-method (ast->action (o <trigger>))
  (let* ((model (ast:parent o <model>))
         (interface (if (is-a? model <interface>) model
                        (.type (.port o))))
         (model (if (is-a? model <interface>) model
                    (.port o)))
         (event-name (format #f "~a'~a" (ast:direction o) (.event.name o))))
    (cond
     ((ast:modeling? o)
      (sm:invoke (%internal-action model) (model-prefix (.event.name o) model)))
     (else
      (sm:invoke (if (ast:in? o) (%in-action model) (%out-action model))
                 (model-prefix event-name interface)
                 #:keep-constructor? #t)))))

(define-method (ast->action (o <action>))
  (let* ((model (ast:parent o <model>))
         (interface (if (is-a? model <interface>) model
                        (.type (.port o))))
         (model (if (is-a? model <interface>) model
                    (.port o)))
         (event-name (format #f "~a'~a" (ast:direction o) (.event.name o))))
    (sm:invoke (if (ast:in? o) (%in-action model) (%out-action model))
               (model-prefix event-name interface)
               #:keep-constructor? #t)))

(define-method (ast->action (o <reply>))
  (let* ((model (ast:parent o <model>))
         (model (or (as model <interface>)
                    (.port o)
                    (let* ((on (ast:parent o <on>))
                           (triggers (and=> on ast:trigger*))
                           (trigger (and=> triggers car)))
                      (and trigger
                           (ast:provides? trigger)
                           (.port trigger)))
                    (ast:provides-port model)))
         (interface (if (is-a? model <interface>) model
                        (.type model)))
         (expression (.expression o))
         (type (ast:type expression))
         (prefix (model-prefix "reply" model))
         (arguments (list (makreel:ast->expression expression))))
    (match type
      (($ <bool>)
       (sm:action (prefix prefix)
                  (event
                   (sm:event (constructor (model-prefix "Bool" interface))
                             (type (%replies interface))
                             (arguments arguments)))))
      (($ <enum>)
       (sm:action (prefix prefix)
                  (event
                   (sm:event (constructor (makreel:full-name type))
                             (type (%replies interface))
                             (arguments arguments)))))
      ((or ($ <int>) ($ <subint>))
       (sm:action (prefix prefix)
                  (event
                   (sm:event (constructor (model-prefix "Int" interface))
                             (type (%replies interface))
                             (arguments arguments)))))
      (($ <void>)
       (sm:action (prefix prefix)
                  (event
                   (sm:event (constructor (model-prefix "Void" interface))
                             (type (%replies interface))
                             (arguments '("void")))))))))


;;;
;;; Ast->process.
;;;
(define-method (ast->process (model <interface>) (o <behavior>) (next <ast>))
  (let* ((members (ast:member* o))
         (members (map .name members)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:union*
        (sm:goto (name (statement->process-name next))
                 (arguments (makreel:process-parens o)))
        (sm:sequence*
         (sm:invoke (%state-action model)
                    (if (null? members) (model-prefix "Void" model)
                        (string-join members ", "))
                    #:keep-constructor? (pair? members))
         (sm:goto (name name)
                  (arguments (makreel:process-parens o)))))))))

(define-method (ast->process (model <component>) (o <behavior>) (next <ast>))
  (let* ((shared (ast:shared* o))
         (ports (filter (negate ast:external?) (ast:port* model)))
         (requires (ast:requires-port* model))
         (defers (makreel:defer* o))
         (defers
           (if (null? defers) defers
               (list
                (sm:sum
                 (type (%locals model))
                 (statement
                  (sm:sequence*
                   (sm:invoke (%defer-qout-action model))
                   (sm:union*
                    (map
                     (lambda (defer)
                       (let* ((statement (.statement defer))
                              (locals (makreel:locals statement)))
                         (sm:if*
                          (sm:invoke (%locals-predicate statement))
                          (sm:goto
                           (name (statement->process-name
                                  (.statement defer)))
                           (arguments
                            (append
                             (map
                              (lambda (variable)
                                (simple-format
                                 #f
                                 "~a=~a (i)"
                                 (.name variable)
                                 (variable-prefix "locals" variable)))
                              locals)))))))
                     defers)))))))))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:union*
        `(,@(map
             (lambda (port)
               (let ((shared (filter (compose (cute ast:eq? port <>) .port)
                                     shared)))
                 (sm:sequence*
                  `(,(sum-state-action port)
                    ,(sm:goto (name (statement->process-name o))
                              (arguments
                               (map (lambda (shared-var)
                                      (let* ((interface (.type port)))
                                        (simple-format
                                         #f "~a=~a (~a)"
                                         (makreel:full-name shared-var)
                                         (model-prefix (ast:name shared-var)
                                                       interface)
                                         (port-prefix "s" port))))
                                    shared)))))))
             ports)
          ,@(map (lambda (port)
                   (sm:sequence*
                    (%flush-action port)
                    (sm:goto (name (statement->process-name o))
                             (arguments (makreel:process-parens o)))))
                 (lset-difference ast:eq? requires (map .port shared)))
          ,@(map (lambda (port)
                   (let ((shared (filter (compose (cute ast:eq? port <>) .port)
                                         shared)))
                    (sm:sequence*
                     `(,(%flush-action port)
                       ,(sum-state-action port)
                       ,(sm:goto (name (statement->process-name o))
                                 (arguments
                                  (map
                                   (lambda (v)
                                     (let* ((p (.port v))
                                            (interface (.type p)))
                                       (simple-format
                                        #f "~a=~a (~a)"
                                        (makreel:full-name v)
                                        (model-prefix (ast:name v) interface)
                                        (port-prefix "s" p))))
                                   shared)))))))
                 (lset-intersection ast:eq? (map .port shared) requires))
          ,@defers
          ,(sm:goto (name (statement->process-name next))
                    (arguments (makreel:process-parens next)))))))))

(define-method (ast->process (model <model>) (o <declarative-compound>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:goto (name (statement->process-name next))
              (arguments (makreel:process-parens next))))))

(define-method (ast->process (model <model>) (o <declarative-compound>) (next <list>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:union*
      (map (lambda (next)
             (sm:goto (name (statement->process-name next))
                      (arguments (makreel:process-parens next))))
           next)))))

(define-method (ast->process (model <model>) (o <compound>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:goto (name (statement->process-name next))
              (arguments (makreel:process-parens next))))))

(define-method (ast->process (model <model>) (o <skip>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:goto (name (statement->process-name next))
              (arguments (makreel:process-parens next))))))

(define-method (ast->process (model <model>) (o <guard>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement (sm:if* (makreel:ast->expression (.expression o))
                       (sm:goto (name (statement->process-name next))
                                (arguments (makreel:process-parens next)))))))

(define-method (ast->process (model <model>) (o <blocking>) (next <ast>))
  (let ((model (or (as model <interface>)
                   (let* ((on (ast:parent o <on>))
                          (trigger (car (ast:trigger* on))))
                     (and (ast:provides? trigger)
                          (.port trigger))))))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence* (%blocking-action model)
                     (sm:goto (name (statement->process-name next))
                              (arguments (makreel:process-parens next))))))))

(define-method (ast->process (interface <interface>) (o <on>) (next <ast>))
  (let ((trigger (car (ast:trigger* o))))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        (ast->action trigger)
        (sm:goto (name (statement->process-name next))
                 (arguments (makreel:process-parens next))))))))

(define-method (ast->process (model <component>) (o <on>) (next <ast>))
  (let* ((trigger (car (ast:trigger* o)))
         (port (.port trigger)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        `(,(ast->action trigger)
          ,(if (ast:in? trigger)
               (sm:union*
                (sm:sequence*
                 %constrained-legal-action
                 (sm:goto (name (statement->process-name next))
                          (arguments (makreel:process-parens next))))
                (sm:goto (name "Constrained_Illegal")))
               (if (not (find (compose (cute ast:eq? port <>) .port)
                              (ast:shared* model)))
                   (sm:goto (name (statement->process-name next))
                            (arguments (makreel:process-parens next)))
                   (sum-state-action
                    port
                    (sm:goto (name (statement->process-name next))
                             (arguments
                              (map
                               (lambda (v)
                                 (let* ((p (.port v))
                                        (interface (.type p)))
                                   (simple-format
                                    #f "~a=~a (~a)"
                                    (makreel:full-name v)
                                    (model-prefix (ast:name v) interface)
                                    (port-prefix "s" p))))
                               (filter (compose (cute ast:eq? port <>) .port)
                                       (ast:shared* model))))))))))))))

(define-method (ast->process (model <model>) (o <declarative-illegal>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:goto (name "Declarative_Illegal")))))

(define-method (ast->process (model <model>) (o <illegal>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:goto (name "Illegal")))))

(define-method (ast->process (model <model>) (o <tag>) (next <ast>))
  (let ((location (.location o)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:tag
        (component? (is-a? model <component>))
        (line (.line location))
        (column (.column location))
        (statement (sm:goto (name (statement->process-name next))
                            (arguments (makreel:process-parens next)))))))))

(define-method (ast->process (model <model>) (o <reply>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:sequence*
      (ast->action o)
      (sm:goto (name (statement->process-name next))
               (arguments (makreel:process-parens next)))))))

(define-method (assign->sum (model <model>) (o <action>) (variable <variable>))
  (let* ((port (.port o))
         (action-type (ast:type o))
         (variable-type (ast:type variable))
         (shared (ast:shared* model))
         (shared? (find (compose (cute ast:eq? <> port) .port) shared)))
    (sm:sum (type (makreel:reply-type o))
            (statement
             (let* ((reply (make <reply>
                             #:port.name (.name port)
                             #:expression o))
                    (reply (clone reply #:parent (.parent o)))
                    (reply (ast->action reply))
                    (reply (sm:invoke reply #:keep-constructor? #t)))
               (sm:sequence*
                `(,(if (not (is-a? action-type <subint>)) reply
                       (sm:sequence*
                        `(,(sm:if* (makreel:type->range-expression action-type "i")
                                   reply)
                          ,@(makreel:type->out-of-range-processes
                             variable-type "i"))))
                  ,@(if (or (ast:external? port)
                            (not (find (compose (cute ast:eq? port <>) .port)
                                       (ast:shared* model)))) '()
                                       `(,(sum-state-action port))))))))))

(define-method (assign->sum (model <model>) (o <action>) (variable <formal>))
  (assign->sum model o (make <variable>
                         #:name (.name variable)
                         #:type.name (.type.name variable))))

(define-method (ast->process (model <interface>) (o <state>) (next <ast>))
  (let* ((members (ast:member* model))
         (members (map .name members)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:union*
        (sm:goto (name (statement->process-name next))
                 (arguments (makreel:process-parens next)))
        (sm:sequence*
         (sm:invoke (%state-action model)
                    (if (null? members) (model-prefix "Void" model)
                        (string-join members ", "))
                    #:keep-constructor? (pair? members))
         (sm:goto (name name)
                  (arguments (makreel:process-parens o)))))))))

(define-method (ast->process (model <model>) (o <action>) (next <ast>))
  (let* ((port (.port o))
         (void-reply? (and (ast:in? o)
                           (is-a? (ast:type o) <void>))))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        `(,(ast->action o)
          ,@(if (makreel:switch-context? o)
                `(,(%switch-context-action port)
                  ,(sm:goto (name (statement->process-name next))
                            (arguments (makreel:process-parens o))))
                `(,@(if (not void-reply?) '()
                        (let* ((reply (make <reply> #:port.name (.name port)))
                               (reply (clone reply #:parent (.parent o))))
                          `(,(ast->action reply))))
                  ,(cond ((and=> port ast:external?)
                          (sm:goto (name (statement->process-name next))
                                   (arguments (makreel:process-parens next))))
                         ((find (compose (cute ast:eq? port <>) .port)
                                (ast:shared* model))
                          (sum-state-action
                           port
                           (sm:goto
                            (name (statement->process-name next))
                            (arguments
                             (cond ((null? (ast:shared* model))
                                    (makreel:process-parens next))
                                   (else
                                    (makreel:shared-process-arguments o)))))))
                         (else
                          (sm:goto (name (statement->process-name next))
                                   (arguments
                                    (makreel:process-parens next)))))))))))))

(define-method (ast->process (model <component>) (o <action-reply>) (next <ast>))
  (let* ((port (.port o))
         (requires (filter (negate (cute ast:eq? port <>))
                           (ast:requires-port* model)))
         (interface (and=> port .type))
         (void-reply? (and (ast:in? o)
                           (is-a? (ast:type o) <void>))))
    (sm:process
      (name (string-append (statement->process-name o)))
      (formals (makreel:process-formals o))
      (statement
       (sm:union*
        `(,(sm:sequence*
            `(,@(if (not (makreel:switch-context? o)) '()
                    `(,(%switch-context-action port)))
              ,@(if (not void-reply?)
                    `(,(assign->sum model (.action o) (.variable o)))
                    (let* ((reply (make <reply> #:port.name (.name port)))
                           (reply (clone reply #:parent (.parent o))))
                      `(,(ast->action reply)
                        ,@(if (not (find (compose (cute ast:eq? port <>) .port)
                                         (ast:shared* model))) '()
                                         `(,(sum-state-action port))))))
              ,(if (or (not (and (is-a? model <component>) (ast:in? o)))
                       (ast:external? port))
                   (sm:goto (name (statement->process-name next))
                            (arguments
                             (makreel:process-parens next)))
                   (sm:goto (name (statement->process-name next))
                            ;; FIXME kopie #3
                            (arguments
                             (let* ((variable (.variable.name o))
                                    (variables (makreel:variables-in-scope next))
                                    (variables
                                     (map (match-lambda
                                            ((and ($ <variable>) variable)
                                             (.name variable))
                                            ((and ($ <formal>) variable)
                                             (.name variable))
                                            ((and ($ <shared-variable>) variable)
                                             (makreel:full-name variable))
                                            ((and ($ <stack>) stack)
                                             "s"))
                                          variables))
                                    (assign? (member variable variables))
                                    (shared (ast:shared* model)))
                               (cond ((and (not (.variable o))
                                           (null? shared))
                                      (makreel:process-parens next))
                                     (else
                                      `(,@(makreel:shared-process-arguments (.action o))
                                        ,@(if (not (.variable.name o)) '()
                                              (list
                                               (sm:is*
                                                variable
                                                "i"))))))))))))
          ,@(if (not (and (ast:requires? port)
                          (find (compose (cute ast:eq? port <>) .port)
                                (ast:shared* model))))
                '()
                `(,(sum-state-action
                    port
                    (sm:goto (name name)
                             (arguments
                              (cond ((or (ast:out? o)
                                         (null? (ast:shared* model)))
                                     (makreel:process-parens next))
                                    (else
                                     (makreel:shared-process-arguments o))))))))
          ,@(map (lambda (port)
                   (sm:sequence*
                    (%flush-action port)
                    (sm:goto (name name)
                             (arguments (makreel:process-parens o)))))
                 requires)))))))

(define-method (ast->process (model <model>) (o <call>) (next <ast>))
  (let* ((function (.function o))
         (statement (.statement function))
         (arguments (ast:argument* o))
         (expressions (map makreel:ast->expression arguments))
         (formals (ast:formal* function))
         (formal-names (map .name formals))
         (types (map ast:type formals)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        `(,@(append-map makreel:type->out-of-range-processes
                        types expressions)
          ,@(if (not (.last? o)) '()
                (list %recurse-action))
          ,(sm:goto
            (name (statement->process-name statement))
            (arguments
             (append
              (map (lambda (f a) (sm:is* f a))
                   formal-names
                   (map makreel:ast->expression arguments))
              (list
               (if (and (.last? o) (ast:parent o <function>)) "s=s"
                   (let* ((return (car (ast:continuation* o)))
                          (locals (makreel:locals return))
                          (locals (map .name locals))
                          (stack-empty? (makreel:stack-empty? o)))
                     (simple-format
                      #f
                      "s=~a_frame (~a'return~a, ~a)"
                      (statement->process-name next)
                      (statement->process-name next)
                      (string-join locals ", " 'prefix)
                      (if (not stack-empty?) "s"
                          (model-prefix "stack_empty" model)))))))))))))))

(define-method (ast->process (model <model>) (o <return>) (next <ast>))
  (let* ((function (ast:parent o <function>))
         (type (ast:type function))
         (expression (.expression o))
         (expression (makreel:ast->expression expression)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        `(,@(makreel:type->out-of-range-processes type expression)
          ,%return-action
          ,(sm:goto
            (name (model-prefix "return" model))
            (arguments (list
                        (sm:is*
                         "return_value"
                         (makreel:return->value o)))))))))))

;;; FIXME: split into interface / component?
(define-method (ast->process (model <model>) (o <assign>) (next <ast>))
  (let* ((expression (.expression o))
         (action? (as expression <action>))
         (port (and=> action? .port))
         (expression (makreel:ast->expression expression))
         (type (and=> (.variable o) .type)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        `(,@(if (not action?)
                (makreel:type->out-of-range-processes type expression)
                `(,(ast->action action?)
                  ,@(if (not (makreel:switch-context? action?))
                        `(,(assign->sum model action? (.variable o)))
                        `(,(%switch-context-action port)))))
          ,(if (and action? (makreel:switch-context? action?))
               (sm:goto (name (statement->process-name next))
                        (arguments (makreel:process-parens next)))
               (sm:goto (name (statement->process-name next))
                        (arguments
                         (let* ((variable (.variable.name o))
                                (variables (makreel:variables-in-scope next))
                                (variables
                                 (map (match-lambda
                                        ((and ($ <variable>) variable)
                                         (.name variable))
                                        ((and ($ <formal>) variable)
                                         (.name variable))
                                        ((and ($ <shared-variable>) variable)
                                         (makreel:full-name variable))
                                        ((and ($ <stack>) stack)
                                         "s"))
                                      variables))
                                (assign? (member variable variables))
                                (shared (ast:shared* model)))
                           (cond ((and (or (not assign?)
                                           (and action?
                                                (ast:out? action?)))
                                       (null? shared))
                                  (makreel:process-parens next))
                                 (else
                                  `(,@(if (not action?) '()
                                          (makreel:shared-process-arguments action?))
                                    ,@(if (not assign?) '()
                                          (list
                                           (sm:is*
                                            variable
                                            (if action? "i"
                                                expression)))))))))))))))))

;;; FIXME: split into interface / component?
(define-method (ast->process (model <model>) (o <variable>) (next <ast>))
  (let* ((expression (.expression o))
         (action? (as expression <action>))
         (port (and=> action? .port))
         (expression (makreel:ast->expression expression))
         (type (.type o)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        `(,@(if (not action?)
                (makreel:type->out-of-range-processes type expression)
                `(,(ast->action action?)
                  ,@(if (not (makreel:switch-context? action?))
                        `(,(assign->sum model action? o))
                        `(,(%switch-context-action port)))))
          ,(if (and action? (makreel:switch-context? action?))
               (sm:goto (name (statement->process-name next))
                        (arguments (makreel:process-parens next)))
               (sm:goto (name (statement->process-name next))
                        (arguments
                         (let* ((variable (.name o))
                                (variables (makreel:variables-in-scope next))
                                (variables (map
                                            (match-lambda
                                              ((and ($ <variable>) variable)
                                               (.name variable))
                                              ((and ($ <formal>) variable)
                                               (.name variable))
                                              ((and ($ <shared-variable>) variable)
                                               (makreel:full-name variable))
                                              ((and ($ <stack>) stack)
                                               "s"))
                                            variables))
                                (assign? (member variable variables))
                                (shared (ast:shared* model)))
                           (cond ((and (or (not assign?)
                                           (and action?
                                                (ast:out? action?)))
                                       (null? shared))
                                  (makreel:process-parens next))
                                 (else
                                  `(,@(if (not action?) '()
                                          (makreel:shared-process-arguments action?))
                                    ,@(if (not assign?) '()
                                          (list
                                           (sm:is*
                                            variable
                                            (if action? "i"
                                                expression)))))))))))))))))

(define-method (ast->process (model <interface>) (o <the-end>) (next <ast>))
  (let* ((members (ast:member* model))
         (members (map .name members))
         (on (ast:parent o <on>))
         (trigger (car (ast:trigger* on))))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        `(,(if (ast:modeling? trigger) (%end-action model)
               (%reorder-end-action model))
          ,(sm:goto (name (statement->process-name next))
                    (arguments (makreel:process-parens next)))))))))

(define-method (ast->process (model <component>) (o <the-end>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:sequence*
      `(,@(if (not (makreel:defer-skip? model)) '()
              (list (%defer-skip-action model)))
        ,(%end-action model)
        ,(sm:goto (name (statement->process-name next))
                  (arguments (makreel:process-parens next))))))))

(define-method (ast->process (model <model>) (o <if>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:if* (makreel:ast->expression (.expression o))
             (sm:goto (name (statement->process-name next))
                      (arguments (makreel:process-parens next)))))))

(define-method (ast->process (model <model>) (o <if>) (next <list>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (match next
       ((then else)
        (sm:if* (makreel:ast->expression (.expression o))
                (sm:goto (name (statement->process-name then))
                         (arguments (makreel:process-parens then)))
                (sm:goto (name (statement->process-name else))
                         (arguments (makreel:process-parens else)))))))))

(define-method (ast->process (model <model>) (o <defer>) (next <ast>))
  (let* ((statement (.statement o))
         (id (makreel:process-identifier statement))
         (members (filter (negate (is? <shared-variable>)) (ast:member* model)))
         (names (map makreel:ast->expression members))
         (locals (makreel:locals statement))
         (local-names (map makreel:ast->expression locals)))
    (sm:process
      (name (statement->process-name o))
      (formals (makreel:process-formals o))
      (statement
       (sm:sequence*
        (sm:invoke %defer-qin-action
                   (list
                    (sm:entity (name "state_vector")
                               (formals (and (pair? names) names)))
                    (sm:set* (map (cute variable-prefix "select" <>)
                                  (ast:defer-variable* o)))
                    (sm:entity (name (simple-format #f "locals~a" id))
                               (formals (and (pair? local-names) local-names))))
                   #:keep-constructor? #t)
        (sm:goto (name (statement->process-name next))
                 (arguments (makreel:process-parens next))))))))

(define-method (ast->process (model <model>) (o <defer-end>) (next <ast>))
  (sm:process
    (name (statement->process-name o))
    (formals (makreel:process-formals o))
    (statement
     (sm:sequence*
      `(,%defer-end-action
        ,@(if (not (makreel:defer-skip? model)) '()
              (list (%defer-skip-action model)))
        ,(%end-action model)
        ,(sm:goto (name (statement->process-name next))
                  (arguments (makreel:process-parens next))))))))


;;;
;;; Processes.
;;;
(define (illegal-processes)
  (list
   (sm:process (name "Illegal")
               (statement
                (sm:sequence* %illegal-action
                              (sm:goto (name name)))))
   (sm:process (name "Constrained_Illegal")
               (statement
                (sm:sequence* %constrained-illegal-action
                              (sm:goto (name name)))))
   (sm:process (name "Declarative_Illegal")
               (statement
                (sm:sequence* %declarative-illegal-action
                              (sm:goto (name name)))))
   (sm:process (name "Non_Compliance")
               (statement
                (sm:sequence* %non-compliant-action
                              (sm:goto (name name)))))))

(define-method (statement->processes (model <model>) (statement <ast>))
  (define (step statement)
    (let ((next (ast:continuation* statement)))
      (match next
        ((next)
         (ast->process model statement next))
        ((next ...)
         (ast->process model statement next)))))
  (let loop ((statements (list statement)))
    (if (null? statements) '()
        (let* ((statement (car statements))
               (statements (cdr statements))
               (processes (cons (step statement)
                                (loop (ast:statement* statement)))))
          (append processes (loop statements))))))

(define-method (behavior->processes (model <model>))
  (let ((behavior (.behavior model)))
    (statement->processes model behavior)))

(define-method (defer->processes (o <defer>))
  (let* ((model (ast:parent o <model>))
         (statement (.statement o)))
    (statement->processes model statement)))

(define-method (function->processes (o <function>))
  (let ((model (ast:parent o <model>))
        (statement (.statement o)))
    (statement->processes model statement)))

(define-method (return-processes (o <model>))
  (define (variable->local v)
    (let ((name (.name v)))
      (simple-format #f "~a=~a (s)" name (variable-prefix "local" v))))
  (let ((returns calls (makreel:call-continuation* o)))
    (if (null? returns) '()
        (list
         (sm:process
           (name (model-prefix "return" o))
           (formals (append
                     (makreel:process-formals o)
                     (list (sm:formal (type (model-prefix "stack" o))
                                      (name "s"))
                           (sm:formal (type (model-prefix "return_type" o))
                                      (name "return_value")))))
           (statement
            (sm:union*
             (map
              (lambda (c r)
                (let* ((type (ast:type c))
                       (locals (makreel:locals r))
                       (f (and (not (ast:parent r <defer>))
                               (or (is-a? r <function>)
                                   (ast:parent r <function>))))
                       (arguments
                        `(,@(map variable->local locals)
                          ,@(if (not f) '()
                                '("s=stack (s)"))
                          ,@(match type
                              (($ <void>)
                               '())
                              (($ <bool>)
                               '("return_value=bool_value (return_value)"))
                              ((or ($ <int>) ($ <subint>))
                               '("return_value=int_value (return_value)"))
                              (($ <enum>)
                               (list
                                (simple-format
                                 #f
                                 "return_value=~a_value (return_value)"
                                 (makreel:type->string type))))))))
                  (sm:if* (sm:invoke (%return-predicate r)
                                     "return (s)")
                          (sm:goto (name (statement->process-name r))
                                   (arguments arguments)))))
              calls returns))))))))


;;;
;;; Interface.
;;;
(define-method (interface-reorder-processes (o <interface>))
  (let* ((interface-reorder
          (sm:process
            (name (model-prefix "reorder"))
            (statement
             (sm:union*
              (sm:sequence*
               (sm:sum (type (%events o))
                       (statement
                        (sm:sequence*
                         (sm:invoke (%in-action o))
                         (sm:union*
                          (sm:sum (var "i")
                                  (type (%state o))
                                  (statement
                                   (sm:sequence*
                                    (sm:invoke (%state-action o) var)
                                    (sm:goto
                                     (name (model-prefix "reorder_reply"))))))
                          (sm:sum (var "i")
                                  (type (%replies o))
                                  (statement
                                   (sm:sequence* (sm:invoke (%reply-action o) var)
                                                 (sm:goto (name
                                                           (model-prefix "reorder_replied"))
                                                          (arguments (list var))))))
                          (sm:sequence* (%reorder-end-action o)
                                        %missing-reply-action
                                        %sm:delta))))))
              (sm:sum (var "i")
                      (type (%modeling o))
                      (statement
                       (sm:sequence* (sm:invoke (%internal-action o) var)
                                     (sm:goto (name (model-prefix "reorder_internal"))))))
              (sm:sum (var "i")
                      (type (%state o))
                      (statement
                       (sm:sequence* (sm:invoke (%state-action o) var)
                                     (sm:goto (name name)))))))))
         (interface-reorder-internal
          (sm:process
            (name (model-prefix "reorder_internal"))
            (statement
             (sm:union*
              (sm:sequence*
               (%end-action o)
               (sm:goto (name (model-prefix "reorder"))))
              (sm:sum (var "i")
                      (type (%state o))
                      (statement
                       (sm:sequence*
                        (sm:invoke (%state-action o) var)
                        (sm:goto (name name)))))))))
         (interface-reorder-reply
          (sm:process
            (name (model-prefix "reorder_reply"))
            (statement
             (sm:union*
              (sm:sum (var "i")
                      (type (%replies o))
                      (statement
                       (sm:sequence* (sm:invoke (%reply-action o) var)
                                     (sm:goto (name
                                               (model-prefix "reorder_replied"))
                                              (arguments (list var))))))
              (sm:sum (var "i")
                      (type (%state o))
                      (statement
                       (sm:sequence*
                        (sm:invoke (%state-action o) var)
                        (sm:goto (name name)))))))))
         (interface-reorder-replied
          (let ((reply "r"))
            (sm:process
              (name (model-prefix "reorder_replied"))
              (formals (list (sm:formal (type (%replies o)) (name reply))))
              (statement
               (sm:union*
                (sm:sequence* (%reorder-end-action o)
                              (sm:invoke (%reply-reordered-action o) reply)
                              (sm:goto (name (model-prefix "reorder"))))
                (sm:sum (var "i")
                        (type (%replies o))
                        (statement
                         (sm:sequence* (sm:invoke (%reply-action o) var)
                                       %second-reply-action
                                       %sm:delta)))
                (sm:sum (var "i")
                        (type (%state o))
                        (statement
                         (sm:sequence* (sm:invoke (%state-action o) var)
                                       (sm:goto (name name)
                                                (arguments '()))))))))))
         (members (ast:variable* o))
         (member-values (map (compose makreel:ast->expression .expression) members))
         (member-types (map .type members))
         (interface-reordered-parallel
          (sm:process
            (name (model-prefix "reordered_parallel"))
            (statement (sm:parallel*
                        (sm:sequence*
                         `(,@(append-map makreel:type->out-of-range-processes
                                         member-types member-values)
                           ,(sm:goto (name (model-prefix "behavior"))
                                     (arguments member-values))))
                        (model-prefix "reorder")))))
         (interface-reordered-comm
          (sm:process
            (name (model-prefix "reordered_comm"))
            (statement
             (sm:comm
              (process interface-reordered-parallel)
              (events
               (list
                (sm:comm-event (from (sm:multi-event (events
                                                      (list (%in-action o))))))
                (sm:comm-event (from
                                (sm:multi-event (events
                                                 (list (%reply-action o))))))
                (sm:comm-event (from (sm:multi-event (events
                                                      (list (%internal-action o))))))
                (sm:comm-event (from
                                (sm:multi-event (events
                                                 (list (%end-action o))))))
                (sm:comm-event (from (sm:multi-event
                                      (events
                                       (list (%reorder-end-action o))))))
                (sm:comm-event (from (sm:multi-event
                                      (events
                                       (list (%state-action o))))))))))))
         (interface-reordered-allow
          (sm:process
            (name (model-prefix "reordered_allow"))
            (statement
             (sm:allow
              (process interface-reordered-comm)
              (events (list %declarative-illegal-action
                            %missing-reply-action
                            %range-error-action
                            %recurse-action
                            %return-action
                            %second-reply-action
                            %tag-action
                            %tau-void-action
                            (%flush-action o)
                            (%out-action o)
                            (%reply-reordered-action o)
                            (%tau-reply-action o)))))))
         (interface-reordered
          (sm:process
            (name (model-prefix "reordered"))
            (statement
             (sm:rename
              (process interface-reordered-allow)
              (events (cons*
                       (sm:rename-event (from (sm:transpose-tick (%reply-action o)))
                                        (to (%tau-reply-action o)))
                       (sm:rename-event (from (%reply-reordered-action o))
                                        (to (%reply-action o)))
                       (sm:rename-event (from (sm:transpose-tick
                                               (%reorder-end-action o)))
                                        (to %tau-void-action))
                       (sm:comm-events (sm:process-statement
                                        interface-reordered-comm))))))))
         (interface-semantics
          (sm:process
            (name (model-prefix "semantics"))
            (statement
             (sm:union*
              (sm:sum (type (%state o))
                      (statement
                       (sm:sequence*
                        (sm:invoke (%state-action o))
                        (sm:goto (name name)))))
              (sm:sum (type (%events o))
                      (statement
                       (sm:sequence*
                        (sm:invoke (%in-action o))
                        (sm:goto (name (model-prefix "semantics_sync"))))))
              (sm:sum (var "i")
                      (type (%modeling o))
                      (statement
                       (sm:sequence*
                        (sm:invoke (%internal-action o) var)
                        (sm:union*
                         (sm:goto (name (model-prefix "semantics_flush")))
                         (sm:sequence*
                          (%end-action o)
                          (%end-reordered-action o)
                          (sm:goto (name (model-prefix "semantics"))))))))))))
         (interface-semantics-sync
          (sm:process
            (name (model-prefix "semantics_sync"))
            (statement
             (sm:union*
              (sm:sequence*
               (sm:sum (var "i")
                       (type (%replies o))
                       (statement
                        (sm:sequence* (sm:invoke (%reply-action o) var)
                                      (sm:goto (name (model-prefix "semantics")))))))
              (sm:sum (type (%events o))
                      (statement
                       (sm:sequence*
                        (sm:invoke (%out-action o))
                        (sm:goto (name name)))))
              (sm:sum (type (%state o))
                      (statement
                       (sm:sequence*
                        (sm:invoke (%state-action o))
                        (sm:goto (name name)))))))))
         (interface-semantics-flush
          (sm:process
            (name (model-prefix "semantics_flush"))
            (statement
             (sm:sum (type (%events o))
                     (statement
                      (sm:union*
                       (sm:sequence*
                        (sm:invoke (%out-action o))
                        (sm:goto (name name)))
                       (sm:sequence*
                        (%end-action o)
                        (%flush-action o)
                        (%end-reordered-action o)
                        (sm:goto (name (model-prefix "semantics"))))
                       (sm:sum (var "i")
                               (type (%state o))
                               (statement
                                (sm:sequence*
                                 (sm:invoke (%state-action o) var)
                                 (sm:goto (name name)))))))))))
         (interface-parallel
          (sm:process
            (name (model-prefix "parallel"))
            (statement
             (sm:parallel* (model-prefix "semantics")
                           (model-prefix "reordered")))))
         (interface-comm
          (sm:process
            (name (model-prefix "comm"))
            (statement
             (sm:comm
              (process interface-parallel)
              (events
               (list
                (sm:comm-event (from (sm:multi-event (events
                                                      (list (%in-action o))))))
                (sm:comm-event (from (sm:multi-event (events
                                                      (list (%out-action o))))))
                (sm:comm-event (from (sm:multi-event (events
                                                      (list (%internal-action o))))))
                (sm:comm-event (from
                                (sm:multi-event (events
                                                 (list (%reply-action o))))))
                (sm:comm-event (from
                                (sm:multi-event (events
                                                 (list (%end-action o))))))
                (sm:comm-event (from
                                (sm:multi-event (events
                                                 (list (%state-action o))))))))))))
         (interface-allow
          (sm:process
            (name (model-prefix "allow"))
            (statement
             (sm:allow
              (process interface-comm)
              (events (list %declarative-illegal-action
                            %missing-reply-action
                            %range-error-action
                            %recurse-action
                            %return-action
                            %second-reply-action
                            %tag-action
                            %tau-void-action
                            (%end-reordered-action o)
                            (%flush-action o)
                            (%tau-reply-action o)))))))
         (interface-internal
          (sm:process
            (name (model-prefix "interface_internal"))
            (statement
             (sm:rename
              (process interface-allow)
              (events (cons*
                       (sm:rename-event (from %recurse-action)
                                        (to %tau-void-action))
                       (sm:rename-event (from %return-action)
                                        (to %tau-void-action))
                       (sm:rename-event (from (sm:transpose-tick (%end-action o)))
                                        (to %tau-void-action))
                       (sm:rename-event (from (%end-reordered-action o))
                                        (to (%end-action o)))
                       (sm:comm-events (sm:process-statement interface-comm))))))))
         (interface
          (sm:process
            (name (model-prefix "interface"))
            (statement
             (sm:hide
              (process interface-internal)
              (events (list %tau-void-action
                            (%end-action o)
                            (%tau-reply-action o))))))))
    (list interface-reorder
          interface-reorder-reply
          interface-reorder-replied
          interface-reorder-internal
          interface-reordered-parallel
          interface-reordered-comm
          interface-reordered-allow
          interface-reordered
          interface-semantics
          interface-semantics-sync
          interface-semantics-flush
          interface-parallel
          interface-comm
          interface-allow
          interface-internal
          interface)))

(define-method (interface-types->scmackerel (o <interface>))
  (let* ((behavior (.behavior o))
         (enums (filter (is? <enum>) (ast:type** o)))
         (enums (map enum->scmackerel enums)))
    (sm:mcrl2 (types enums))))

(define-method (interface->scmackerel-unmemoized (o <interface>))
  (let* ((model-name (ast:dotted-name o))
         (behavior (.behavior o))
         (enums (filter (is? <enum>) (ast:type** o)))
         (enums (map enum->scmackerel enums))
         (behavior-processes (behavior->processes o))
         (functions (makreel:called-function* behavior))
         (function-processes (append-map function->processes functions))
         (return-processes (return-processes o))
         (reorder-processes (interface-reorder-processes o))
         (processes `(,@behavior-processes
                      ,(makreel:caption "FUNCTIONS")
                      ,@function-processes
                      ,@return-processes
                      ,@reorder-processes))
         (return-type (%return-type o))
         (returns-type (%returns-type o))
         (types `(,%void
                  ,(%actions o)
                  ,(%modeling o)
                  ,(%replies o)
                  ,(%events o)
                  ,@enums
                  ,(%state o)
                  ,@(if (not return-type) '()
                        (list return-type))
                  ,(%stack-type o)
                  ,@(if (not returns-type) '()
                        (list returns-type)))))
    (sm:mcrl2 (types types) (processes processes))))

(define interface->scmackerel
  (ast:perfect-funcq interface->scmackerel-unmemoized))

(define-method (model->scmackerel (o <interface>))
  (makreel:proc-list (.behavior o))
  (parameterize ((%model-name (makreel:full-name o)))
    (interface->scmackerel o)))


;;;
;;; Component.
;;;
(define-method (port-processes (o <port>))
  (parameterize ((%model-name (makreel:full-name (.type o)))
                 (%port-name (.name o)))
    (let* ((interface (.type o))
           (rename
            (sm:process
              (name (port-prefix "port"))
              (statement
               (sm:rename
                (process (model-prefix "interface_internal"))
                (events (list
                         (sm:rename-event (from (%in-action interface))
                                          (to (%in-action o)))
                         (sm:rename-event (from (%out-action interface))
                                          (to (%out-action o)))
                         (sm:rename-event (from (%reply-action interface))
                                          (to (%reply-action o)))
                         (sm:rename-event (from (%internal-action interface))
                                          (to (%internal-action o)))
                         (sm:rename-event (from (%end-action interface))
                                          (to (%end-action o)))
                         (sm:rename-event (from (%state-action interface))
                                          (to (%state-action o)))
                         (sm:rename-event (from (%flush-action interface))
                                          (to (%flush-action o))))))))))
      (list rename))))

(define-method (provides-processes (o <component>))
  (let* ((ports (ast:port* o))
         (provides (ast:provides-port* o))
;;; XXX TODO: when/how to print a type's entities in a formal?
         (provides-ports-set
          (sm:type (name "Set")
                   (entities (list (sm:entity (name "provides_ports")
                                              (formals '("provides_ports")))))))
         (provides-ports-set
          (sm:type (name "Set (provides_ports)")
                   (entities (list (sm:entity (name "provides_ports")
                                              (formals '("provides_ports")))))))
         (blocking? (find ast:blocking? provides))
         (provides-r2c
          (if (not blocking?)
              (sm:process
                (name "provides_r2c")
                (formals (list (sm:formal (name "ports")
                                          (type provides-ports-set))))
                (statement
                 (sm:union*
                  (append-map
                   (lambda (p)
                     (let ((interface (.type p)))
                       (list
                        (sm:sum (type (%actions interface))
                                (statement
                                 (sm:sequence* (sm:invoke (%in-action p)
                                                          #:keep-constructor? #t)
                                               (sm:goto (name "provides_out")
                                                        (arguments '())))))
                        (sm:sum (type (%modeling interface))
                                (statement
                                 (sm:sequence* (sm:invoke (%internal-action p))
                                               (sm:goto (name "provides_out")
                                                        (arguments '()))))))))
                   provides))))
              (sm:process
                (name "provides_r2c")
                (formals (list (sm:formal (name "ports")
                                          (type provides-ports-set))))
                (statement
                 (sm:union*
                  (append-map
                   (lambda (p)
                     (let ((interface (.type p)))
                       (list
                        (sm:sum (type (%actions interface))
                                (statement
                                 (sm:sequence*
                                  (sm:invoke (%in-action p)
                                             #:keep-constructor? #t)
                                  (sm:union*
                                   (sm:sequence* "tau" (%blocking-action p))
                                   "tau")
                                  (sm:goto (name name)
                                           (arguments
                                            (list
                                             (sm:plus*
                                              "ports"
                                              (sm:set* (port-prefix "port" p)))))))))
                        (sm:if* (sm:in* (port-prefix "port" p) "ports")
                                (sm:sequence*
                                 "tau"
                                 (sm:union*
                                  (sm:sum (type (%replies interface))
                                          (statement
                                           (sm:sequence*
                                            (sm:invoke (%reply-action p))
                                            (sm:goto (name name)
                                                     (arguments
                                                      (list
                                                       (sm:minus*
                                                        "ports"
                                                        (sm:set* (port-prefix "port" p)))))))))
                                  (sm:sum (type (%actions interface))
                                          (statement
                                           (sm:sequence*
                                            (sm:invoke (%out-action p)
                                                       (sm:construct
                                                        (%out-action interface)))
                                            (sm:goto (name name)
                                                     (arguments '()))))))))
                        (sm:sum (type (%modeling interface))
                                (statement
                                 (sm:sequence* (sm:invoke (%internal-action p))
                                               (sm:goto (name "provides_out")
                                                        (arguments '())))))
                        (sm:sequence* (%end-action p)
                                      (sm:goto (name name)
                                               (arguments '()))))))
                   provides))))))
         (provides-out
          (sm:process
            (name "provides_out")
            (formals (list (sm:formal (name "ports") (type provides-ports-set))))
            (statement
             (sm:union*
              (append-map
               (lambda (p)
                 (let ((interface (.type p)))
                   (list
                    (sm:sum (type (%replies interface))
                            (statement
                             (sm:sequence* (sm:invoke (%reply-action p))
                                           (sm:goto (name "provides_r2c")
                                                    (arguments '())))))
                    (sm:sequence* (%end-action p)
                                  (sm:goto (name "provides_r2c")
                                           (arguments '())))
                    (sm:sum (type (%actions interface))
                            (statement
                             (sm:sequence* (sm:invoke (%out-action p)
                                                      #:keep-constructor? #t)
                                           (sm:goto (name name)
                                                    (arguments '()))))))))
               provides)))))
         (provides-parallel
          (sm:process
            (name "provides_parallel")
            (statement (sm:parallel
                        (processes
                         `(,@(map (cute port-prefix "port" <>) provides)
                           ,(sm:goto (name "provides_r2c")
                                     (arguments (list (sm:set*))))))))))
         (provides-comm
          (sm:process
            (name "provides_comm")
            (statement
             (sm:comm
              (process provides-parallel)
              (events
               (append-map
                (lambda (p)
                  (list
                   (sm:comm-event (from (sm:multi-event
                                         (events
                                          (list (%in-action p))))))
                   (sm:comm-event (from (sm:multi-event
                                         (events
                                          (list (%out-action p))))))
                   (sm:comm-event (from (sm:multi-event
                                         (events
                                          (list (%internal-action p))))))
                   (sm:comm-event (from
                                   (sm:multi-event (events
                                                    (list (%reply-action p))))))
                   (sm:comm-event (from
                                   (sm:multi-event (events
                                                    (list (%end-action p))))))))
                provides))))))
         (provides-interfaces (delete-duplicates (map .type provides) ast:eq?))
         (provides-allow
          (sm:process
            (name "provides_allow")
            (statement
             (sm:allow
              (process provides-comm)
              (events (cons* %declarative-illegal-action
                             %illegal-action
                             %missing-reply-action
                             %range-error-action
                             %second-reply-action
                             %tag-action
                             %tau-void-action
                             (append
                              (map %tau-reply-action provides-interfaces)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%blocking-action p)
                                  (%flush-action p)
                                  (%state-action p)))
                               provides))))))))
         (provides-rename
          (sm:process
            (name "provides_rename")
            (statement
             (sm:rename
              (process provides-allow)
              (events (append
                       (map (lambda (p)
                              (sm:rename-event (from (sm:transpose-tick (%end-action p)))
                                               (to %tau-void-action)))
                            provides)
                       (sm:comm-events (sm:process-statement provides-comm))))))))
         (interfaces (delete-duplicates (map .type ports) ast:eq?))
         (provides
          (sm:process
            (name "provides")
            (statement
             (sm:hide
              (process provides-rename)
              (events (cons %tau-void-action
                            (append-map (lambda (i)
                                          (list (%tau-event-action i)
                                                (%tau-modeling-action i)
                                                (%tau-reply-action i)))
                                        interfaces))))))))
    (list provides-r2c
          provides-out
          provides-parallel
          provides-comm
          provides-allow
          provides-rename
          provides)))

(define-method (component-defer-processes (o <component>))
  (let* ((ports (ast:port* o))
         (provides (ast:provides-port* o))
         (requires (ast:requires-port* o))
         (members (ast:variable* o))
         (member-types (map .type members))
         (member-values (and
                         (pair? members)
                         (map (compose makreel:ast->expression .expression) members)))
         (component-defer-parallel
          (sm:process (name "component_defer_parallel")
                      (statement
                       (sm:parallel*
                        (sm:sequence*
                         `(,@(if (not member-values) '()
                                 (append-map makreel:type->out-of-range-processes
                                             member-types member-values))
                           ,(sm:goto (name (model-prefix "behavior"))
                                     (arguments member-values))))
                        (sm:goto (name "defer_queue")
                                 (arguments '("[]")))))))
         (component-defer-comm
          (sm:process
            (name "component_defer_comm")
            (statement
             (sm:comm
              (process component-defer-parallel)
              (events
               (list
                (sm:comm-event (from (sm:multi-event (events
                                                      (list %defer-qin-action)))))
                (sm:comm-event (from (sm:multi-event (events
                                                      (list (%defer-qout-action o))))))
                (sm:comm-event (from (sm:multi-event (events
                                                      (list (%defer-skip-action o))))))))))))
         (interfaces (delete-duplicates (map .type ports) ast:eq?))
         (component-defer-allow
          (sm:process
            (name "component_defer_allow")
            (statement
             (sm:allow
              (process component-defer-comm)
              (events (cons* %constrained-legal-action
                             %constrained-illegal-action
                             %declarative-illegal-action
                             %defer-end-action
                             %illegal-action
                             %queue-full-action
                             %range-error-action
                             %recurse-action
                             %return-action
                             %tag-action
                             ;; XXX wip-shared-state does not have this
                             ;; %tau-void-action
                             (%end-action o)
                             (append
                              (append-map
                               (lambda (p)
                                 (list
                                  (%blocking-action p)
                                  (%flush-action p)
                                  (%in-action p)
                                  (%out-action p)
                                  (%reply-action p)
                                  (%state-action p)))
                               provides)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%flush-action p)
                                  (%in-action p)
                                  (%out-action p)
                                  (%reply-action p)
                                  (%state-action p)
                                  (%switch-context-action p)))
                               requires))))))))
         (component-defer-rename
          (sm:process
            (name "component_defer_rename")
            (statement
             (sm:rename
              (process component-defer-allow)
              (events
               (append
                (sm:comm-events (sm:process-statement component-defer-comm))
                (map (lambda (p)
                       ;; FIXME: comm-event is weird, from and to are reversed
                       (sm:comm-event (from (%qout-action p)) (to (%out-action p))))
                     (ast:requires-port* o))))))))
         (component-defer
          (sm:process
            (name "component_defer")
            (statement
             (sm:hide
              (process component-defer-rename)
              (events (list %defer-qin-action))))))
         (component-behavior
          (sm:process (name "component_behavior")
                      (statement (sm:goto (name "component_defer"))))))
    (list component-defer-parallel
          component-defer-comm
          component-defer-allow
          component-defer-rename
          component-defer
          component-behavior)))

(define-method (component-no-q-processes (o <component>))
  (list
   (sm:process (name "component_queue")
               (statement (sm:sequence*
                           %queue-empty-action
                           (sm:goto (name name)))))
   (sm:process (name "req_and_queue")
               (statement  (sm:goto (name "component_queue"))))))

(define-method (port->external-processes (o <port>))
  (let* ((interface (.type o))
         (port-name (.name o))
         (external-q
          (sm:process
            (name (port-prefix "external_queue" o))
            (formals (list (sm:formal (type "List (requires_events)")
                                      (name "l"))))
            (statement
             (sm:union*
              (sm:if* "#l < ExternalQueueLength"
                      (sm:sum (type (%events interface))
                              (statement
                               (sm:sequence*
                                (sm:invoke (%qin-action o))
                                (sm:goto (name name)
                                         (arguments
                                          (list
                                           (simple-format #f "l=l <| ~a (i)"
                                                          (port-prefix "events" o)))))))))
              (sm:if* (simple-format #f "l != [] && ~a (head (l))"
                                     (port-prefix "is" o))
                      (sm:sequence*
                       (sm:invoke (%internal-action o)
                                  (model-prefix "inevitable" interface))
                       (sm:invoke (%qout-action o)
                                  ;; XXX WTF "projection?"
                                  (port-prefix "projection (head (l))" o))
                       (%end-action o)
                       (sm:goto (name name)
                                (arguments '("l=tail (l)")))))
              (sm:if* "#l == ExternalQueueLength"
                      (sm:sum (type (%events interface))
                              (statement
                               (sm:sequence*
                                (sm:invoke (%qin-action o))
                                (%port-queue-full-action o)
                                (sm:goto (name "Illegal"))))))))))
         (port-internal
          (sm:process
            (name (port-prefix "port_internal" o))
            (statement
             (sm:rename
              (process (port-prefix "port" o))
              (events (list
                       (sm:rename-event (from (%internal-action o))
                                        (to (%tau-modeling-action interface)))
                       (sm:rename-event (from (%end-action o))
                                        (to %tau-void-action))
                       (sm:rename-event (from (%flush-action o))
                                        (to %tau-void-action))))))))
         (external-parallel
          (sm:process
            (name (port-prefix "external_parallel" o))
            (statement (sm:parallel
                        (processes
                         (list
                          (sm:goto (name (sm:process-name external-q))
                                   (arguments '("[]")))
                          (sm:goto (name (port-prefix "port_internal" o)))))))))
         (external-comm
          (sm:process
            (name (port-prefix "external_comm" o))
            (statement
             (sm:comm
              (process external-parallel)
              (events
               (list
                (sm:comm-event (from (sm:multi-event
                                      (events (list (%out-action o)
                                                    (%qin-action o)))))
                               (to (sm:transpose-tick (%qin-action o))))))))))
         (external-allow
          (sm:process
            (name (port-prefix "external_allow" o))
            (statement
             (sm:allow
              (process external-comm)
              (events (list %declarative-illegal-action
                            %illegal-action
                            %tag-action
                            %tau-void-action
                            (%tau-modeling-action interface)
                            (%tau-reply-action interface)
                            (%in-action o)
                            (%qout-action o)
                            (%reply-action o)
                            (%internal-action o)
                            (%end-action o)
                            (%flush-action o)
                            (%port-queue-full-action o)))))))
         (external
          (sm:process
            (name (port-prefix "external" o))
            (statement
             (sm:rename
              (process external-allow)
              (events (list
                       (sm:rename-event (from (sm:transpose-tick (%qin-action o)))
                                        (to (%tau-event-action interface)))
                       (sm:rename-event (from (%qout-action o))
                                        (to (%out-action o))))))))))
    (list
     external-q
     port-internal
     external-parallel
     external-comm
     external-allow
     external)))

(define-method (component-q-processes (o <component>))
  (let* ((ports (ast:port* o))
         (interfaces (ast:interface* o))
         (requires (ast:requires-port* o))
         (component-queue
          (sm:process
            (name "component_queue")
            (formals (list (sm:formal (name "l") (type "List (requires_events)"))))
            (statement
             (sm:union*
              (cons*
               (sm:if* "#l == 0"
                       (sm:sequence*
                        %queue-empty-action
                        (sm:goto (name name)
                                 (arguments '()))))
               (sm:if* "l != []"
                       (sm:sequence*
                        %queue-not-empty-action
                        (sm:goto (name name)
                                 (arguments '()))))
               (append-map
                (lambda (p)
                  (let* ((interface (.type p)))
                    (list
                     (sm:if* "#l < QueueLength"
                             (sm:sequence*
                              (sm:sum (type (%events interface))
                                      (statement (sm:invoke (%qin-action p))))
                              (sm:goto (name name)
                                       (arguments
                                        (list
                                         (string-append "l=l <| "
                                                        (port-prefix "events" p)
                                                        " (i)"))))))
                     (sm:if* "#l == QueueLength"
                             (sm:sequence*
                              (sm:sum (type (%events interface))
                                      (statement (sm:invoke (%qin-action p))))
                              %queue-full-action
                              (sm:goto (name "Illegal"))))
                     (sm:if* (string-append "l != [] && "
                                            (port-prefix "is (head (l))" p))
                             (sm:sequence*
                              (sm:invoke (%qout-action p)
                                         (port-prefix "projection (head (l))" p))
                              (sm:goto (name name)
                                       (arguments
                                        '("l=tail (l)"))))))))
                requires))))))
         (external (filter ast:external? requires))
         (non-external (filter (negate ast:external?) requires))
         (port-external-processes
          (append-map port->external-processes external))
         (req-and-queue-parallel
          (sm:process
            (name "req_and_queue_parallel")
            (statement
             (sm:parallel
              (processes
               (cons*
                (sm:goto (name "component_queue")
                         (arguments (list "[]")))
                (append
                 (map (lambda (p)
                        (sm:goto (name (port-prefix "port" p))))
                      non-external)
                 (map (lambda (p)
                        (sm:goto (name (port-prefix "external" p))))
                      external))))))))
         (req-and-queue-comm
          (sm:process
            (name "req_and_queue_comm")
            (statement
             (sm:comm
              (process req-and-queue-parallel)
              (events
               (map
                (lambda (p)
                  (sm:comm-event
                   (from
                    (sm:multi-event
                     (events
                      (list (%qin-action p)
                            (%out-action p)))))))
                requires))))))
         (req-and-queue-allow
          (sm:process
            (name "req_and_queue_allow")
            (statement
             (sm:allow
              (process req-and-queue-comm)
              (events
               (cons* %declarative-illegal-action
                      %illegal-action
                      %queue-empty-action
                      %queue-full-action
                      %queue-not-empty-action
                      %tau-void-action
                      (append
                       (append-map
                        (lambda (i)
                          (list
                           (%tau-event-action i)
                           (%tau-modeling-action i)
                           (%tau-reply-action i)))
                        interfaces)
                       (append-map
                        (lambda (p)
                          (list
                           (%in-action p)
                           (%reply-action p)
                           (%internal-action p)
                           (%qout-action p)
                           (%end-action p)
                           (%flush-action p)
                           (%state-action p)
                           (%port-queue-full-action p)))
                        requires))))))))
         (req-and-queue
          (sm:process
            (name "req_and_queue")
            (statement
             (sm:rename
              (process req-and-queue-allow)
              (events
               (cons*
                (sm:rename-event (from %declarative-illegal-action)
                                 (to %illegal-action))
                (map
                 (lambda (p)
                   (sm:rename-event (from (sm:transpose-tick (%qin-action p)))
                                    (to (%qin-action p))))
                 requires))))))))
    `(,component-queue
      ,@port-external-processes
      ,req-and-queue-parallel
      ,req-and-queue-comm
      ,req-and-queue-allow
      ,req-and-queue)))


;;;
;;; Semantics.
;;;
(define %blocking-formals '("blocking: Set (provides_ports)"
                            "released: List (provides_ports)"
                            "reply: reply_values"))

(define %blocked-formals `("rtc: provides_ports"
                           ,@%blocking-formals))

(define-method (out-actions (p <port>) name)
  (let ((interface (.type p)))
    (sm:sum (type (%actions interface))
            (statement
             (sm:sequence*
              (sm:invoke (%out-action p) #:keep-constructor? #t)
              (sm:union*
               (sm:goto (name name)
                        (arguments '()))
               (sum-state-action
                p (sm:goto (name name)
                           (arguments '())))))))))

(define-method (qout-actions (p <port>) name)
  (let ((interface (.type p)))
    (sm:sum (type (%actions interface))
            (statement
             (sm:sequence*
              (sm:invoke (%qout-action p) #:keep-constructor? #t)
              (sm:goto (name name)
                       (arguments '())))))))

(define-method (switch-context-actions (p <port>) name)
  (sm:sequence*
   (%switch-context-action p)
   (sm:goto (name name)
            (arguments '()))))

(define-method (modeling+switch-context (p <port>) name continuation)
  (let ((interface (.type p)))
    (list
     (sm:if* (sm:not-equal* (port-prefix "port" p)
                            "switch")
             (sm:sum (type (%modeling interface))
                     (statement
                      (sm:sequence*
                       (sm:invoke (%internal-action p))
                       (%end-action p)
                       (sm:goto (name name)
                                (arguments '()))))))
     (sm:if* (sm:equal* (port-prefix "port" p)
                        "switch")
             (sm:sequence*
              (%switch-context-action p)
              (sm:goto (name continuation)
                       (arguments '())))))))

(define-method (defer-qout-actions (o <component>) name)
  (sm:sum (type (%locals o))
          (statement
           (sm:sequence*
            (sm:invoke (%defer-qout-action o))
            (sm:goto (name name)
                     (arguments '()))))))

(define-method (defer-qout-actions-async (o <component>) name)
  (sm:sum (type (%locals o))
          (statement
           (sm:sequence*
            (sm:invoke (%defer-qout-action o))
            (sm:goto (name "semantics_async")
                     (arguments '("port=no_port")))))))

(define-method (semantics-main (o <component>))
  (let* ((provides (ast:provides-port* o))
         (provides-interfaces (map .type provides))
         (requires (ast:requires-port* o))
         (shared (map .port (ast:shared* o)))
         (shared-provides (lset-intersection ast:eq? provides shared)))
    (define (semantics-provides p name)
      (let ((interface (.type p)))
        (sm:sum (type (%actions interface))
                (statement
                 (sm:sequence*
                  (sm:invoke (%in-action p) #:keep-constructor? #t)
                  (sm:goto (name "semantics_unblocked")
                           (arguments
                            (list
                             (sm:is*
                              "reply"
                              (sm:entity
                               (name "replies")
                               (formals
                                (map
                                 (cute model-prefix "nil" <>)
                                 provides-interfaces))))))))))))
    (define (semantics-requires p name)
      (let ((interface (.type p)))
        (sm:sum (type (%modeling interface))
                (statement
                 (sm:sequence*
                  `(,(sm:invoke (%internal-action p))
                    ,(%end-action p)
                    ,(sm:union*
                      (list
                       (sm:sequence*
                        `(,%queue-empty-action
                          ,@(map sum-state-action shared-provides)
                          ,(sm:goto (name name))))
                       (sm:goto (name "semantics_async")
                                (arguments
                                 (list
                                  "blocking={}"
                                  "released=[]"
                                  (sm:is*
                                   "reply"
                                   (sm:entity
                                    (name "replies")
                                    (formals
                                     (map
                                      (cute model-prefix "nil" <>)
                                      provides-interfaces))))
                                  "port=no_port")))))))))))
    ;; invariant QMT => allow provides in, or requires modeling
    (sm:process
      (name "semantics_main")
      (statement
       (sm:union*
        `(,@(map (cute semantics-provides <> name) provides)
          ,@(map (cute semantics-requires <> name) requires)))))))

(define-method (semantics-unblocked (o  <component>))
  (let* ((provides (ast:provides-port* o))
         (provides-interfaces (map .type provides))
         (provides-interface (.type (ast:provides-port o)))
         (requires (ast:requires-port* o)))
    (define (handle-provides-reply p name)
      (let ((interface (.type p)))
        (list
         (sm:sequence* (%blocking-action p)
                       (sm:goto (name "semantics_blocking")
                                (arguments
                                 `(,(sm:is*
                                     "blocking"
                                     (sm:set*
                                      (port-prefix "port" p)))
                                   "released=[]"))))
         (sm:sum (type (%replies interface))
                 (statement
                  (sm:sequence*
                   (sm:invoke (%reply-action p))
                   (sm:if* (sm:invoke
                            (%nil-predicate interface)
                            (sm:construct (%port-predicate p) "reply"))
                           (sm:goto (name name)
                                    (arguments
                                     ;; XXX loop var i
                                     (list
                                      (sm:is*
                                       "reply"
                                       (sm:entity
                                        (name "replies")
                                        (formals
                                         (map
                                          (lambda (i)
                                            (if (ast:eq? i p) "i"
                                                (port-prefix "port (reply)"
                                                             i)))
                                          provides)))))))
                           (sm:sequence*
                            %second-reply-action
                            %sm:delta))))))))
    (define (handle-missing-reply name)
      (sm:if* (sm:expression
               (operator "&&")
               (operands
                (map
                 (lambda (p)
                   (let ((interface (.type p)))
                     (sm:invoke (%nil-predicate interface)
                                (sm:construct (%port-predicate p)
                                              "reply"))))
                 provides)))
              (sm:sequence* %missing-reply-action %sm:delta)))
    (define (handle-defer-qout+goto-main p name)
      (let ((interface (.type p)))
        (sm:if* (sm:not*
                 (sm:invoke (%nil-predicate interface)
                            (sm:construct (%port-predicate p)
                                          "reply")))
                (sm:sequence*
                 `(,(sm:invoke (%reply-reordered-action p)
                               ;; FIXME constructor?
                               (port-prefix "port (reply)" p))
                   ,@(let ((port p))
                       (if (not (find (compose (cute ast:eq? port <>) .port)
                                      (ast:shared* o)))
                           '()
                           `(,(sum-state-action port))))
                   ,(sm:union*
                     (sm:goto (name "semantics_main"))
                     (sm:sum (type (%locals o))
                             (statement
                              (sm:sequence*
                               (sm:invoke (%defer-qout-action o))
                               (sm:goto
                                (name "semantics_async")
                                (arguments
                                 `("blocking={}"
                                   "released=[]"
                                   ,(sm:is*
                                     "reply"
                                     (sm:entity
                                      (name "replies")
                                      (formals
                                       (map
                                        (cute model-prefix
                                              "nil" <>)
                                        provides-interfaces))))
                                   "port=no_port"))))))))))))
    (define (handle-qout name)
      (if (null? requires) '()
          (list
           (sm:sequence*
            `(,%queue-not-empty-action
              ,(sm:union*
                (map (cute qout-actions <> name) requires)))))))
    ;; switch-context-actions but goto
    ;; semantics_unblocked_switch_context with switch=h'port
    (define (unblocked-switch-context-actions+switch=p p name)
      (sm:sequence*
       (%switch-context-action p)
       (sm:goto (name "semantics_unblocked_switch_context")
                (arguments (list (string-append "switch="
                                                (port-prefix "port" p)))))))
    ;; handle (provides) replies (missing, second, reorder) and
    ;; synchronous (provides) out events, or continue as blocked
    ;; => emit blocking event
    (sm:process
      (name "semantics_unblocked")
      (formals (list (sm:formal (name "reply") (type (%reply-values o)))))
      (statement
       (sm:union*
        `(,@(append-map (cute handle-provides-reply <> name) provides)
          ,@(map (cute out-actions <> name) provides)
          ,(sm:sequence*
            (%end-action o)
            (sm:union*
             `(,(sm:sequence*
                 %queue-empty-action
                 (sm:union*
                  `(,(handle-missing-reply name)
                    ,@(map
                       (cute handle-defer-qout+goto-main <> name)
                       provides))))
               ,@(handle-qout name))))
          ,@(map (cute unblocked-switch-context-actions+switch=p <> name) requires)))))))

(define-method (unblocked-semantics-modeling+switch-context (o <component>))
  ;; handle switch_context: collateral blocking => allow
  ;; requires modeling for other ports
  (let ((requires (ast:requires-port* o)))
    (sm:process
      (name "semantics_unblocked_switch_context")
      (formals '("reply: reply_values" "switch: requires_ports"))
      (statement
       (sm:union*
        `(,%sm:delta
          ,@(append-map
             (cute modeling+switch-context <> name "semantics_unblocked")
             requires)))))))

(define-method (blocking-semantics-modeling+switch-context (o <component>))
  ;; handle switch_context: collateral blocking => allow
  ;; requires modeling for other ports
  (let ((requires (ast:requires-port* o)))
    (sm:process
      (name "semantics_blocking_switch_context")
      (formals '("blocking: Set (provides_ports), released: List (provides_ports), reply: reply_values" "switch: requires_ports"))
      (statement
       (sm:union*
        `(,%sm:delta
          ,@(append-map
             (cute modeling+switch-context <> name "semantics_blocking")
             requires)))))))

(define-method (semantics-blocking (o <component>))
  (let ((provides (ast:provides-port* o))
        (requires (ast:requires-port* o)))
    (define (blocking-switch-context-actions+switch=p p name)
      (sm:sequence*
       (%switch-context-action p)
       (sm:goto (name "semantics_blocking_switch_context")
                (arguments (list (string-append "switch="
                                                (port-prefix "port" p)))))))
    (define (reordered p name)
      (let ((interface (.type p)))
        (sm:sequence*
         `(,(sm:if* (sm:not*
                     (sm:invoke (%nil-predicate interface)
                                (sm:construct (%port-predicate p)
                                              "reply")))
                    (sm:invoke (%reply-reordered-action p)
                               ;; FIXME constructor?
                               (port-prefix "port (reply)" p)))
           ,@(let ((port p))
               (if (not (find (compose (cute ast:eq? port <>) .port)
                              (ast:shared* o)))
                   '()
                   `(,(sum-state-action port))))
           ,(sm:union*
             (sm:goto (name "semantics_main"))
             (defer-qout-actions-async o name))))))
    (define (provides-reply p name)
      (let ((interface (.type p)))
        (sm:sum (type (%replies interface))
                (statement
                 (sm:sequence*
                  (sm:invoke (%reply-action p))
                  (sm:goto (name name)
                           (arguments
                            (list
                             (sm:is*
                              "released"
                              (string-append
                               "released <| "
                               ;; XXX FIXME: port-constructor?
                               (port-prefix "port" p)))
                             (sm:is*
                              "reply"
                              (sm:entity
                               (name "replies")
                               (formals
                                (map
                                 (lambda (i)
                                   (if (ast:eq? i p) "i"
                                       (port-prefix "port (reply)" i)))
                                 provides))))))))))))
    ;; at the end of the component behavior =>
    ;;   when not all blocked ports are released =>
    ;;     goto blocked rtc or skip blocked (qout
    ;;   when all blocked ports are released =>
    ;;     if QMT perform reordered reply => allow defer or goto main
    ;;     else (!QMT) goto skip blocked
    ;; or handle provides synchronous out and reply
    ;; or collateral block => switch context
    (sm:process
      (name "semantics_blocking")
      (formals %blocking-formals)
      (statement
       (sm:union*
        `(,(sm:if* "blocking - list2set (released) != {}"
                   (sm:sequence*
                    (%end-action o)
                    (sm:union*
                     (sm:sequence*
                      %queue-empty-action
                      (sm:goto (name "semantics_blocked_rtc") (arguments '())))
                     (sm:sequence*
                      %queue-not-empty-action
                      (sm:goto (name "semantics_skip_blocked") (arguments '()))))))
          ,(sm:if* "blocking - list2set (released) == {}"
                   (sm:sequence*
                    (%end-action o)
                    (sm:union*
                     (sm:sequence* %queue-empty-action
                                   (sm:union*
                                    `(,(sm:goto (name "semantics_skip_blocked")
                                                (arguments '()))
                                      ,@(map (cute reordered <> name) provides))))
                     (sm:sequence* %queue-not-empty-action
                                   (sm:goto (name "semantics_skip_blocked")
                                            (arguments '()))))))
          ,@(append-map (seq (cute provides-reply <> name)
                             (cute out-actions <> name))
                        provides)
          ,@(map (cute blocking-switch-context-actions+switch=p <> name) requires)))))))

(define-method (semantics-skip-blocking (o <component>))
  (let ((provides (ast:provides-port* o))
        (requires (ast:requires-port* o)))
    (define (handle-provides-replies p name)
      (let ((interface (.type p)))
        (list
         (sm:sum (type (%replies interface))
                 (statement
                  (sm:sequence*
                   (sm:invoke (%reply-action p))
                   (sm:goto (name name)
                            (arguments
                             (list
                              (sm:is*
                               "released"
                               (string-append
                                "released <| "
                                ;; XXX FIXME: port-constructor?
                                (port-prefix "port" p)))
                              (sm:is*
                               "reply"
                               (sm:entity
                                (name "replies")
                                (formals
                                 (map
                                  (lambda (i)
                                    (if (ast:eq? i p) "i"
                                        (port-prefix "port (reply)" i)))
                                  provides))))))))))
         (sm:if* (simple-format
                  #f
                  "released != [] && ~a == head (released)"
                  ;; XXX FIXME: port-constructor?
                  (port-prefix "port" p))
                 (sm:sequence*
                  (%end-action o)
                  (sm:union*
                   `(,(sm:sequence*
                       `(,%queue-empty-action
                         ,(sm:invoke (%reply-reordered-action p)
                                     (sm:construct (%port-predicate p)
                                                   "reply"))
                         ,@(let ((port p))
                             (if (not (find (compose (cute ast:eq? port <>) .port)
                                            (ast:shared* o)))
                                 '()
                                 `(,(sum-state-action port))))
                         ,(sm:union*
                           (sm:goto (name "semantics_main"))
                           (sm:sum (type (%locals o))
                                   (statement
                                    (sm:sequence*
                                     (sm:invoke (%defer-qout-action o))
                                     (sm:goto (name "semantics_async")
                                              (arguments '("blocking={}"
                                                           "port=no_port")))))))))
                     ,@(append-map (seq (cute switch-context-actions <> name)
                                        (cute qout-actions <> name))
                                   requires))))))))
    (define (handle-no-provides-reply name)
      (sm:if* (sm:not*
               (list ;;; XXX TODO sm:group*?
                ;; XXX TODO sm:or*?
                (string-join
                 (map
                  (lambda (p)
                    (simple-format
                     #f
                     "released != [] && ~a == head (released)"
                     (port-prefix "port" p)))
                  provides)
                 " || ")))
              (sm:sequence* (%end-action o)
                            (sm:union*
                             `(,(sm:goto (name "semantics_blocked_rtc")
                                         (arguments '()))
                               ,@(append-map
                                  (seq (cute switch-context-actions <> name)
                                       (cute qout-actions <> name))
                                  requires))))))
    ;; when at the end of the component behavior
    ;;   if reply seen and QMT => reordered (switch context or goto main or defer qout)
    ;;                            or handle requires qout
    ;;   if not reply seen => switch context or goto blocked_rtc
    ;;                        or handle requires qout
    ;; or handle requires qout and reply
    (sm:process
      (name "semantics_skip_blocked")
      (formals %blocking-formals)
      (statement
       (sm:union*
        `(,@(append-map (cute handle-provides-replies <> name) provides)
          ,@(map (cute out-actions <> name) provides)
          ,(handle-no-provides-reply name)
          ,@(append-map (seq (cute switch-context-actions <> name)
                             (cute qout-actions <> name))
                        requires)
          ,(defer-qout-actions o name)
          ,(sm:sequence* %defer-end-action
                         (sm:goto (name name) (arguments '())))))))))

(define-method (semantics-blocked (o <component>))
  (let* ((provides (ast:provides-port* o))
         (requires (ast:requires-port* o))
         (shared (map .port (ast:shared* o)))
         (shared-provides (lset-intersection ast:eq? provides shared)))
    (define (handle-queue-empty name)
      (sm:sequence*
       %queue-empty-action
       (sm:if* "released != []"
               (sm:goto (name "semantics_reply")
                        (arguments '()))
               (sm:goto (name "semantics_blocked_rtc")
                        (arguments '())))))
    (define (handle-blocked-port-reply p name)
      (let ((interface (.type p)))
        (sm:if* (sm:invoke (%nil-predicate interface)
                           (sm:construct (%port-predicate p) "reply"))
                (sm:sum (type (%replies interface))
                        (statement
                         (sm:sequence*
                          (sm:invoke (%reply-action p))
                          (sm:goto (name name)
                                   (arguments
                                    (list
                                     (sm:is*
                                      "released"
                                      (string-append
                                       "released <| "
                                       ;; XXX FIXME: port-constructor?
                                       (port-prefix "port" p)))
                                     (sm:is*
                                      "reply"
                                      (sm:entity (name "replies")
                                                 (formals
                                                  (map
                                                   (lambda (i)
                                                     (if (ast:eq? i p) "i"
                                                         (port-prefix "port (reply)" i)))
                                                   provides))))))))))
                (sm:sum (type (%replies interface))
                        (statement
                         (sm:sequence*
                          (sm:invoke (%reply-action p))
                          %second-reply-action
                          %sm:delta))))))
    (define (handle-port-blocking p name)
      (sm:sequence*
       (%blocking-action p)
       (sm:goto (name name)
                (arguments
                 (list
                  "rtc=no_port"
                  (sm:is* "blocking"
                          (sm:plus* "blocking"
                                    ;; XXX FIXME: port-constructor?
                                    (sm:set* (port-prefix "port" p)))))))))
    ;; nothing blocked => goto main or defer qout
    ;; when at end of behavior and queue_empty
    ;;    if nothing released => goto blocked_rtc
    ;;    else                => goto reply
    ;;    or handle requires qout
    ;;    or handle switch context
    (sm:process
      (name "semantics_blocked")
      (formals %blocked-formals)
      (statement
       (sm:union*
        `(,(sm:if* "blocking == {} && released == []"
                   (sm:sequence*
                    `(,@(map sum-state-action shared-provides)
                      ,(sm:union*
                        (sm:goto (name "semantics_main"))
                        (defer-qout-actions-async o name)))))
          ,(sm:sequence*
            (%end-action o)
            (sm:union*
             `(,(handle-queue-empty name)
               ,@(append-map (seq (cute qout-actions <> name)
                                  (cute switch-context-actions <> name))
                             requires))))
          ,@(append-map (seq (cute out-actions <> name)
                             (cute handle-blocked-port-reply <> name)
                             (cute handle-port-blocking <> name))
                        provides)
          ,@(append-map (seq (cute qout-actions <> name)
                             (cute switch-context-actions <> name))
                        requires)
          ,(sm:sequence* %defer-end-action
                         (sm:goto (name name) (arguments '())))))))))

(define-method (semantics-reply (o <component>))
  (let ((provides (ast:provides-port* o)))
    (define (semantics-reply p name)
      (let ((interface (.type p)))
        (sm:if* (sm:or*
                 (sm:and*
                  (sm:equal* "rtc" (port-prefix "port" p))
                  (sm:not*
                   (sm:invoke (%nil-predicate interface)
                              (sm:construct (%port-predicate p)
                                            "reply"))))
                 (sm:and*
                  ;; FIXME: no_port predicate?
                  (sm:equal* "rtc" "no_port")
                  (sm:not-equal* "released" "[]")
                  (sm:equal*
                   ;; FIXME: constructor?
                   (port-prefix "port" p)
                   "head (released)")))
                (sm:sequence*
                 `(,(sm:invoke (%reply-reordered-action p)
                               (sm:construct (%port-predicate p)
                                             "reply"))
                   ,@(let ((port p))
                       (if (not (find (compose (cute ast:eq? port <>) .port)
                                      (ast:shared* o)))
                           '()
                           `(,(sum-state-action port))))
                   ,(let ((arguments
                           (list
                            (sm:is*
                             "blocking"
                             (sm:minus*
                              "blocking"
                              (sm:set* (port-prefix "port" p))))
                            (simple-format
                             #f
                             "released=remove_port (~a, released)"
                             (port-prefix "port" p))
                            (sm:is* "reply"
                                    (sm:entity
                                     (name "replies")
                                     (formals
                                      (map
                                       (lambda (i)
                                         (if (ast:eq? i p) (model-prefix
                                                            "nil" interface)
                                             (port-prefix "port (reply)" i)))
                                       provides)))))))
                      (sm:union*
                       (sm:goto (name name)
                                (arguments (cons
                                            "rtc=no_port"
                                            arguments)))
                       (sm:goto (name "semantics_blocked_rtc")
                                (arguments arguments)))))))))
    (sm:process
      (name "semantics_reply")
      (formals %blocked-formals)
      (statement
       (sm:union*
        `(,(sm:if* "blocking == {} && released == []"
                   (sm:goto (name "semantics_blocked")
                            (arguments '())))
          ,(sm:if* "blocking - list2set (released) != {} && rtc == no_port"
                   (sm:goto (name "semantics_blocked_rtc")
                            (arguments '())))
          ,@(map (cute semantics-reply <> name) provides)))))))

;;TODO refactor: extract commonality in main and blocked-main
(define-method (semantics-blocked-main (o <component>))
  (let* ((provides (ast:provides-port* o))
         (requires (ast:requires-port* o))
         (shared (map .port (ast:shared* o)))
         (shared-provides (lset-intersection ast:eq? provides shared)))
    (define (blocked-main-provides p name)
      (let ((interface (.type p)))
        (sm:if* (sm:not* (sm:in* (port-prefix "port" p) "blocking"))
                (sm:sequence*
                 (sm:sum (type (%actions interface))
                         (statement
                          (sm:sequence*
                           (sm:invoke (%in-action p)
                                      #:keep-constructor? #t)
                           (sm:goto
                            (name "semantics_blocked")
                            (arguments
                             (list
                              (sm:is*
                               "rtc"
                               (port-prefix "port" p))))))))))))
    (define (blocked-main-requires p name)
      (let ((interface (.type p)))
        (sm:if* "released == []"
                (sm:sum (type (%modeling interface))
                        (statement
                         (sm:sequence*
                          `(,(sm:invoke (%internal-action p))
                            ,(%end-action p)
                            ,(sm:goto
                              (name "semantics_blocked")
                              (arguments '("rtc=no_port"))))))))))
    (sm:process
      (name "semantics_blocked_rtc")
      (formals %blocking-formals)
      (statement
       (sm:union*
        (sm:if* "blocking == {}"
                (sm:sequence*
                 `(,@(map sum-state-action shared-provides)
                   ,(sm:union* (sm:goto (name "semantics_main"))
                               (defer-qout-actions-async o name)))))
        (sm:if* "blocking != {}"
                (sm:union*
                 `(,@(map (cute blocked-main-provides <> name) provides)
                   ,@(map (cute blocked-main-requires <> name) requires)
                   ,(sm:sum (type (%locals o))
                            (statement
                             (sm:sequence*
                              (sm:invoke (%defer-qout-action o))
                              (sm:goto (name "semantics_blocked")
                                       (arguments '("rtc=no_port"))))))))))))))

(define-method (semantics-async-modeling+switch-context (o <component>))
  (let ((requires (ast:requires-port* o)))
    (sm:process
      (name "semantics_async_switch_context")
      (formals `(,@%blocking-formals
                 "port: provides_ports"
                 "switch: requires_ports"))
      (statement
       (sm:union*
        `(,%sm:delta
          ,@(append-map
             (cute modeling+switch-context <> name "semantics_async")
             requires)))))))

(define-method (semantics-async (o <component>))
  (let* ((provides (ast:provides-port* o))
         (requires (ast:requires-port* o))
         (shared (map .port (ast:shared* o)))
         (shared-provides (lset-intersection ast:eq? provides shared)))
    (define (requires-flush p name)
      (let ((interface (.type p)))
        (sm:if* (sm:invoke (%port-predicate p) "port")
                (sm:sequence*
                 `(,(%flush-action p)
                   ,@(map sum-state-action shared-provides)
                   ,(sm:union*
                     (sm:goto (name "semantics_main"))
                     (defer-qout-actions-async o name)))))))
    ;; XXX out-actions, but with port=h'port
    (define (out-actions+port=p p name)
      (let ((interface (.type p)))
        (sm:sum (type (%actions interface))
                (statement
                 (sm:sequence*
                  (sm:invoke (%out-action p) #:keep-constructor? #t)
                  (if (not (find (compose (cute ast:eq? p <>) .port)
                                 (ast:shared* o)))
                      (sm:goto (name name)
                               (arguments
                                (list
                                 (sm:is* "port" (port-prefix "port" p)))))
                      (sum-state-action
                       p (sm:goto
                          (name name)
                          (arguments
                           (list
                            (sm:is* "port" (port-prefix "port" p))))))))))))
    ;; XXX switch-context-actions but goto
    ;; semantics_async_switch_context with switch=h'port
    (define (switch-context-actions+switch=p p name)
      (sm:sequence*
       (%switch-context-action p)
       (sm:goto (name "semantics_async_switch_context")
                (arguments
                 (list (sm:is* "switch" (port-prefix "port" p)))))))
    (let* ((provides (ast:provides-port* o))
           (shared (map .port (ast:shared* o)))
           (shared-provides (lset-intersection ast:eq? provides shared)))
      (sm:process
        (name "semantics_async")
        (formals `(,@%blocking-formals
                   "port: provides_ports"))
        (statement
         (sm:union*
          `(,(sm:sequence*
              (%end-action o)

              (sm:union*
               (sm:sequence*
                `(,%queue-empty-action
                  ,@(map sum-state-action shared-provides)
                  ,(sm:union*
                    `(,(sm:if* (sm:invoke %no-port-predicate "port")
                               (sm:union* (sm:goto (name "semantics_main"))
                                          (sm:sequence*
                                           (defer-qout-actions o name))))
                      ,@(map (cute requires-flush <> name) provides)))))
               (sm:sequence* %queue-not-empty-action
                             (sm:goto (name name) (arguments '())))))
            ,(sm:sequence* %defer-end-action
                           (sm:goto (name name) (arguments '())))
            ,@(map (cute out-actions+port=p <> name) provides)
            ,@(append-map (seq (cute switch-context-actions+switch=p <> name)
                               (cute qout-actions <> name))
                          requires))))))))

(define-method (component-semantics-processes (o <component>))
  (let* ((provides (ast:provides-port* o))
         (requires (ast:requires-port* o))
         (shared (map .port (ast:shared* o)))
         (shared-requires (lset-intersection ast:eq? requires shared))
         (component-semantics-parallel
          (sm:process (name "component_semantics_parallel")
                      (statement
                       (sm:parallel* (sm:goto (name "component_behavior"))
                                     (sm:goto (name "semantics_main"))))))
         (component-semantics-comm
          (sm:process
            (name "component_semantics_comm")
            (statement
             (sm:comm
              (process component-semantics-parallel)
              (events
               (append
                (list
                 (sm:comm-event (from (sm:multi-event
                                       (events
                                        (list %defer-end-action)))))
                 (sm:comm-event (from (sm:multi-event
                                       (events
                                        (list (%defer-qout-action o))))))
                 (sm:comm-event (from (sm:multi-event
                                       (events
                                        (list (%end-action o)))))))
                (append-map
                 (lambda (p)
                   (list
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%in-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%out-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%blocking-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%reply-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%state-action p))))))))
                 provides)
                (append-map
                 (lambda (p)
                   (list
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%qout-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%switch-context-action p))))))))
                 requires)
                (append-map
                 (lambda (p)
                   (list
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%state-action p))))))))
                 ;;TODO simplify + filter non external?
                 shared-requires)))))))
         (interfaces (ast:interface* o))
         (component-semantics-allow
          (sm:process
            (name "component_semantics_allow")
            (statement
             (sm:allow
              (process component-semantics-comm)
              (events (cons* %constrained-legal-action
                             %constrained-illegal-action
                             %declarative-illegal-action
                             (%defer-skip-action o)
                             %illegal-action
                             %missing-reply-action
                             %queue-empty-action
                             %queue-full-action
                             %queue-not-empty-action
                             %range-error-action
                             %recurse-action
                             %return-action
                             %second-reply-action
                             %tag-action
                             %tau-void-action
                             (append
                              (append-map
                               (lambda (i)
                                 (list
                                  (%tau-event-action i)
                                  (%tau-modeling-action i)
                                  (%tau-reply-action i)))
                               interfaces)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%reply-reordered-action p)
                                  (%flush-action p)))
                               provides)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%flush-action p)
                                  (%in-action p)
                                  (%internal-action p)
                                  (%end-action p)
                                  (%reply-action p)
                                  (%state-action p)))
                               requires))))))))
         (component-semantics
          (sm:process
            (name "component_semantics")
            (statement
             (sm:rename
              (process component-semantics-allow)
              (events
               (append
                (append-map
                 (lambda (p)
                   (let ((interface (.type p)))
                     (list
                      (sm:rename-event (from (sm:transpose-tick (%reply-action p)))
                                       (to (%tau-reply-action interface)))
                      (sm:rename-event (from (%reply-reordered-action p))
                                       (to (%reply-action p))))))
                 provides)
                (sm:comm-events (sm:process-statement
                                 component-semantics-comm)))))))))

    ;; when QMT, i.e. RTC => allow defer qout
    (list (semantics-main o)
          (sm:comment* "% semantics main\n")

          (semantics-unblocked o)
          (sm:comment* "% semantics_unblocked\n")

          (unblocked-semantics-modeling+switch-context o)
          (sm:comment* "% semantics_unblocked switch-context\n")

          (semantics-blocking o)
          (sm:comment* "% semantics_blocking\n")

          (blocking-semantics-modeling+switch-context o)
          (sm:comment* "% semantics_blocking switch-context\n")

          (semantics-skip-blocking o)
          (sm:comment* "% semantics_skip blocked\n")

          (semantics-blocked o)
          (sm:comment* "% semantics_blocked\n")

          (semantics-reply o)
          (sm:comment* "% semantics reply\n")

          (semantics-blocked-main o)
          (sm:comment* "% semantics blocked rtc\n")

          (semantics-async o)
          (sm:comment* "% semantics async\n")

          (semantics-async-modeling+switch-context o)
          (sm:comment* "% semantics async switch-context\n")

          component-semantics-parallel
          component-semantics-comm
          component-semantics-allow
          component-semantics)))

(define-method (component-assembly-processes (o <component>))
  (let* ((provides (ast:provides-port* o))
         (requires (ast:requires-port* o))
         (component-assembly-parallel
          (sm:process (name "component_assembly_parallel")
                      (statement
                       (sm:parallel* (sm:goto (name "component_constrained"))
                                     (sm:goto (name "req_and_queue"))))))
         (component-assembly-comm
          (sm:process
            (name "component_assembly_comm")
            (statement
             (sm:comm
              (process component-assembly-parallel)
              (events
               (cons*
                (sm:comm-event (from (sm:multi-event
                                      (events
                                       (list %queue-empty-action)))))
                (sm:comm-event (from (sm:multi-event
                                      (events
                                       (list %queue-not-empty-action)))))
                (append
                 (append-map
                  (lambda (p)
                    (list
                     (sm:comm-event (from (sm:multi-event
                                           (events
                                            (list (%flush-action p))))))
                     (sm:comm-event (from (sm:multi-event
                                           (events
                                            (list (%in-action p))))))
                     (sm:comm-event (from (sm:multi-event
                                           (events
                                            (list (%reply-action p))))))
                     (sm:comm-event (from (sm:multi-event
                                           (events
                                            (list (%internal-action p))))))
                     (sm:comm-event (from (sm:multi-event
                                           (events
                                            (list (%qout-action p))))))
                     (sm:comm-event (from (sm:multi-event
                                           (events
                                            (list (%end-action p))))))))
                  requires)
                 (map
                  (lambda (p)
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%state-action p)))))))
                  (filter (negate ast:external?) requires)))))))))
         (interfaces (ast:interface* o))
         (component-assembly-allow
          (sm:process
            (name "component_assembly_allow")
            (statement
             (sm:allow
              (process component-assembly-comm)
              (events (cons* %constrained-legal-action
                             %declarative-illegal-action
                             %defer-end-action
                             (%defer-qout-action o)
                             (%defer-skip-action o)
                             %illegal-action
                             %missing-reply-action
                             %non-compliant-action
                             %queue-full-action
                             %range-error-action
                             %recurse-action
                             %return-action
                             %second-reply-action
                             %tag-action
                             %tau-void-action
                             (%end-action o)
                             (append
                              (append-map
                               (lambda (i)
                                 (list
                                  (%tau-event-action i)
                                  (%tau-modeling-action i)
                                  (%tau-reply-action i)))
                               interfaces)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%in-action p)
                                  (%blocking-action p)
                                  (%compliant-action p)
                                  (%out-action p)
                                  (%reply-action p)
                                  (%flush-action p)
                                  (%state-action p)))
                               provides)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%qin-action p)
                                  (%port-queue-full-action p)
                                  (%switch-context-action p)))
                               requires)
                              (filter-map (conjoin ast:external? %state-action)
                                          requires))))))))
         (component-assembly-rename
          (sm:process
            (name "component_assembly_rename")
            (statement
             (sm:rename
              (process component-assembly-allow)
              (events
               (sm:comm-events (sm:process-statement component-assembly-comm)))))))
         (component-compliant
          (sm:process
            (name "component_compliant")
            (statement
             (sm:union*
              (append
               (map (lambda (p)
                      (sm:sum
                       (type (%actions (.type p)))
                       (statement
                        (sm:sequence*
                         (sm:invoke (%in-action p) #:keep-constructor? #t)
                         (sm:goto (name name))))))
                    provides)
               (map (lambda (p)
                      (sm:sum
                       (type (%actions (.type p)))
                       (statement
                        (sm:sequence*
                         (sm:invoke (%out-action p) #:keep-constructor? #t)
                         (sm:union* (sm:sequence* (%compliant-action p)
                                                  (sm:goto (name name)))
                                    (sm:goto (name "Non_Compliance")))))))
                    provides)
               (map (lambda (p)
                      (sm:sum
                       (type (%replies (.type p)))
                       (statement
                        (sm:sequence*
                         (sm:invoke (%reply-action p))
                         (sm:union* (sm:sequence* (%compliant-action p)
                                                  (sm:goto (name name)))
                                    (sm:goto (name "Non_Compliance")))))))
                    provides)
               (map (lambda (p)
                      (sm:sum
                       (type (%modeling (.type p)))
                       (statement
                        (sm:sequence*
                         (sm:invoke (%internal-action p))
                         (sm:goto (name name))))))
                    requires))))))
         (component-compliant-parallel
          (sm:process (name "component_compliant_parallel")
                      (statement
                       (sm:parallel* (sm:goto (name "component_assembly_rename"))
                                     (sm:goto (name "component_compliant"))))))
         (component-compliant-comm
          (sm:process
            (name "component_compliant_comm")
            (statement
             (sm:comm
              (process component-compliant-parallel)
              (events
               `(,@(map
                    (lambda (p)
                      (sm:comm-event (from (sm:multi-event
                                            (events
                                             (list (%compliant-action p)))))))
                    provides)
                 ,(sm:comm-event (from (sm:multi-event
                                        (events
                                         (list %non-compliant-action)))))
                 ,@(map
                    (lambda (p)
                      (sm:comm-event (from (sm:multi-event
                                            (events
                                             (list (%in-action p)))))))
                    provides)
                 ,@(map
                    (lambda (p)
                      (sm:comm-event (from (sm:multi-event
                                            (events
                                             (list (%out-action p)))))))
                    provides)
                 ,@(map
                    (lambda (p)
                      (sm:comm-event (from (sm:multi-event
                                            (events
                                             (list (%reply-action p)))))))
                    provides)
                 ,@(map
                    (lambda (p)
                      (sm:comm-event (from
                                      (sm:multi-event
                                       (events
                                        (list (%internal-action p)))))))
                    requires)))))))
         (component-compliant-allow
          (sm:process
            (name "component_compliant_allow")
            (statement
             (sm:allow
              (process component-compliant-comm)
              (events (cons* %constrained-legal-action
                             %declarative-illegal-action
                             %defer-end-action
                             (%defer-qout-action o)
                             (%defer-skip-action o)
                             %illegal-action
                             %missing-reply-action
                             %queue-empty-action
                             %queue-full-action
                             %queue-not-empty-action
                             %range-error-action
                             %recurse-action
                             %return-action
                             %second-reply-action
                             %tag-action
                             %tau-void-action
                             (%end-action o)
                             (append
                              (append-map
                               (lambda (i)
                                 (list
                                  (%tau-event-action i)
                                  (%tau-modeling-action i)
                                  (%tau-reply-action i)))
                               interfaces)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%blocking-action p)
                                  (%flush-action p)
                                  (%state-action p)))
                               provides)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%end-action p)
                                  (%flush-action p)
                                  (%in-action p)
                                  (%port-queue-full-action p)
                                  (%qin-action p)
                                  (%qout-action p)
                                  (%reply-action p)
                                  (%state-action p)
                                  (%switch-context-action p)))
                               requires)
                              (filter-map (conjoin ast:external? %state-action)
                                          requires))))))))
         (component-compliant-rename
          (sm:process
            (name "component_compliant_rename")
            (statement
             (sm:rename
              (process component-compliant-allow)
              (events
               (sm:comm-events (sm:process-statement component-compliant-comm)))))))
         (component
          (sm:process
            (name "component")
            (statement
             (sm:hide
              (process (if (%no-constraint?) component-assembly-rename
                           component-compliant-rename))
              (events (cons* %constrained-legal-action
                             %defer-end-action
                             (%defer-skip-action o)
                             %queue-empty-action
                             %queue-not-empty-action
                             %recurse-action
                             %return-action
                             %tau-void-action
                             (%end-action o)
                             (append
                              (map %compliant-action provides)
                              (append-map
                               (lambda (i)
                                 (list
                                  (%tau-event-action i)
                                  (%tau-modeling-action i)
                                  (%tau-reply-action i)))
                               interfaces)
                              (map %state-action provides)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%end-action p)
                                  (%switch-context-action p)))
                               requires)))))))))
    (list component-assembly-parallel
          component-assembly-comm
          component-assembly-allow
          component-assembly-rename
          component-compliant
          component-compliant-parallel
          component-compliant-comm
          component-compliant-allow
          component-compliant-rename
          component)))

(define-method (interface-constraint-processes (o <interface>))
  (let ((constraint (makreel:constraint o)))
    (list
     (sm:comment* constraint))))

(define-method (provides-constraint-processes (o <port>))
  (let* ((interface (.type o))
         (provides-constraint
          (sm:process
            (name (port-prefix "provides_constraint" o))
            (statement
             (sm:rename
              (process (model-prefix "constraint_start" interface))
              (events
               (list
                (sm:rename-event (from (%compliant-action #f))
                                 (to (%compliant-action o)))
                (sm:rename-event (from (%flush-action interface))
                                 (to (%flush-action o)))
                (sm:rename-event (from (%in-action interface))
                                 (to (%in-action o)))
                (sm:rename-event (from (%out-action interface))
                                 (to (%out-action o)))
                (sm:rename-event (from (%reply-action interface))
                                 (to (%reply-action o)))
                (sm:rename-event (from (%state-action interface))
                                 (to (%state-action o))))))))))
    (list provides-constraint)))

(define-method (constraint-processes (o <component>))
  (let* ((provides (ast:provides-port* o))
         (provides-port (ast:provides-port o))
         (requires (ast:requires-port* o))
         (provides-constraint-parallel
          (sm:process
            (name "provides_constraint_parallel")
            (statement
             (sm:parallel
              (processes
               (map
                (lambda (p)
                  (sm:goto (name (port-prefix "provides_constraint" p))))
                provides))))))
         (provides-constraint
          (if (= (length provides) 1)
              (sm:process
                (name "provides_constraint")
                (statement
                 (sm:goto (name "provides_constraint_parallel"))))
              (sm:process
                (name "provides_constraint")
                (statement
                 (sm:allow
                  (process provides-constraint-parallel)
                  (events (cons* %constrained-legal-action
                                 %constrained-illegal-action
                                 %non-compliant-action
                                 (append-map
                                  (lambda (p)
                                    (list
                                     (%compliant-action p)
                                     (%flush-action p)
                                     (%in-action p)
                                     (%out-action p)
                                     (%reply-action p)
                                     (%state-action p)))
                                  provides))))))))
         (component-constrained-parallel
          (sm:process (name "component_constrained_parallel")
                      (statement
                       (sm:parallel* (sm:goto (name "component_semantics"))
                                     (sm:goto (name "provides_constraint"))))))
         (component-constrained-comm
          (sm:process
            (name "component_constrained_comm")
            (statement
             (sm:comm
              (process component-constrained-parallel)
              (events
               (cons*
                (sm:comm-event (from (sm:multi-event
                                      (events
                                       (list %constrained-legal-action)))))
                (sm:comm-event (from (sm:multi-event
                                      (events
                                       (list %constrained-illegal-action)))))
                (append-map
                 (lambda (p)
                   (list
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%flush-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%in-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%out-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%reply-action p))))))
                    (sm:comm-event (from (sm:multi-event
                                          (events
                                           (list (%state-action p))))))))
                 provides)))))))
         (provides-interfaces (map .type provides))
         (provides-interfaces (delete-duplicates provides-interfaces ast:eq?))
         (interfaces (ast:interface* o))
         (component-constrained-allow
          (sm:process
            (name "component_constrained_allow")
            (statement
             (sm:allow
              (process component-constrained-comm)
              (events (cons* %declarative-illegal-action
                             %defer-end-action
                             (%defer-qout-action o)
                             (%defer-skip-action o)
                             %illegal-action
                             %missing-reply-action
                             %non-compliant-action
                             %queue-empty-action
                             %queue-full-action
                             %queue-not-empty-action
                             %range-error-action
                             %recurse-action
                             %return-action
                             %second-reply-action
                             %tag-action
                             %tau-void-action
                             (%end-action o)
                             (append
                              (append-map
                               (lambda (i)
                                 (list
                                  (%tau-event-action i)
                                  (%tau-modeling-action i)
                                  (%tau-reply-action i)))
                               interfaces)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%blocking-action p)
                                  (%compliant-action p)))
                               provides)
                              (append-map
                               (lambda (p)
                                 (list
                                  (%flush-action p)
                                  (%in-action p)
                                  (%reply-action p)
                                  (%internal-action p)
                                  (%end-action p)
                                  (%qout-action p)
                                  (%state-action p)
                                  (%switch-context-action p)))
                               requires))))))))
         (component-constrained
          (sm:process
            (name "component_constrained")
            (statement
             (sm:rename
              (process component-constrained-allow)
              (events
               (cons*
                (sm:rename-event
                 (from (sm:transpose-tick %constrained-illegal-action))
                 (to %declarative-illegal-action))
                (sm:comm-events (sm:process-statement
                                 component-constrained-comm)))))))))
    (list provides-constraint-parallel
          provides-constraint
          component-constrained-parallel
          component-constrained-comm
          component-constrained-allow
          component-constrained)))

(define-method (unconstrained-processes (o <component>))
  (list
   (sm:comment* "% actually unconstrained")
   (sm:process
     (name "component_constrained")
     (statement
      (sm:block
       (process "component_semantics")
       (events (list %constrained-illegal-action)))))))

(define-method (model->scmackerel (o <component>))
  (makreel:proc-list (.behavior o))
  (parameterize ((%model-name (makreel:full-name o)))
    (let* ((ports (ast:port* o))
           (requires (ast:requires-port* o))
           (port-processes (append-map port-processes ports))
           (provides (ast:provides-port* o))
           (provides-interfaces (map .type provides))
           (provides-interfaces (delete-duplicates provides-interfaces ast:eq?))
           (provides-processes (provides-processes o))
           (behavior (.behavior o))
           (enums (filter (is? <enum>) (ast:type* behavior)))
           (enums (map enum->scmackerel enums))
           (members (filter (negate (is? <shared-variable>)) (ast:variable* o)))
           (behavior-processes (behavior->processes o))
           (functions (makreel:called-function* behavior))
           (function-processes (append-map function->processes functions))
           (return-processes (return-processes o))
           (defer-processes (append-map defer->processes (makreel:defer* o)))
           (component-defer-processes (component-defer-processes o))
           (q-processes
            (if (null? requires) (component-no-q-processes o)
                (component-q-processes o)))
           (semantics-processes (component-semantics-processes o))
           (interface-constraint-processes
            (if (%no-constraint?) '()
                (append-map interface-constraint-processes provides-interfaces)))
           (provides-constraint-processes
            (if (%no-constraint?) '()
                (append-map provides-constraint-processes provides)))
           (constraint-processes
            (if (%no-constraint?) (unconstrained-processes o)
                (constraint-processes o)))
           (static-defer-processes
            (file-comments "defer.mcrl2"))
           (component-assembly-processes (component-assembly-processes o))
           (return-type (%return-type o))
           (returns-type (%returns-type o))
           (processes `(,(makreel:caption "PORT PROCESSES")
                        ,@port-processes
                        ,(makreel:caption "PROVIDES")
                        ,@provides-processes
                        ,(makreel:caption "COMPONENT")
                        ,@static-defer-processes
                        ,@behavior-processes
                        ,(makreel:caption "FUNCTIONS")
                        ,@(if (not return-type) '()
                              (list return-type))
                        ,(%stack-type o)
                        ,@(if (not returns-type) '()
                              (list returns-type))
                        ,@function-processes
                        ,@return-processes
                        ,(makreel:caption "DEFER")
                        ,@defer-processes
                        ,@component-defer-processes
                        ,(makreel:caption "Q")
                        ,@q-processes
                        ,(makreel:caption "SEMANTICS")
                        ,@semantics-processes
                        ,@interface-constraint-processes
                        ,(makreel:caption "PROVIDES CONSTRAINT")
                        ,@provides-constraint-processes
                        ,(makreel:caption "COMPONENT CONSTRAINED")
                        ,@constraint-processes
                        ,(makreel:caption "COMPONENT ASSEMBLY")
                        ,@component-assembly-processes))
           (provides-ports-type
            (sm:type (name "provides_ports")
                     (entities (cons* %no-port-predicate
                                      (map (lambda (p)
                                             (%port-predicate p))
                                           provides)))))
           (requires-ports (ast:requires-port* o))
           (requires-ports-type
            (sm:type (name "requires_ports")
                     (entities (cons* "no_requires"
                                      (map (cute port-prefix "port" <>)
                                           requires)))))
           (requires-events-type
            (sm:type
             (name "requires_events")
             (entities
              (cons*
               "no_type"
               (map
                (lambda (p)
                  (let ((interface (.type p)))
                    ;; FIXME: predicate name + type?
                    (sm:predicate
                     (name (port-prefix "is" p))
                     (type (sm:entity
                            (name (port-prefix "events" p))
                            (formals
                             (list (sm:formal
                                    (type (model-prefix "events" interface))
                                    (name (port-prefix "projection" p))))))))))
                requires)))))
           (state-type (%state o))
           (selection-type (sm:type (name "Selection")
                                    (entities
                                     (cons* "selection_empty"
                                            (map
                                             (cute variable-prefix "select" <>)
                                             members)))))
           (queue-size-equation (sm:equation (left "QueueLength")
                                             (right (%queue-size))))
           (queue-size-defer-equation (sm:equation (left "DeferQueueLength")
                                                   (right (%queue-size-defer))))
           (queue-size-external-equation (sm:equation (left "ExternalQueueLength")
                                                      (right (%queue-size-external))))
           (state-current-equation
            (sm:equation (left "state_current (current, context)")
                         (right (if (or (null? defer-processes)
                                        (null? members)) "(true)"
                                        (sm:and*
                                         (map variable->current? members)))))))
      (sm:mcrl2 (types `(,provides-ports-type
                         ,(%reply-values o)
                         ,requires-ports-type
                         ,requires-events-type
                         ,@enums
                         ,state-type
                         ,selection-type
                         ,%defer-context-type
                         ,(%locals o)))
                (equations (list
                            state-current-equation
                            queue-size-equation
                            queue-size-defer-equation
                            queue-size-external-equation))
                (processes processes)))))


;;;
;;; Entry point.
;;;
(define (root->scmackerel root)
  (let* ((models (ast:model** root))
         (component (find (is? <component>) models))
         (model-name (or (%model-name) (ast:dotted-name (ast:get-model root))))
         (act-processes (list (dummy-tag-process)))
         (interfaces (if component (filter (is? <interface>) models)
                         (filter (conjoin
                                  (is? <interface>)
                                  (compose (cute equal? <> model-name)
                                           ast:dotted-name))
                                 models)))
         (types-mcrl2
          (map
           interface-types->scmackerel
           (filter (conjoin (is? <interface>)
                            (compose not (cute equal? (%model-name) <>)
                                     ast:dotted-name))
                   models)))
         (interface-mcrl2 (map model->scmackerel interfaces))
         (mcrl2 (reduce merge-scmackerels
                        (sm:mcrl2)
                        (append types-mcrl2 interface-mcrl2)))
         (component-mcrl2 (and=> component model->scmackerel))
         (mcrl2 (if (not component-mcrl2) mcrl2
                    (merge-scmackerels component-mcrl2 mcrl2)))
         (mcrl2 (if (not component-mcrl2) mcrl2
                    (sm:mcrl2
                      (inherit mcrl2)
                      (processes
                       (cons (makreel:caption "INTERFACE")
                             (sm:mcrl2-processes mcrl2))))))
         (enums (filter (is? <enum>) (ast:type** root)))
         (enums (map enum->scmackerel enums))
         (illegal-processes (illegal-processes))
         (static-equations
          `(,(makreel:generated-comment root)
            ,@(if (not component) '()
                  (file-comments "component.mcrl2")))))
    (sm:mcrl2
      (inherit mcrl2)
      (equations
       (append static-equations
               (sm:mcrl2-equations mcrl2)
               enums))
      (types
       (append (sm:mcrl2-types mcrl2)
               enums))
      (processes
       (append act-processes
               illegal-processes
               (sm:mcrl2-processes mcrl2)
               (list (makreel:version-comment)))))))

(define (scmackerel:display mcrl2)
  (display mcrl2))
