/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: get_args.c,v 1.4 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: get_args.c,v $
** Revision 1.4  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.3  2009/03/18 00:28:29  keith
** Added checks for NDEBUG macro.
**
** Revision 1.2  2009/03/07 03:04:01  keith
** Standardized some error messages.
**
** Revision 1.1  2009/03/07 02:58:39  keith
** Initial revision
**
*/

static void		print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], const char **divvy_fname, unsigned short *max_concurrent_jobs, unsigned short *retries, const char **command, const char **output_name, const char **input_file_pattern, const char **recovery_log, unsigned short *max_failure_cnt, const char **cpu_semkey_fname )
{
	int	ch;
	extern char	*optarg;
	unsigned short	max_forks;

	assert( argv != (char **) 0 );
	assert( divvy_fname != (const char **) 0 );
	assert( max_concurrent_jobs != (unsigned short *) 0 );
	assert( retries != (unsigned short *) 0 );
	assert( command != (const char **) 0 );
	assert( output_name != (const char **) 0 );
	assert( input_file_pattern != (const char **) 0 );
	assert( recovery_log != (const char **) 0 );
	assert( max_failure_cnt != (unsigned short *) 0 );
	assert( cpu_semkey_fname != (const char **) 0 );

	max_forks = (unsigned short)sysconf( _SC_CHILD_MAX ) / (unsigned short)4;

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	/*
	** Initialize return values.
	*/
	*divvy_fname = (const char *)0;
	*max_concurrent_jobs = (unsigned short)0;
	*retries = (unsigned short)0;
	*command = (const char *)0;
	*output_name = (const char *)0;
	*input_file_pattern = (const char *)0;
	*recovery_log = (const char *)0;
	*max_failure_cnt = (unsigned short)USHRT_MAX;
	*cpu_semkey_fname = (const char *)0;

	while ( ( ch = getopt( argc, argv, "c:dl:m:o:f:g:i:r:s:" ) ) != EOF ) {
		switch ( ch ) {
		case 'c':
			*command = optarg;
			break;
		case 'd':
#ifdef NDEBUG
			(void) fputs( "Not compiled for debug.\n", stderr );
#else
			Debug = 1;
#endif
			break;
		case 'f':
			*divvy_fname = optarg;
			break;
		case 'g':
			*max_failure_cnt = (unsigned short) atoi( optarg );
			break;
		case 'i':
			*input_file_pattern = optarg;
			break;
		case 'l':
			*recovery_log = optarg;
			break;
		case 'm':
			*max_concurrent_jobs = (unsigned short) atoi( optarg );
			break;
		case 'o':
			*output_name = optarg;
			break;
		case 'r':
			*retries = (unsigned short) atoi( optarg );
			break;
		case 's':
			*cpu_semkey_fname = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *command == (const char *)0 || strlen( *command ) == (size_t)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Command argument is missing.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *input_file_pattern == (const char *)0 && *divvy_fname == (const char *)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Cannot process zero rows per slice.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *max_concurrent_jobs == (unsigned short)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Max concurrent jobs cannot be zero.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *max_concurrent_jobs > max_forks ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Decrease max_concurrent_jobs (", stderr );
		(void) fprintf( stderr, "%hu", *max_concurrent_jobs );
		(void) fputs( ") to maximum allowable processes (", stderr );
		(void) fprintf( stderr, "%hu", max_forks );
		(void) fputs( ").\n", stderr );

		*max_concurrent_jobs = max_forks;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Max concurrent jobs %hu.\n", *max_concurrent_jobs );
		(void) fprintf( stderr, "Retries %hu.\n", *retries );
		(void) fprintf( stderr, "Max failures %hu.\n", *max_failure_cnt );
		(void) fprintf( stderr, "Command [%s]\n", *command );
		(void) fprintf( stderr, "Output name [%s]\n", ( *output_name == (const char *)0 ) ? "NULL" : *output_name );
		(void) fprintf( stderr, "Divvy name [%s]\n", ( *divvy_fname == (const char *)0 ) ? "NULL" : *divvy_fname );
		(void) fprintf( stderr, "Recovery log [%s]\n", ( *recovery_log == (const char *)0 ) ? "NULL" : *recovery_log );
		(void) fprintf( stderr, "CPU semaphore key file name [%s]\n", ( *cpu_semkey_fname == (const char *)0 ) ? "NULL" : *cpu_semkey_fname );
		(void) fprintf( stderr, "Input file pattern [%s]\n", ( *input_file_pattern == (const char *)0 ) ? "NULL" : *input_file_pattern );
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " [-l file]", stderr );
	(void) fputs( " -m n", stderr );
	(void) fputs( " [-o file]", stderr );
	(void) fputs( " [-r n]", stderr );
	(void) fputs( " [-f file]", stderr );
	(void) fputs( " [-g n]", stderr );
	(void) fputs( " [-i file]", stderr );
	(void) fputs( " [-s file]", stderr );
	(void) fputs( " -c command", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-d -> debug (optional)\n", stderr );
	(void) fputs( "\t-l -> recovery log (optional)\n", stderr );
	(void) fputs( "\t-m -> maximum concurrent jobs (required)\n", stderr );
	(void) fputs( "\t-o -> output file prefix (optional)\n", stderr );
	(void) fputs( "\t-r -> retry attempts for failed processes (optional)\n", stderr );
	(void) fputs( "\t-s -> CPU semaphore key file name (optional)\n", stderr );
	(void) fputs( "\t-f -> file containing already defined partitions (optional)\n", stderr );
	(void) fputs( "\t-g -> maximum job failures allowed (optional)\n", stderr );
	(void) fputs( "\t-i -> file pattern for input files (optional)\n", stderr );
	(void) fputs( "\t-c -> command to execute (required)\n", stderr );
}
