/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>
#include "tbox.h"
#include "dfile_exec.h"

static const char       rcsid[] = "$Id: get_divvy_text.c,v 1.2 2009/10/16 20:05:35 keith Exp $";

/*
** $Log: get_divvy_text.c,v $
** Revision 1.2  2009/10/16 20:05:35  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/07 03:55:32  keith
** Initial revision
**
*/

static int load_file( const char ***, unsigned short *, const char * );
static int construct_divvy( const char ***, unsigned short *, unsigned short );

/*
** This function gets partition definition values.
*/
int get_divvy_text( const char ***divvy_text, unsigned short *divvy_text_cnt, const char *divvy_fname )
{
	static const char	func[] = "get_divvy_text";
	unsigned long	divvy_int;
	size_t	len;
	char	*int_ptr;

	assert( divvy_text != (const char ***)0 );
	assert( divvy_text_cnt != (unsigned short *)0 );
	assert( divvy_fname != (const char *)0 );

	DEBUG_FUNC_START;

	*divvy_text_cnt = (unsigned short)0;
	*divvy_text = (const char **)0;

	if ( isdigit( *divvy_fname ) ) {
		len = strlen( divvy_fname );
		divvy_int = strtoul( divvy_fname, &int_ptr, 10 );
		if ( int_ptr == &divvy_fname[ len ] && divvy_int > 0L && divvy_int < (unsigned long)USHRT_MAX ) {
			if ( construct_divvy( divvy_text, divvy_text_cnt, (unsigned short)divvy_int ) == -1 ) {
				RETURN_INT( -1 );
			}

			RETURN_INT( 0 );
		}
	}

	if ( load_file( divvy_text, divvy_text_cnt, divvy_fname ) == -1 ) {
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}

static int load_file( const char ***divvy_text, unsigned short *divvy_text_cnt, const char *divvy_fname )
{
	unsigned long	divvy_text_cnt_tmp;

	divvy_text_cnt_tmp = 0UL;

	if ( load_text_file( divvy_text, &divvy_text_cnt_tmp, divvy_fname ) == -1 ) {
		return -1;
	}

	*divvy_text_cnt = (unsigned short)divvy_text_cnt_tmp;

	if ( *divvy_text_cnt == (unsigned short)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Divvy file was empty.\n", stderr );
		return -1;
	}

	return 0;
}

static int construct_divvy( const char ***divvy_text, unsigned short *divvy_text_cnt, unsigned short divvy_cnt )
{
	int	width;
	size_t	alloc_size, str_size;
	char	**new, *str;
	unsigned short	ndx;
	char	int_str[24];

	if ( divvy_cnt > (unsigned short)10000 ) {
		width = 5;
	} else {
		if ( divvy_cnt > (unsigned short)1000 ) {
			width = 4;
		} else {
			if ( divvy_cnt > (unsigned short)100 ) {
				width = 3;
			} else {
				if ( divvy_cnt > (unsigned short)10 ) {
					width = 2;
				} else {
					width = 1;
				}
			}
		}
	}

	alloc_size = (size_t)divvy_cnt * sizeof( char * );

	new = (char **)malloc( alloc_size );

	if ( new == (char **)0 ) {
		UNIX_ERROR( "malloc() failed" );
		return -1;
	}

	str_size = (size_t)( width + 1 );
	alloc_size = (size_t)divvy_cnt * str_size;

	str = (char *)malloc( alloc_size );
	if ( str == (char *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		return -1;
	}

	for ( ndx = (unsigned short)0; ndx < divvy_cnt; ++ndx ) {
		new[ ndx ] = &str[ ndx * str_size ];
		if ( snprintf( int_str, sizeof( int_str ), "%0*d", width, ndx ) != width ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "snprintf() failed to format integer.\n", stderr );
			return -1;
		}

		if ( strlen( int_str ) != width ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Integer string [", stderr );
			(void) fputs( int_str, stderr );
			(void) fputs( "] is too long.\n", stderr );
			return -1;
		}

		(void) strcpy( new[ ndx ], int_str );
	}

	*divvy_text = (const char **)new;
	*divvy_text_cnt = divvy_cnt;

	return 0;
}
