/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "where.h"
#include "dfile_dynamic.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "dfile_join.h"

static const char       rcsid[] = "$Id: get_join_records.c,v 1.2 2009/10/16 20:07:58 keith Exp $";

/*
** $Log: get_join_records.c,v $
** Revision 1.2  2009/10/16 20:07:58  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/08 11:23:45  keith
** Initial revision
**
*/

static int read_join_file( dfile_t *, int *, void * );

/*
** This function retrieves a set of records from the join file that joins to
** the current input record.
*/
int get_join_records( field_t ***ret_key_record, size_t **ret_key_record_size, field_t ***ret_cp_record, size_t **ret_cp_record_size, unsigned long *ret_record_cnt, dfile_t *join_dfile, const dfile_bind_t **join_key_tbl, const dfile_bind_t **input_key_tbl, unsigned short key_tbl_cnt, void *join_filter, int *initial_read_flag, int *end_of_file_flag, const dfile_bind_t **join_field_tbl, unsigned short join_field_tbl_cnt, unsigned long alloc_record_cnt )
{
	static const char	func[] = "get_join_records";
	field_t	**key_record, **cp_record;
	unsigned long	record_cnt;
	int	compare_result;
	size_t	*key_record_size, *cp_record_size;

	assert( ret_key_record != (field_t ***)0 );
	assert( ret_cp_record != (field_t ***)0 );
	assert( ret_record_cnt != (unsigned long *)0 );
	assert( join_dfile != (dfile_t *)0 );
	assert( join_key_tbl != (const dfile_bind_t **)0 );
	assert( input_key_tbl != (const dfile_bind_t **)0 );
	assert( initial_read_flag != (int *)0 );
	assert( end_of_file_flag != (int *)0 );

	DEBUG_FUNC_START;

	if ( *initial_read_flag ) {
		/*
		** Set initial read flag to false.
		*/
		*initial_read_flag = 0;

		/*
		** Priming read.
		*/
		if ( read_join_file( join_dfile, end_of_file_flag, join_filter ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	record_cnt = 0UL;
	cp_record = *ret_cp_record;
	key_record = *ret_key_record;
	cp_record_size = *ret_cp_record_size;
	key_record_size = *ret_key_record_size;

	while ( !*end_of_file_flag ) {
		compare_result = compare_bind( join_key_tbl, input_key_tbl, key_tbl_cnt );

		if ( compare_result < 0 ) {
			/*
			** Discard record from join file.
			*/
			if ( read_join_file( join_dfile, end_of_file_flag, join_filter ) == -1 ) {
				RETURN_INT( -1 );
			}
			continue;
		}

		if ( compare_result > 0 ) {
			/*
			** Join file key is greater than input file key.
			*/
			break;
		}

		assert( record_cnt <= alloc_record_cnt );

		if ( record_cnt == alloc_record_cnt ) {
			if ( alloc_record( &cp_record, &cp_record_size, record_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}

			if ( alloc_record( &key_record, &key_record_size, record_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}
			++alloc_record_cnt;
		}

		if ( join_field_tbl_cnt > (unsigned short)0 ) {
			if ( copy_record( &cp_record[ record_cnt ], &cp_record_size[ record_cnt ], join_field_tbl, join_field_tbl_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}
		}

		if ( copy_record( &key_record[ record_cnt ], &key_record_size[ record_cnt ], join_key_tbl, key_tbl_cnt ) == -1 ) {
			RETURN_INT( -1 );
		}

		++record_cnt;

		if ( read_join_file( join_dfile, end_of_file_flag, join_filter ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	*ret_cp_record = cp_record;
	*ret_cp_record_size = cp_record_size;
	*ret_key_record = key_record;
	*ret_key_record_size = key_record_size;
	*ret_record_cnt = record_cnt;

	if ( Debug ) {
		(void) fprintf( stderr, "%lu records loaded to join\n", record_cnt );
	}

	RETURN_INT( 0 );
}

static int read_join_file( dfile_t *dfile, int *end_of_file_flag, void *join_filter )
{
	int	read_ret;
	where_result_t	where_result;
	char	err_msg[ 256 ];

	while ( ( read_ret = dfile_read( dfile ) ) == 0 ) {
		if ( join_filter == (void *)0 ) {
			break;
		} else {
			if ( where_condition( &where_result, err_msg, sizeof( err_msg ), join_filter ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "join where_condition() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );
				return -1;
			}

			if ( where_result == Where_result_true ) {
				break;
			}
		}
	}

	if ( read_ret == 0 ) {
		return 0;
	}

	if ( dfile->error == Dfile_all_data_processed ) {
		/*
		** Set end of file flag to true.
		*/
		*end_of_file_flag = 1;
		return 0;
	}

	FPUT_SRC_CODE( stderr );
	(void) fputs( "Failed to read all join file data.\n", stderr );

	return -1;
}
