/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "where.h"
#include "dfile_sort.h"

static const char       rcsid[] = "$Id: load_sort.c,v 1.2 2009/10/16 20:22:24 keith Exp $";

/*
** $Log: load_sort.c,v $
** Revision 1.2  2009/10/16 20:22:24  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/10 14:59:15  keith
** Initial revision
**
*/

/*
** This function loads data into memory for sorting.
*/
int load_sort( field_t ***ret_record, unsigned long *ret_record_cnt, const char *dfile_name, dfile_tag_t *tag_tbl, unsigned short tag_tbl_cnt, int thread_input_flag, sm_dfile_t *sort, unsigned short sort_cnt, dfile_bind_t *output_bind, unsigned short output_bind_cnt, unsigned short *key_ndx_tbl, unsigned short key_tbl_cnt )
{
	static const char	func[] = "load_sort";
	sm_dfile_t	cmd_line_sort;
	unsigned short	*io_field_map;
	field_t	**record;
	unsigned long	record_cnt;
	dfile_t	*input_dfile;
	void	*where;
	char	err_msg[ 256 ];
	where_result_t	where_result;

	assert( ret_record != (field_t ***)0 );
	assert( ret_record_cnt != (unsigned long *)0 );
	assert( output_bind != (dfile_bind_t *)0 );
	assert( key_ndx_tbl != (unsigned short *)0 );
	assert( key_tbl_cnt > (unsigned short)0 );

	DEBUG_FUNC_START;

	if ( dfile_name != (const char *)0 ) {
		/*
		** Did not specify control file.
		*/
		cmd_line_sort.dfile_name = dfile_name;
		cmd_line_sort.tag = tag_tbl;
		cmd_line_sort.tag_cnt = tag_tbl_cnt;
		cmd_line_sort.blocks_per_buffer_cnt = (unsigned short)4;
		cmd_line_sort.where = (sexpr_t *)0;
		if ( thread_input_flag ) {
			cmd_line_sort.buffer_cnt = (unsigned short)3;
		} else {
			cmd_line_sort.buffer_cnt = (unsigned short)1;
		}

		sort = &cmd_line_sort;
		sort_cnt = (unsigned short)1;
	}

	record_cnt = 0UL;
	record = (field_t **)0;

	for ( ; sort_cnt > (unsigned short)0; --sort_cnt ) {
		if ( dfile_name == (const char *)0 ) {
			/*
			** Specified control file.
			*/
			if ( append_tag( &sort->tag, &sort->tag_cnt, tag_tbl, tag_tbl_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}
		}

		if ( open_input( &input_dfile, sort->dfile_name, sort->tag, sort->tag_cnt, sort->blocks_per_buffer_cnt, sort->buffer_cnt ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( map_io_fields( &io_field_map, input_dfile->bind_hash_table, input_dfile->bind, output_bind, output_bind_cnt, key_ndx_tbl, key_tbl_cnt, dfile_name ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( sort->where == (sexpr_t *)0 ) {
			where = (void *)0;
		} else {
			if ( where_compile_sexpr( &where, err_msg, sizeof( err_msg ), (void *)sort->where, input_dfile ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "where_compile_sexpr() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );

				RETURN_INT( -1 );
			}

			assert( where != (void *)0 );
		}

		while ( dfile_read( input_dfile ) == 0 ) {
			if ( where != (void *)0 ) {
				 if ( where_condition( &where_result, err_msg, sizeof( err_msg ), where ) != WHERE_NOERR ) {
					FPUT_SRC_CODE( stderr );
					(void) fputs( "where_condition() failed [", stderr );
					(void) fputs( err_msg, stderr );
					(void) fputs( "].\n", stderr );

					RETURN_INT( -1 );
                                }

				if ( where_result == Where_result_false ) {
					/*
					** Skip record.
					*/
					continue;
				}
			}

			if ( alloc_record( &record, (size_t **)0, record_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}

			if ( copy_sort_record( &record[ record_cnt ], input_dfile->bind, input_dfile->bind_cnt, io_field_map, output_bind_cnt ) == -1 ) {
				RETURN_INT( -1 );
			}

			++record_cnt;
		}

		if ( input_dfile->error != Dfile_all_data_processed ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Failed to read all data.\n", stderr );
			RETURN_INT( -1 );
		}

		(void) dfile_read_close( input_dfile );
		++sort;
	}

	*ret_record = record;
	*ret_record_cnt = record_cnt;

	if ( Debug ) {
		(void) fprintf( stderr, "%lu records loaded to sort\n", record_cnt );
	}

	RETURN_INT( 0 );
}
