/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "dfile_dynamic.h"
#include "dfile_sort.h"

static const char       rcsid[] = "$Id: map_io_fields.c,v 1.2 2009/10/16 20:22:24 keith Exp $";

/*
** $Log: map_io_fields.c,v $
** Revision 1.2  2009/10/16 20:22:24  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/10 15:12:57  keith
** Initial revision
**
*/

static int verify_not_key_field( unsigned short *, unsigned short, unsigned short, const char *, const char * );

/*
** This function creates a map between input and output record layouts.
*/
int map_io_fields( unsigned short **io_field_map, void *input_bind_hash_table, dfile_bind_t *input_bind, dfile_bind_t *output_bind, unsigned short output_bind_cnt, unsigned short *key_ndx_tbl, unsigned short key_tbl_cnt, const char *input_dfile_name )
{
	static const char	func[] = "map_io_fields";
	size_t	alloc_size;
	unsigned short	ndx, mapped_cnt;
	unsigned short	*map;
	dfile_bind_t	**bind_entry, *out_bind;

	assert( input_bind_hash_table != (void *)0 );
	assert( io_field_map != (unsigned short **)0 );
	assert( input_bind != (dfile_bind_t *)0 );
	assert( output_bind != (dfile_bind_t *)0 );

	DEBUG_FUNC_START;
	if ( Debug ) {
		(void) fprintf( stderr, "%s() started at %s\n", func, get_ctime() );
	}

	*io_field_map = (unsigned short *)0;

	alloc_size = sizeof( unsigned short ) * output_bind_cnt;
	map = (unsigned short *)malloc( alloc_size );
	if ( map == (unsigned short *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	mapped_cnt = (unsigned short)0;

	for ( ndx = (unsigned short)0; ndx < output_bind_cnt; ++ndx ) {
		out_bind = &output_bind[ ndx ];
		bind_entry = (dfile_bind_t **)dhfind( (void *)out_bind, input_bind_hash_table, strhkey( out_bind->field_name ), dfile_bind_field_name_cmp );

		if ( bind_entry == (dfile_bind_t **)0 ) {
			if ( verify_not_key_field( key_ndx_tbl, key_tbl_cnt, ndx, input_dfile_name, output_bind[ ndx ].field_name ) == -1 ) {
				RETURN_INT( -1 );
			}

			/*
			** Missing field was not part of key.
			*/

			map[ ndx ] = USHRT_MAX;
		} else {
			assert( *bind_entry >= input_bind );
			map[ ndx ] = (unsigned short)( *bind_entry - input_bind );
			++mapped_cnt;
		}
	}

	if ( mapped_cnt == (unsigned short)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "No output fields match input fields.\n", stderr );
		RETURN_INT( -1 );
	}

	*io_field_map = map;

	if ( Debug ) {
		(void) fprintf( stderr, "Mapped %hu of %hu output fields.\n", mapped_cnt, output_bind_cnt );
	}

	RETURN_INT( 0 );
}

static int verify_not_key_field( unsigned short *key_ndx_tbl, unsigned short key_tbl_cnt, unsigned short missing_field_ndx, const char *input_dfile_name, const char *field_name )
{
	if ( key_ndx_tbl == (unsigned short *)0 ) {
		/*
		** This is a dfile to be merged. This verification is
		** performed later (somewhere else) for dfiles to be merged.
		*/
		return 0;
	}

	for ( ; key_tbl_cnt > (unsigned short)0; --key_tbl_cnt ) {
		if ( *key_ndx_tbl == missing_field_ndx ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Key field [", stderr );
			(void) fputs( field_name, stderr );
			(void) fputs( "] was not defined in input dfile [", stderr );
			(void) fputs( input_dfile_name, stderr );
			(void) fputs( "].\n", stderr );
			return -1;
		}

		++key_ndx_tbl;
	}

	return 0;
}
