/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "where.h"
#include "dfile_sort.h"

static const char       rcsid[] = "$Id: merge_write.c,v 1.2 2009/10/16 20:22:24 keith Exp $";

/*
** $Log: merge_write.c,v $
** Revision 1.2  2009/10/16 20:22:24  keith
** Added GPL to source code.
**
** Revision 1.1  2009/03/12 08:31:40  keith
** Initial revision
**
*/

/*
** This function writes a merge record from the top of the merge queue.
** Another record is then read from the source file that contained the written
** record. The new record is added to merge queue.
*/
int merge_write( dfile_t *dfile, merge_queue_t **merge_queue, const order_by_t *key_tbl, unsigned short key_tbl_cnt )
{
	static const char	func[] = "merge_write";
	dfile_t	*input_dfile;
	dfile_bind_t	*output_bind, *input_bind, *bind;
	unsigned short	output_bind_cnt, *io_field_map, offset;
	merge_queue_t	*link;
	int	read_ret;
	where_result_t	where_result;
	char	err_msg[ 256 ];

	assert( dfile != (dfile_t *)0 );
	assert( merge_queue != (merge_queue_t **)0 );
	assert( key_tbl != (const order_by_t *)0 );

	DEBUG_FUNC_START;

	output_bind = dfile->bind;
	output_bind_cnt = dfile->bind_cnt;
	link = *merge_queue;
	io_field_map = link->io_field_map;
	input_dfile = link->dfile;
	input_bind = input_dfile->bind;

	for ( ; output_bind_cnt > (unsigned short)0; --output_bind_cnt ) {
		offset = *io_field_map;

		if ( offset == USHRT_MAX ) {
			*output_bind->field_buffer = "";
			*output_bind->field_length = (size_t)0;
		} else {
			bind = &input_bind[ offset ];
			*output_bind->field_buffer = *bind->field_buffer;
			*output_bind->field_length = *bind->field_length;
		}

		++output_bind;
		++io_field_map;
	}

	if ( dfile_write( dfile ) == -1 ) {
		RETURN_INT( -1 );
	}

	*merge_queue = link->next;

	while ( ( read_ret = dfile_read( input_dfile ) ) == 0 ) {
		if ( link->where == (void *)0 ) {
			/*
			** No filter defined.
			*/
			break;
		}

		if ( where_condition( &where_result, err_msg, sizeof( err_msg ), link->where ) != WHERE_NOERR ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "where_condition() failed [", stderr );
			(void) fputs( err_msg, stderr );
			(void) fputs( "].\n", stderr );

			RETURN_INT( -1 );
		}

		if ( where_result == Where_result_true ) {
			/*
			** Unfiltered record found.
			*/
			break;
		}
	}

	if ( read_ret == -1 ) {
		if ( input_dfile->error == Dfile_all_data_processed ) {
			/*
			** End of file.
			*/
			RETURN_INT( 0 );
		} else {
			RETURN_INT( -1 );
		}
	}

	if ( add_to_merge_queue( merge_queue, link, key_tbl, key_tbl_cnt ) == -1 ) {
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
