/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_process_threaded_read.c,v 1.3 2009/10/16 07:35:55 keith Exp $";

/*
** $Log: _dfile_process_threaded_read.c,v $
** Revision 1.3  2009/10/16 07:35:55  keith
** Add GPL to source code.
**
** Revision 1.2  2009/05/21 20:55:21  keith
** Made changes to use DFILE_THREAD as compile time switch for threading.
**
** Revision 1.1  2009/02/27 04:57:25  keith
** Initial revision
**
*/

#ifdef DFILE_THREAD
/*
** This function calls parsing routines. If end of buffer is reached,
** buffers are rotated and remaining portion of record is parsed.
*/

int _dfile_process_threaded_read( dfile_t *dfile )
{
	static const char	func[] = "_dfile_process_threaded_read";
	dfile_parse_t	parse;
	dfile_buffer_t	*buffer;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	buffer = dfile->application_buffer;

	assert( buffer != (dfile_buffer_t *)0 );
	assert( buffer->state != (dfile_state_t *)0 );
	assert( dfile->buffer_cnt > (unsigned short)0 );

	for ( ;; ) {
		if ( _dfile_buffer_wait( buffer, Dfile_parse_state, dfile->buffer_cnt ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( buffer->processing_complete_flag == 'Y' ) {
			if ( dfile->error == Dfile_all_data_processed ) {
				_dfile_read_eof_processing( dfile, buffer );
			}
			RETURN_INT( -1 );
		}

		if ( ( buffer->buf_ptr >= buffer->buf_ptr_end ) && ( buffer->state->buffer_state == Dfile_parse_state ) ) {
			/*
			** Buffer is exhausted.
			** Advance to next buffer state.
			*/
			if ( _dfile_release_buffer( buffer ) == -1 ) {
				RETURN_INT( -1 );
			}
			buffer = buffer->next;
			dfile->application_buffer = buffer;
			continue;
		}

		if ( dfile->record_type == Dfile_delimited ) {
			parse = _dfile_get_record( dfile );
		} else {
			assert( dfile->record_type == Dfile_variable_length );

			parse = _dfile_get_var_record( dfile );
		}

		switch ( parse ) {
		case Dfile_parsed_record:
			RETURN_INT( 0 );

		case Dfile_parse_failed:
			break;

		case Dfile_fatal_parse_error:
			RETURN_INT( -1 );
		default:
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Programming error--did not expect return value ", stderr );
			(void) fput_int( (int)parse, stderr );
			(void) fputs( " from get_record().\n", stderr );

			RETURN_INT( -1 );
		}
	}

	/*
	** Should never reach this point.
	*/
	abort();
}
#endif
