/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include <zlib.h>
#include "tbox.h"
#include "dfile.h"
#include "_dfile.h"

static const char       rcsid[] = "$Id: _dfile_var_rec_refill_zipped_buffer.c,v 1.2 2009/10/16 07:35:57 keith Exp $";

/*
** $Log: _dfile_var_rec_refill_zipped_buffer.c,v $
** Revision 1.2  2009/10/16 07:35:57  keith
** Add GPL to source code.
**
** Revision 1.1  2009/02/28 05:17:41  keith
** Initial revision
**
*/

/*
** This function refills a buffer with ASCII data from a buffer of GZIP
** compressed data. When the GZIP buffer empties, it is refilled by doing
** another read().
*/

int _dfile_var_rec_refill_zipped_buffer( dfile_t *dfile )
{
	static const char	func[] = "_dfile_var_rec_refill_zipped_buffer";
	char	*buffer_area;
	dfile_buffer_t	*buffer;
	size_t	available_cnt, usable_cnt, inflated_cnt;
	z_streamp	zstream;
	int	zerr;
	Bytef	*inflated_data;

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	buffer = dfile->io_buffer;
	assert( buffer != (dfile_buffer_t *)0 );
	buffer_area = buffer->buffer_area;
	assert( buffer_area != (char *)0 );
	available_cnt = dfile->buffer_length;
	usable_cnt = (size_t)0;
	zstream = (z_streamp)dfile->zstream;
	assert( zstream != (z_stream *)0 );
	assert( dfile->zipped_buffer != (unsigned char *)0 );
	assert( dfile->buffer_length > (size_t)0 );

	buffer->buffer_rec_cnt = 0UL;

	while ( available_cnt > (size_t)0 && dfile->error == Dfile_ok ) {
		if ( zstream->avail_in == 0U ) {
			if ( _dfile_read_gz_file( dfile ) == -1 ) {
				RETURN_INT( -1 );
			}
		}

		inflated_data = (Bytef *)&buffer_area[ usable_cnt ];
		zstream->next_out = inflated_data;
		zstream->avail_out = available_cnt;

		zerr = inflate( zstream, Z_NO_FLUSH );

		inflated_cnt = available_cnt - zstream->avail_out;
		usable_cnt += inflated_cnt;
		available_cnt -= inflated_cnt;

		dfile->crc = crc32( dfile->crc, inflated_data, inflated_cnt );
		dfile->file_char_cnt += inflated_cnt;

		if ( zerr == Z_STREAM_END ) {
			if ( _dfile_check_zipped_crc( dfile ) == -1 ) {
				RETURN_INT( -1 );
			}
			if ( _dfile_read_gz_header( dfile ) == -1 ) {
				if ( dfile->error == Dfile_end_of_file ) {
					break;
				}
				RETURN_INT( -1 );
			}
		} else {
			if ( zerr != Z_OK ) {
				dfile->error = Dfile_invalid_gzip_format;
				(void) fput_src_code( __FILE__, __LINE__, stderr );
				(void) fputs( "inflate() failed (", stderr );
				(void) fput_int( zerr, stderr );
				(void) fputs( "): ", stderr );
				if ( zstream->msg == (char *)0 ) {
					(void) fputs( "no description available of error", stderr );
				} else {
					(void) fputs( zstream->msg, stderr );
				}
				(void) fputs( ".\n", stderr );
				RETURN_INT( -1 );
			}
		}
	}

	buffer->buf_ptr = buffer_area;

	if ( usable_cnt == (size_t)0 ) {
		assert( dfile->error == Dfile_end_of_file );
		buffer->buf_ptr_end = buffer_area;
		if ( Debug ) {
			(void) fputs( "Zero length data was compressed.\n", stderr );
		}
		RETURN_INT( -1 );
	}

	buffer->buf_ptr_end = &buffer_area[ usable_cnt ];

	/*
	** Null terminate buffer because parsing routine will sometimes
	** use value at end of buffer as field length value.
	*/
	*buffer->buf_ptr_end = (char)0;

	if ( Debug ) {
		(void) fprintf( stderr, "refilled buffer: %u usable\n", usable_cnt );
		(void) fprintf( stderr, "buf_ptr = %p, buf_ptr_end = %p\n", buffer->buf_ptr, buffer->buf_ptr_end );
	}
	RETURN_INT( 0 );
}
