/* Copyright (C) 2009 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <limits.h>
#include <assert.h>
#include "tbox.h"

static const char       rcsid[] = "$Id: cfg_open.c,v 1.2 2009/10/16 18:00:43 keith Exp $";

/*
** $Log: cfg_open.c,v $
** Revision 1.2  2009/10/16 18:00:43  keith
** Added GPL to source code.
**
** Revision 1.1  2009/02/14 09:08:31  keith
** Initial revision
**
*/

static int open_cfg_file( FILE **, const char *, size_t, const char *, size_t );

/*
** This function opens a configuration file for reading.
*/

FILE *cfg_open( const char *fname, const char *env_path_var )
{
	static const char	func[] = "cfg_open";
	FILE	*fptr;
	const char	*slash_ptr, *cfg_path, *path_delim, *orig_cfg_path;
	size_t	dir_name_len, fname_len;

	assert( fname != (const char *)0 );
	assert( env_path_var != (const char *)0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( fname = [%s], env_path_var = [%s] )\n", func, fname, env_path_var );
	}

	DEBUG_FUNC_START;

	slash_ptr = strchr( fname, '/' );

	if ( slash_ptr != (const char *)0 ) {
		/*
		** Do not use environment path variable.
		*/
		fptr = fopen( fname, "r" );
		if ( fptr == (FILE *)0 ) {
			UNIX_ERROR( "fopen() failed" );
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Could not open DFile field definitions [", stderr );
			(void) fputs( fname, stderr );
			(void) fputs( "].\n", stderr );
			RETURN_POINTER( (FILE *)0 );
		}

		RETURN_POINTER( fptr );
	}

	cfg_path = getenv( env_path_var );
	if ( cfg_path == (const char *)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Environment variable [", stderr );
		(void) fputs( env_path_var, stderr );
		(void) fputs( "] not defined.\n", stderr );

		RETURN_POINTER( (FILE *)0 );
	}

	orig_cfg_path = cfg_path;
	fptr = (FILE *)0;
	fname_len = strlen( fname );

	while ( ( path_delim = strchr( cfg_path, ':' ) ) != (const char *)0 ) {
		assert( path_delim >= cfg_path );
		dir_name_len = (size_t)( path_delim - cfg_path );
		if ( open_cfg_file( &fptr, cfg_path, dir_name_len, fname, fname_len ) == 0 ) {
			break;
		}

		cfg_path = &path_delim[ 1 ];
	}

	if ( fptr == (FILE *)0 ) {
		if ( open_cfg_file( &fptr, cfg_path, strlen( cfg_path ), fname, fname_len ) == -1 ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Could not find configuration file [", stderr );
			(void) fputs( fname, stderr );
			(void) fputs( "] in [", stderr );
			(void) fputs( env_path_var, stderr );
			(void) fputs( "] containing [", stderr );
			(void) fputs( orig_cfg_path, stderr );
			(void) fputs( "].\n", stderr );
		}
	}

	RETURN_POINTER( fptr );
}

static int open_cfg_file( FILE **fptr, const char *dir_name, size_t dir_name_len, const char *fname, size_t fname_len )
{
	char	file_path[PATH_MAX];
	size_t	path_len;

	/*
	** +1 is for '/' character between directory and file name.
	*/
	path_len = dir_name_len + fname_len + (size_t)1;

	if ( path_len >= sizeof( file_path ) ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "File path [", stderr );
		(void) fwrite( (void *)dir_name, sizeof( char ), dir_name_len, stderr );
		(void) fputs( "] / [", stderr );
		(void) fwrite( (void *)fname, sizeof( char ), fname_len, stderr );
		(void) fputs( "] exceeds maximum length ", stderr );
		(void) fput_uint( sizeof( file_path ), stderr );
		(void) fputs( ".\n", stderr );

		return -1;
	}

	(void) memcpy( (void *)file_path, (void *)dir_name, dir_name_len );
	file_path[ dir_name_len ] = '/';
	(void) memcpy( (void *)&file_path[ dir_name_len + 1 ], (void *)fname, fname_len );
	file_path[ path_len ] = (char)0;

	if ( access( file_path, F_OK ) == -1 ) {
		if ( Debug ) {
			(void) fprintf( stderr, "file [%s] does not exist\n", file_path );
		}
		return -1;
	}

	*fptr = fopen( file_path, "r" );
	if ( *fptr == (FILE *)0 ) {
		UNIX_ERROR( "fopen() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could not open configuration file [", stderr );
		(void) fputs( file_path, stderr );
		(void) fputs( "].\n", stderr );
		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "opened configuration file [%s]\n", file_path );
	}

	return 0;
}
