/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "where.h"
#include "dfile_agfunc.h"


/*
** This function processes all records from input file.
*/
int process_file( dfile_t *output_dfile, dfile_t *input_dfile, const dfile_bind_t **input_key_tbl, dfile_bind_t **output_key_tbl, unsigned short key_tbl_cnt, func_t *func_tbl, unsigned short func_tbl_cnt, void *input_filter, void *output_filter )
{
	static int	initial_control_break = 1;
	unsigned long	record_cnt;
	unsigned short	ndx;
	where_result_t	where_result;
	char	err_msg[ 256 ];
	size_t	alloc_size, *alloc_key_size;

	assert( output_dfile != (dfile_t *)0 );
	assert( input_dfile != (dfile_t *)0 );
	assert( func_tbl != (func_t *)0 );
	/*
	** When filters are not used, pointers will be null.
	*/

	DEBUG_FUNC_START;

	alloc_size = sizeof( size_t ) * (size_t)key_tbl_cnt;
	alloc_key_size = (size_t *)malloc( alloc_size );

	if ( alloc_key_size == (size_t *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	(void) memset( (void *)alloc_key_size, 0, alloc_size );

	record_cnt = 0UL;

	while ( dfile_read( input_dfile ) == 0 ) {
		if ( input_filter != (void *)0 ) {
			if ( where_condition( &where_result, err_msg, sizeof( err_msg ), input_filter ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "input where_condition() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );
				RETURN_INT( -1 );
			}
			if ( where_result == Where_result_false ) {
				/*
				** Skip record.
				*/
				continue;
			}
		}

		if ( key_tbl_cnt > (unsigned short)0 ) {
			if ( initial_control_break ) {
				if ( copy_bind( output_key_tbl, input_key_tbl, alloc_key_size, key_tbl_cnt ) == -1 ) {
					RETURN_INT( -1 );
				}
				/*
				** Set initial control break to false.
				*/
				initial_control_break = 0;
			} else {
				if ( compare_bind( (const dfile_bind_t **)output_key_tbl, input_key_tbl, key_tbl_cnt ) != 0 ) {
					if ( output_record( output_dfile, record_cnt, func_tbl, func_tbl_cnt, output_filter ) == -1 ) {
						RETURN_INT( -1 );
					}

					if ( copy_bind( output_key_tbl, input_key_tbl, alloc_key_size, key_tbl_cnt ) == -1 ) {
						RETURN_INT( -1 );
					}

					record_cnt = 0UL;
				}
			}
		}

		++record_cnt;

		for ( ndx = (unsigned short)0; ndx < func_tbl_cnt; ++ndx ) {
			if ( process_agfunc( &func_tbl[ ndx ] ) == -1 ) {
				RETURN_INT( -1 );
			}
		}
	}

	if ( record_cnt > 0UL ) {
		if ( output_record( output_dfile, record_cnt, func_tbl, func_tbl_cnt, output_filter ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	if ( input_dfile->error != Dfile_all_data_processed ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to read all data.\n", stderr );
		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
