/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <unistd.h>
#include <sys/shm.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "where.h"
#include "dfile_cache_create.h"


/*
** This function reads records from file and stores them in shared memory.
*/
int fill_shared_memory( int shmid, size_t record_size, size_t record_cnt, size_t *field_size, size_t field_cnt, dfile_t *dfile, void *input_filter )
{
	int	shmflg;
	size_t	*header, len, rec_cnt, offset;
	void	*shmptr;
	char	*data;
	unsigned short	ndx, bind_cnt;
	dfile_bind_t	*bind;
	where_result_t	where_result;
	char	err_msg[ 256 ];

	assert( dfile != (dfile_t *)0 );

	DEBUG_FUNC_START;

	shmflg = SHM_R | SHM_W | SHM_RND;

	shmptr = shmat( shmid, (const void *)0, shmflg );

	if ( shmptr == (void *)-1 ) {
		UNIX_ERROR( "shmat() failed" );
		RETURN_INT( -1 );
	}

	/*
	** Populate header.
	*/
	header = (size_t *)shmptr;
	*header = record_cnt;
	++header;
	*header = record_size;
	++header;
	*header = field_cnt;
	++header;

	offset = (size_t)0;
	for ( ndx = (unsigned short)0; (size_t)ndx < field_cnt; ++ndx ) {
		*header = offset;
		++header;
		offset += field_size[ ndx ];
	}

	assert( offset == record_size );

	data = (char *)header;

	bind_cnt = dfile->bind_cnt;
	rec_cnt = (size_t)0;

	while ( dfile_read( dfile ) == 0 ) {
		if ( input_filter != (void *)0 ) {
			if ( where_condition( &where_result, err_msg, sizeof( err_msg ), input_filter ) != WHERE_NOERR ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "input where_condition() failed [", stderr );
				(void) fputs( err_msg, stderr );
				(void) fputs( "].\n", stderr );
				RETURN_INT( -1 );
			}
			if ( where_result == Where_result_false ) {
				/*
				** Skip record.
				*/
				continue;
			}
		}

		++rec_cnt;
		if ( rec_cnt > record_cnt ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Number of records in file changed during execution.\n", stderr );
			RETURN_INT( -1 );
		}
		bind = dfile->bind;
		for ( ndx = (unsigned short)0; ndx < bind_cnt; ++ndx ) {
			/*
			** Copy null byte.
			*/
			len = *bind->field_length + (size_t)1;

			(void) memcpy( (void *)data, *bind->field_buffer, len );

			data += field_size[ ndx ];

			++bind;
		}
	}

	if ( dfile->error != Dfile_all_data_processed ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to read all data.\n", stderr );
		RETURN_INT( -1 );
	}

	if ( rec_cnt != record_cnt ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Number of records in file (", stderr );
		(void) fput_uint( record_cnt, stderr );
		(void) fputs( " vs. ", stderr );
		(void) fput_uint( rec_cnt, stderr );
		(void) fputs( ") changed during execution.\n", stderr );

		RETURN_INT( -1 );
	}

	RETURN_INT( 0 );
}
