/* Copyright (C) 2009, 2010, 2011 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "dfile_dynamic.h"
#include "sexpr.h"
#include "dfile_sort.h"


#define NULL_CHECK( x ) ( ( *x == (const char *)0 ) ? "null" : *x )

static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], const char **input_dfile_name, const char **output_dfile_name, int *thread_input_flag, int *thread_output_flag, dfile_tag_t **dfile_tag_tbl, unsigned short *dfile_tag_tbl_cnt, const char **sort_key, char *sort_algorithm, const char **merge_dfile_name, const char **control_file )
{
	int	ch, ndx;
	extern char	*optarg;

	assert( argv != (char * const *) 0 );
	assert( input_dfile_name != (const char **) 0 );
	assert( output_dfile_name != (const char **) 0 );
	assert( thread_input_flag != (int *) 0 );
	assert( thread_output_flag != (int *) 0 );
	assert( dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( sort_key != (const char **)0 );
	assert( sort_algorithm != (char *)0 );
	assert( merge_dfile_name != (const char **)0 );
	assert( control_file != (const char **)0 );

	/*
	** Initialize debug to off.
	*/
#ifndef NDEBUG
	Debug = 0;
#endif

	*input_dfile_name = (const char *)0;
	*output_dfile_name = (const char *)0;
	*dfile_tag_tbl = (dfile_tag_t *)0;
	*dfile_tag_tbl_cnt = (unsigned short)0;
	*sort_key = (const char *)0;
	*sort_algorithm = 'Q';	/* default quicksort */
	*merge_dfile_name = (const char *)0;
	*control_file = (const char *)0;

	/*
	** Initialize I/O threading flags to false.
	*/
	*thread_input_flag = 0;
	*thread_output_flag = 0;

	while ( ( ch = getopt( argc, argv, "a:c:di:k:m:o:t:rw" ) ) != EOF ) {
		switch ( ch ) {
		case 'a':
			*sort_algorithm = *optarg;
			break;
		case 'c':
			*control_file = optarg;
			break;
		case 'd':
#ifdef NDEBUG
			(void) fputs( "Not compiled for debug.\n", stderr );
#else
			Debug = 1;
#endif
			break;
		case 'i':
			*input_dfile_name = optarg;
			break;
		case 'k':
			*sort_key = optarg;
			break;
		case 'm':
			*merge_dfile_name = optarg;
			break;
		case 'o':
			*output_dfile_name = optarg;
			break;
		case 'r':
			/*
			** Set threaded input to true.
			*/
			*thread_input_flag = 1;
			break;
		case 't':
			if ( cmd_line_parse_tag( dfile_tag_tbl, dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'w':
			/*
			** Set threaded output to true.
			*/
			*thread_output_flag = 1;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *control_file == (const char *)0 ) {
		if ( *input_dfile_name == (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Must specify input dfile name.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}

		if ( *output_dfile_name == (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Must specify output dfile name.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}

		if ( *sort_key == (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Must specify sort key.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}
	} else {
		if ( *input_dfile_name != (char *) 0 || *output_dfile_name != (char *) 0 || *sort_key != (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Cannot specify control file with other arguments.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Sort Algorithm: [%c]\n", *sort_algorithm );
		(void) fprintf( stderr, "Input DFile Name: [%s]\n", NULL_CHECK( input_dfile_name ) );
		(void) fprintf( stderr, "Output DFile Name: [%s]\n", NULL_CHECK( input_dfile_name ) );
		(void) fprintf( stderr, "Thread Input Flag: [%s]\n", ( *thread_input_flag ) ? "true" : "false" );
		(void) fprintf( stderr, "Thread Output Flag: [%s]\n", ( *thread_output_flag ) ? "true" : "false" );
		(void) fputs( "Input files:\n", stderr );
		for ( ndx = (unsigned short)0; ndx < *dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "DFile Tag [%s] [%s]\n", ( *dfile_tag_tbl )[ ndx ].tag, ( *dfile_tag_tbl )[ ndx ].tag_value );
		}
		(void) fprintf( stderr, "Merge DFile Name: [%s]\n", NULL_CHECK( merge_dfile_name ) );
		(void) fprintf( stderr, "Sort Key: [%s]\n", NULL_CHECK( sort_key ) );
		(void) fprintf( stderr, "Control File: [%s]\n", NULL_CHECK( control_file ) );
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-a c]", stderr );
	(void) fputs( " [-c file]", stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " -k field.[AD].[ANH],field...", stderr );
	(void) fputs( " -i dfile", stderr );
	(void) fputs( " [-m dfile]", stderr );
	(void) fputs( " -o dfile", stderr );
	(void) fputs( " [-t %x=abc]", stderr );
	(void) fputs( " [-r]", stderr );
	(void) fputs( " [-w]", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-a -> sort algorithm (Q)uick(default), (M)erge, (H)eap, (S)hell, (I)nsertion\n", stderr );
	(void) fputs( "\t-c -> control file name\n", stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-k -> key fields with Asending/Decending and ASCII/Numeric/High value nulls\n", stderr );
	(void) fputs( "\t-i -> input DFile\n", stderr );
	(void) fputs( "\t-m -> sorted DFile to merge\n", stderr );
	(void) fputs( "\t-o -> output DFile\n", stderr );
	(void) fputs( "\t-t -> DFile path tags\n", stderr );
	(void) fputs( "\t-r -> thread input (default no threading)\n", stderr );
	(void) fputs( "\t-w -> thread output (default no threading)\n", stderr );
}
