/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dcat.h"


static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], const char **dfile_name, dfile_tag_t **tag_tbl, unsigned short *tag_tbl_cnt, char *heading_flag, char *field_separator, const char **filter_file )
{
	int	ch, ndx;
	extern char	*optarg;

	assert( argv != (char * const *) 0 );
	assert( dfile_name != (const char **) 0 );
	assert( tag_tbl != (dfile_tag_t **) 0 );
	assert( tag_tbl_cnt != (unsigned short *) 0 );
	assert( heading_flag != (char *) 0 );
	assert( field_separator != (char *) 0 );
	assert( filter_file != (const char **) 0 );

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	*dfile_name = (const char *)0;
	*tag_tbl = (dfile_tag_t *)0;
	*tag_tbl_cnt = (unsigned short)0;
	*heading_flag = 'N';
	*field_separator = '|';
	*filter_file = (const char *)0;

	while ( ( ch = getopt( argc, argv, "dF:ht:x:" ) ) != EOF ) {
		switch ( ch ) {
		case 'd':
#ifdef NDEBUG
			(void) fputs( argv[ 0 ], stderr );
			(void) fputs( " not compiled for debug.\n", stderr );
			return -1;
#else
			Debug = 1;
#endif
			break;
		case 'h':
			*heading_flag = 'Y';
			break;
		case 'F':
			*field_separator = *optarg;
			break;
		case 't':
			if ( cmd_line_parse_tag( tag_tbl, tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'x':
			*filter_file = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( optind >= argc ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify input dfile name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	*dfile_name = argv[ optind ];

	if ( Debug ) {
		(void) fprintf( stderr, "DFile Name: [%s]\n", *dfile_name );
		(void) fprintf( stderr, "Field Separator: [%c]\n", *field_separator );
		for ( ndx = (unsigned short)0; ndx < *tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "DFile Tag [%s] [%s]\n", ( *tag_tbl )[ ndx ].tag, ( *tag_tbl )[ ndx ].tag_value );
		}
		(void) fprintf( stderr, "filter file: [%s]\n", ( *filter_file ) ? *filter_file : "null" );
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " [-F]", stderr );
	(void) fputs( " [-h]", stderr );
	(void) fputs( " [-t %x=abc]", stderr );
	(void) fputs( " [-x file]", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-F -> field separator (default |)\n", stderr );
	(void) fputs( "\t-h -> field heading\n", stderr );
	(void) fputs( "\t-t -> DFile path tags\n", stderr );
	(void) fputs( "\t-x -> record filter file\n", stderr );
}
