/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile_exec.h"

#define NULL_CHECK( x )	( ( x == (const char *)0 ) ? "NULL" : x )

static void redirect_output( int, int, const char * );
static void redirect_input( const char * );
static void set_env_var( char ** );
static const size_t	max_fname_size = 100;

/*
** This function opens stdout and stderr output files and starts a new
** instance of application. Application is expected to process a single slice.
*/
pid_t exec_proc( int *stdout_fd, int *stderr_fd, char * const *args, char **env_var, const char *stdin_file, const char *stdout_file, const char *stderr_file )

{
	mode_t	perm;
	pid_t	pid;
	char	msg[300], int_str[40];
	size_t	len;

	assert( args != (char **) 0 );
	assert( stdout_fd != (int *) 0 );
	assert( stderr_fd != (int *) 0 );
	/*
	** stdout_file and stderr_file may be null pointers when inheriting
	** parent's stdout and stderr.
	*/

	DEBUG_FUNC_START;

	if ( Debug ) {
#if 0
		char	* const *debug_arg = args;
		while ( *debug_arg ) {
			(void) fprintf( stderr, "[%s] ", *debug_arg );
			++debug_arg;
		}
#endif
		(void) fprintf( stderr, "\nstdin [%s], stdout [%s], stderr [%s]\n", NULL_CHECK( stdin_file ), NULL_CHECK( stdout_file ), NULL_CHECK( stderr_file ) );
	}

	perm = S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH | S_IWOTH;

	if ( stdout_file == (const char *) 0 ) {
		/*
		** Use parent stdout.
		*/
		*stdout_fd = 1;
	} else {
		/*
		** Open stdout output file.
		*/
		if ( ( *stdout_fd = open( stdout_file, O_CREAT|O_WRONLY|O_TRUNC, perm ) ) < 0 ) {
			(void) strcpy( msg, "Cannot open stdout file [" );
			(void) strncat( msg, stdout_file, max_fname_size );
			(void) strcat( msg, "]" );
			UNIX_ERROR( msg );
			return (pid_t) -1;
		}
	}

	if ( stderr_file == (const char *) 0 ) {
		/*
		** Use parent stderr.
		*/
		*stderr_fd = 2;
	} else {
		/*
		** Open stderr output file.
		*/
		if ( ( *stderr_fd = open( stderr_file, O_CREAT|O_WRONLY|O_APPEND, (mode_t) perm ) ) < 0 ) {
			(void) strcpy( msg, "Cannot open stderr file [" );
			(void) strncat( msg, stderr_file, max_fname_size );
			(void) strcat( msg, "]" );
			UNIX_ERROR( msg );
			return (pid_t) -1;
		}
	}

	/*
	** Create child process.
	*/
	if ( ( pid = fork() ) == (pid_t) -1 ) {
		UNIX_ERROR( "Cannot fork() child process." );
		return (pid_t) -1;
	}

	if ( pid == 0 ) {
		/*
		** Child process.
		*/
		if ( stdout_file != (const char *) 0 ) {
			/*
			** Redirect child's stdout output.
			*/
			redirect_output( 1, *stdout_fd, stdout_file );
		}
		if ( stderr_file != (const char *) 0 ) {
			/*
			** Redirect child's stderr output.
			*/
			redirect_output( 2, *stderr_fd, stderr_file );

			/*
			** Log a process started message to child's stderr.
			*/
			(void) strcpy( msg, get_ctime() );
			(void) strcat( msg, " PROCESS " );
			(void) snprintf( int_str, sizeof( int_str ), "%d", getpid() );
			(void) strcat( msg, int_str );
			(void) strcat( msg, " STARTED (stdin " );
			if ( stdin_file == (const char *)0 ) {
				(void) strcat( msg, "closed" );
			} else {
				(void) strncat( msg, stdin_file, max_fname_size );
			}
			(void) strcat( msg, ")\n" );
			len = strlen( msg );

			(void) write( 2, (void *) msg, len );
		}

		/*
		** Open input file.
		*/
		redirect_input( stdin_file );

		set_env_var( env_var );

		/*
		** Child becomes application to process slice.
		*/
		if ( execvp( *args, args ) == -1 ) {
			(void) strcpy( msg, "Cannot execvp() process [" );
			(void) strcat( msg, *args );
			(void) strcat( msg, "]" );
			UNIX_ERROR( msg );
			exit( 1 );
		}
		/*
		** Child should never reach this point.
		*/
		exit( 1 );
	}

	/*
	** Parent process.
	*/

	return pid;
}

/*
** This function redirects child's stdout or stderr output to open file.
*/
static void redirect_output( int stream_fd, int file_fd, const char *fname )
{
	char	msg[300], int_str[40];

#if 0
	/*
	** Seek to end of file.
	*/
	if ( lseek( file_fd, (off_t) 0, SEEK_END ) < (off_t) 0 ) {
		(void) strcpy( msg, "Cannot lseek() to end of file [" );
		(void) strncat( msg, fname, max_fname_size );
		(void) strcat( msg, "]" );
		UNIX_ERROR( msg );
		exit( 1 );
	}
#endif

	/*
	** Close existing stdout or stderr.
	*/
	if ( close( stream_fd ) < 0 ) {
		(void) strcpy( msg, "Cannot close() stream " );
		(void) snprintf( int_str, sizeof( int_str ), "%d", stream_fd );
		(void) strcat( msg, int_str );
		(void) strcat( msg, " for file [" );
		(void) strncat( msg, fname, max_fname_size );
		(void) strcat( msg, "]" );
		UNIX_ERROR( msg );
		exit( 1 );
	}

	/*
	** Re-assign stdout or stderr file descriptor to the open
	** output file.
	*/
	if ( dup( file_fd ) < 0 ) {
		(void) strcpy( msg, "Cannot dup() file descriptor for file [" );
		(void) strncat( msg, fname, max_fname_size );
		(void) strcat( msg, "]" );
		UNIX_ERROR( msg );
		exit( 1 );
	}

	/*
	** No longer need this file descriptor.
	*/
	if ( close( file_fd ) < 0 ) {
		(void) strcpy( msg, "Cannot close() file descriptor for file [" );
		(void) strncat( msg, fname, max_fname_size );
		(void) strcat( msg, "]" );
		UNIX_ERROR( msg );
		exit( 1 );
	}
}

static void redirect_input( const char *fname )
{
	int	fd;
	char	msg[300];

	if ( close( 0 ) < 0 ) {
		UNIX_ERROR( "close() failed on stdin" );
		exit( 1 );
	}

	if ( fname == (const char *)0 ) {
		return;
	}

	fd = open( fname, O_RDONLY );
	if ( fd != 0 ) {
		(void) strcpy( msg, "Cannot open stdin file [" );
		(void) strncat( msg, fname, max_fname_size );
		(void) strcat( msg, "]" );
		UNIX_ERROR( msg );
		exit( 1 );
	}
}

static void set_env_var( char **env_var )
{
	if ( env_var == (char **)0 ) {
		return;
	}

	while ( *env_var != (char *)0 ) {
		if ( putenv( *env_var ) != 0 ) {
			UNIX_ERROR( "Failed to create environment variable." );
			exit( 1 );
		}
		++env_var;
	}
}
