/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "dfile_join.h"


static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], char **input_dfile_name, char **output_dfile_name, int *thread_input_flag, int *thread_output_flag, dfile_tag_t **dfile_tag_tbl, unsigned short *dfile_tag_tbl_cnt, char **sort_key, char **join_dfile_name, char **data_fields, char *inner_outer_join_flag, char **outer_join_status_field, char *unique_join, char **input_filter_file_name, char **join_filter_file_name, char **output_filter_file_name, char **control_file, char **ipc_key )
{
	int	ch, ndx;
	extern char	*optarg;

	assert( argv != (char * const *) 0 );
	assert( input_dfile_name != (char **) 0 );
	assert( output_dfile_name != (char **) 0 );
	assert( thread_input_flag != (int *) 0 );
	assert( thread_output_flag != (int *) 0 );
	assert( dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( sort_key != (char **)0 );
	assert( join_dfile_name != (char **)0 );
	assert( data_fields != (char **)0 );
	assert( inner_outer_join_flag != (char *)0 );
	assert( outer_join_status_field != (char **)0 );
	assert( unique_join != (char *)0 );
	assert( input_filter_file_name != (char **)0 );
	assert( join_filter_file_name != (char **)0 );
	assert( output_filter_file_name != (char **)0 );
	assert( control_file != (char **)0 );
	assert( ipc_key != (char **)0 );

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	*input_dfile_name = (char *)0;
	*output_dfile_name = (char *)0;
	*dfile_tag_tbl = (dfile_tag_t *)0;
	*dfile_tag_tbl_cnt = (unsigned short)0;
	*sort_key = (char *)0;
	*data_fields = (char *)0;
	*join_dfile_name = (char *)0;
	*inner_outer_join_flag = 'I';
	*outer_join_status_field = (char *)0;
	*unique_join = (char)0;
	*input_filter_file_name = (char *)0;
	*join_filter_file_name = (char *)0;
	*output_filter_file_name = (char *)0;
	*ipc_key = (char *)0;
	*control_file = (char *)0;

	/*
	** Initialize I/O threading flags to false.
	*/
	*thread_input_flag = 0;
	*thread_output_flag = 0;

	while ( ( ch = getopt( argc, argv, "a:c:df:i:j:k:m:o:s:t:rwu:x:y:z:" ) ) != EOF ) {
		switch ( ch ) {
		case 'a':
			*ipc_key = optarg;
			break;
		case 'c':
			*control_file = optarg;
			break;
		case 'd':
#ifdef NDEBUG
			(void) fputs( "Not compiled for debug.\n", stderr );
#else
			Debug = 1;
#endif
			break;
		case 'i':
			*input_dfile_name = optarg;
			break;
		case 'j':
			*join_dfile_name = optarg;
			break;
		case 'k':
			*sort_key = optarg;
			break;
		case 'f':
			*data_fields = optarg;
			break;
		case 'm':
			*inner_outer_join_flag = toupper( *optarg );
			break;
		case 'o':
			*output_dfile_name = optarg;
			break;
		case 'r':
			/*
			** Set threaded input to true.
			*/
			*thread_input_flag = 1;
			break;
		case 's':
			*outer_join_status_field = optarg;
			break;
		case 't':
			if ( cmd_line_parse_tag( dfile_tag_tbl, dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'u':
			*unique_join = toupper( *optarg );
			break;
		case 'w':
			/*
			** Set threaded output to true.
			*/
			*thread_output_flag = 1;
			break;
		case 'x':
			*input_filter_file_name = optarg;
			break;
		case 'y':
			*join_filter_file_name = optarg;
			break;
		case 'z':
			*output_filter_file_name = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *control_file == (char *)0 ) {
		if ( *inner_outer_join_flag != 'I' && *inner_outer_join_flag != 'O' ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Join flag must be (I)nner or (O)uter.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}

		if ( *input_dfile_name == (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Must specify input dfile name.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}

		if ( *join_dfile_name == (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Must specify join dfile name.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}

		if ( *output_dfile_name == (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Must specify output dfile name.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}

		if ( *sort_key == (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Must specify sort key.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}

		if ( *unique_join != (char)0 && *unique_join != 'F' && *unique_join != 'L' ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Unique join must be (F)irst or (L)ast.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;
		}
	} else {
		if ( *input_dfile_name != (char *) 0 || *join_dfile_name != (char *) 0 || *output_dfile_name != (char *) 0 || *output_dfile_name != (char *) 0 || *sort_key != (char *) 0 || *data_fields != (char *) 0 ) {
			fput_src_code( __FILE__, __LINE__, stderr );
			(void) fputs( "Cannot specify control file with other arguments.\n", stderr );
			print_usage( argv[ 0 ] );
			return -1;

		}
	}

	if ( Debug ) {
		(void) fprintf( stderr, "IPC Key: [%s]\n", NULL_CHECK( *ipc_key ) );
		(void) fprintf( stderr, "Control File: [%s]\n", NULL_CHECK( *control_file ) );
		(void) fprintf( stderr, "Input DFile Name: [%s]\n", NULL_CHECK( *input_dfile_name ) );
		(void) fprintf( stderr, "Join DFile Name: [%s]\n", NULL_CHECK( *join_dfile_name ) );
		(void) fprintf( stderr, "Output DFile Name: [%s]\n", NULL_CHECK( *output_dfile_name ) );
		(void) fprintf( stderr, "Input Filter File Name: [%s]\n", NULL_CHECK( *input_filter_file_name ) );
		(void) fprintf( stderr, "Join Filter File Name: [%s]\n", NULL_CHECK( *join_filter_file_name ) );
		(void) fprintf( stderr, "Output Filter File Name: [%s]\n", NULL_CHECK( *output_filter_file_name ) );
		(void) fprintf( stderr, "Thread Input Flag: [%s]\n", ( *thread_input_flag ) ? "true" : "false" );
		(void) fprintf( stderr, "Thread Output Flag: [%s]\n", ( *thread_output_flag ) ? "true" : "false" );
		(void) fputs( "Input files:\n", stderr );
		for ( ndx = (unsigned short)0; ndx < *dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "DFile Tag [%s] [%s]\n", ( *dfile_tag_tbl )[ ndx ].tag, ( *dfile_tag_tbl )[ ndx ].tag_value );
		}
		(void) fprintf( stderr, "Sort Key: [%s]\n", NULL_CHECK( *sort_key ) );
		(void) fprintf( stderr, "Join Method: [%c]\n", *inner_outer_join_flag );
		(void) fprintf( stderr, "Outer Join Status Field: [%s]\n", NULL_CHECK( *outer_join_status_field ) );
		(void) fprintf( stderr, "Unique Join: [%c]\n", ( *unique_join ) ? *unique_join : '-' );
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-a 0xN]", stderr );
	(void) fputs( " [-c file]", stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " -k field,field...", stderr );
	(void) fputs( " -i dfile", stderr );
	(void) fputs( " -j dfile", stderr );
	(void) fputs( " -f field,field...", stderr );
	(void) fputs( " -o dfile", stderr );
	(void) fputs( " [-m c]", stderr );
	(void) fputs( " [-t %x=abc]", stderr );
	(void) fputs( " [-r]", stderr );
	(void) fputs( " [-s field]", stderr );
	(void) fputs( " [-u c]", stderr );
	(void) fputs( " [-w]", stderr );
	(void) fputs( " [-x file]", stderr );
	(void) fputs( " [-y file]", stderr );
	(void) fputs( " [-z file]", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-a -> shared memory IPC key\n", stderr );
	(void) fputs( "\t-c -> control file name\n", stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-k -> key fields\n", stderr );
	(void) fputs( "\t-i -> input DFile\n", stderr );
	(void) fputs( "\t-j -> join DFile\n", stderr );
	(void) fputs( "\t-f -> fields to use from joined dfile\n", stderr );
	(void) fputs( "\t-o -> output DFile\n", stderr );
	(void) fputs( "\t-m -> join method: (I)nner(default) or (O)uter\n", stderr );
	(void) fputs( "\t-t -> DFile path tags\n", stderr );
	(void) fputs( "\t-r -> thread input (default no threading)\n", stderr );
	(void) fputs( "\t-s -> outer join status field\n", stderr );
	(void) fputs( "\t-u -> unique join w/ (f)irst or (l)ast\n", stderr );
	(void) fputs( "\t-w -> thread output (default no threading)\n", stderr );
	(void) fputs( "\t-x -> input filter file\n", stderr );
	(void) fputs( "\t-y -> join filter file\n", stderr );
	(void) fputs( "\t-z -> output filter file\n", stderr );
}
