/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <regex.h>
#include <ctype.h>
#include <limits.h>
#include <string.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_transform.h"

#define NUM_OF_FIELDS	5

/*
** This function loads the control file.
*/
int load_control_file( trnsfrm_t **ret_trnsfrm, unsigned short *ret_trnsfrm_cnt, const char *filename )
{
	size_t	alloc_size;
	trnsfrm_t	*new, *trnsfrm;
	unsigned short	trnsfrm_cnt;
	FILE	*fptr;
	readstat_t	readstat;
	const char	*field[ NUM_OF_FIELDS ];
	const unsigned short	num_of_fields = NUM_OF_FIELDS;
	unsigned long	line_cnt = 0;
	const char	*trnsfrm_flag, *replacement, field_delim = ':';
	char	err_msg[128];
	int	ret, regcomp_flags;
	unsigned short	ndx;

	assert( ret_trnsfrm != (trnsfrm_t **)0 );
	assert( ret_trnsfrm_cnt != (unsigned short *)0 );
	assert( filename != (const char *)0 );

	DEBUG_FUNC_START;

	*ret_trnsfrm = (trnsfrm_t *)0;
	*ret_trnsfrm_cnt = (unsigned short)0;

	fptr = fopen( filename, "r" );

	if ( fptr == (FILE *)0 ) {
		UNIX_ERROR( "fopen() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could not open control file [", stderr );
		(void) fputs( filename, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	trnsfrm = (trnsfrm_t *)0;
	trnsfrm_cnt = (unsigned short)0;

	for ( ;; ) {
		readstat = cfg_read( field, num_of_fields, &line_cnt, field_delim, fptr, filename );

		if ( readstat == Read_fatal ) {
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		if ( readstat == Read_eof ) {
			break;
		}

		if ( readstat != Read_ok ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Unexpected return code (", stderr );
			(void) fput_int( readstat, stderr );
			(void) fputs( ") received from read_cfg().\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		for ( ndx = 0; ndx < num_of_fields; ++ndx ) {
			if ( field[ ndx ] == (const char *)0 ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Unexpected null value returned from read_cfg().\n", stderr );
				(void) fclose( fptr );
				RETURN_INT( -1 );
			}
		}

		if ( trnsfrm_cnt == USHRT_MAX ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Number of entries in control file exceeded maximum ", stderr );
			(void) fput_uint( USHRT_MAX, stderr );
			(void) fputs( ".\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		alloc_size = sizeof( trnsfrm_t ) * ( trnsfrm_cnt + (unsigned short)1 );

		new = (trnsfrm_t *)realloc( (void *)trnsfrm, alloc_size );
		if ( new == (trnsfrm_t *)0 ) {
			UNIX_ERROR( "realloc() failed" );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		trnsfrm = new;
		new = &new[ trnsfrm_cnt ];

		(void) memset( (void *)new, 0, sizeof( trnsfrm_t ) );

		new->input_field_name = strdup( field[ 0 ] );
		if ( new->input_field_name == (char *)0 ) {
			UNIX_ERROR( "strdup() failed" );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		trnsfrm_flag = field[ 1 ];
		if ( strlen( trnsfrm_flag ) != (size_t)1 ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Search flag must be one character (B,E,S).\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		new->trnsfrm_flag = toupper( *trnsfrm_flag );

		new->search_str = strdup( field[ 2 ] );
		if ( new->search_str == (char *)0 ) {
			UNIX_ERROR( "strdup() failed" );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		replacement = field[ 3 ];
		new->replacement_len = strlen( replacement );
		alloc_size = new->replacement_len + (size_t)1;

		new->replacement = (char *)malloc( alloc_size );
		if ( new->replacement == (char *)0 ) {
			UNIX_ERROR( "malloc() failed" );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		(void) memcpy( (void *)new->replacement, (void *)replacement, alloc_size );

		if ( *field[ 4 ] == (char)0 ) {
			/*
			** Default output field name to input field name.
			*/
			new->output_field_name = new->input_field_name;
		} else {
			new->output_field_name = strdup( field[ 4 ] );
			if ( new->output_field_name == (char *)0 ) {
				UNIX_ERROR( "strdup() failed" );
				(void) fclose( fptr );
				RETURN_INT( -1 );
			}
		}

		switch ( new->trnsfrm_flag ) {
		case 'R':
		case 'S':
			/*
			** replace or substitute
			*/
			break;

		case 'B':
			/*
			** basic regular expression.
			*/
			regcomp_flags = 0;
			break;

		case 'E':
			/*
			** extended regular expression.
			*/
			regcomp_flags = REG_EXTENDED;
			break;

		default:
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Search flag must be (S)tring compare, (B)asic regular expression, or (E)xtended regular expression.\n", stderr );
			(void) fclose( fptr );
			RETURN_INT( -1 );
		}

		switch ( new->trnsfrm_flag ) {
		case 'B':
		case 'E':
			ret = regcomp( &new->reg_exp, new->search_str, regcomp_flags );
			if ( ret != 0 ) {
				(void) regerror( ret, &new->reg_exp, err_msg, sizeof( err_msg ) );
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Failed to compile regular expression [", stderr );
				(void) fputs( new->search_str, stderr );
				(void) fputs( "]: ", stderr  );
				(void) fputs( err_msg, stderr );
				(void) fputc( '\n', stderr );
				(void) fclose( fptr );
				RETURN_INT( -1 );
			}

			if ( new->reg_exp.re_nsub > (size_t)64 ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Regular expression contain more than 64 sub-expressions.\n", stderr );
				(void) fclose( fptr );
				RETURN_INT( -1 );
			}

			alloc_size = sizeof( regmatch_t ) * ( new->reg_exp.re_nsub + (size_t)1 );
			new->regmatch = (regmatch_t *)malloc( alloc_size );
			if ( new->regmatch == (regmatch_t *)0 ) {
				UNIX_ERROR( "malloc() failed" );
				(void) fclose( fptr );
				RETURN_INT( -1 );
			}
			if ( Debug ) {
				fprintf( stderr, "reg expr [%s] re_nsub = %u\n", new->search_str, new->reg_exp.re_nsub );
			}
		}

		new->input_bind = (dfile_bind_t *)0;
		new->output_bind = (dfile_bind_t *)0;
		++trnsfrm_cnt;
	}

	(void) fclose( fptr );

	*ret_trnsfrm = trnsfrm;
	*ret_trnsfrm_cnt = trnsfrm_cnt;

	RETURN_INT( 0 );
}
