/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <assert.h>
#include <regex.h>
#include <string.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_transform.h"

#define REPLACE_SIZE	1024

static int validate_start_end( regoff_t, regoff_t, size_t );

/*
** This function transforms a regular expression.
*/
int trnsfrm_regex( char **output_field_value, size_t *output_field_len, const char *input_field_value, size_t input_field_len, trnsfrm_t *trnsfrm )
{
	unsigned short	ndx, trnsfrm_expr_cnt;
	regmatch_t	*regmatch;
	regoff_t	start, end;
	size_t	prefix_len, suffix_len, len;
	int	ret;
	char	replace[REPLACE_SIZE], replace_trnsfrm[REPLACE_SIZE];
	char	*result, err_msg[128], search[12];

	assert( output_field_value != (char **)0 );
	assert( output_field_len != (size_t *)0 );
	assert( input_field_value != (const char *)0 );
	assert( trnsfrm != (trnsfrm_t *)0 );

	DEBUG_FUNC_START;

	trnsfrm_expr_cnt = trnsfrm->reg_exp.re_nsub + 1;
	regmatch = trnsfrm->regmatch;

	ret = regexec( &trnsfrm->reg_exp, input_field_value, (size_t)trnsfrm_expr_cnt, regmatch, 0 );

	switch ( ret ) {
	case 0:
		break;

	case REG_NOMATCH:
		if ( Debug ) {
			(void) fprintf( stderr, "Field [%s]: value [%s] did not match [%s].\n", trnsfrm->input_field_name, input_field_value, trnsfrm->search_str );
		}
		RETURN_INT( 0 );

	default:
		(void) regerror( ret, &trnsfrm->reg_exp, err_msg, sizeof( err_msg ) );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "regexec() failed: ", stderr );
		(void) fputs( err_msg, stderr );
		(void) fputc( '\n', stderr );
		RETURN_INT( -1 );
	}

	/*
	** Go through replacement string and replace \1, \2, ... with
	** values matched by regular expressions surounded by \(\) in
	** search string.
	*/
	if ( trnsfrm->replacement_len + (size_t)1 >= (size_t)REPLACE_SIZE ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Replacement string length is greater than ", stderr );
		(void) fput_uint( sizeof( replace ), stderr );
		(void) fputs( ".\n", stderr );
		RETURN_INT( -1 );
	}

	(void) memcpy( (void *)replace, (void *)trnsfrm->replacement, trnsfrm->replacement_len + (size_t)1 );

	/*
	** First regmatch is used for entire regular expression.
	** Skip first regmatch when processing trnsfrm-expressions.
	*/
	++regmatch;

	for ( ndx = 1; ndx <= trnsfrm->reg_exp.re_nsub; ++ndx, ++regmatch ) {
		start = regmatch->rm_so;
		end = regmatch->rm_eo;
		if ( validate_start_end( start, end, input_field_len ) == -1 ) {
			RETURN_INT( -1 );
		}
		(void) snprintf( search, sizeof( search ), "\\%hu", ndx );
		len = end - start;
		(void) memcpy( (void *)replace_trnsfrm, (void *)( input_field_value + start ), len );
		replace_trnsfrm[ len ] = (char)0;
		if ( substitute_str( replace, sizeof( replace ), search, replace_trnsfrm ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	/*
	** Get start and end to regular express match.
	*/
	regmatch = trnsfrm->regmatch;
	start = regmatch->rm_so;
	end = regmatch->rm_eo;
	result = trnsfrm->result;

	if ( validate_start_end( start, end, input_field_len ) == -1 ) {
		RETURN_INT( -1 );
	}

	len = strlen( replace );
	prefix_len = (size_t)start + len;
	suffix_len = input_field_len - end;

	if ( prefix_len + suffix_len >= sizeof( trnsfrm->result ) ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Result length is longer than ", stderr );
		(void) fput_uint( sizeof( trnsfrm->result ), stderr );
		(void) fputs( ".\n", stderr );
		RETURN_INT( -1 );
	}

	/*
	** Regular expression may match value in middle of field value.
	*/
	(void) memcpy( (void *)result, (void *)input_field_value, start );

	(void) memcpy( (void *)&result[ start ], (void *)replace, len );

	(void) memcpy( (void *)&result[ prefix_len ], (void *)( input_field_value + end ), suffix_len + (size_t)1 );

	if ( Debug ) {
		(void) fprintf( stderr, "Field [%s], [%s]: replacing [%s] with [%s].\n", trnsfrm->input_field_name, trnsfrm->output_field_name, input_field_value, result );
	}

	*output_field_value = result;
	*output_field_len = prefix_len + suffix_len;

	RETURN_INT( 0 );
}

static int validate_start_end( regoff_t start, regoff_t end, size_t field_len )
{
	if ( start < (regoff_t)0 || end < (regoff_t)0 || start >= end || end > field_len ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "regexec() failed to assign regmatch_t properly.\n", stderr );
		return -1;
	}

	return 0;
}
