/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"


static int strip_comments( char **, size_t *, char *, size_t, char, char );

/*
** This function loads a script file into memory.
*/
int load_script_file( char **script, size_t *script_len, const char *fname, char comment )
{
	int	fd;
	char	*buf, *input, *new;
	size_t	alloc_size;
	size_t	input_len;
	const char	newline = '\n';
	ssize_t	read_cnt;

	assert( script != (char **)0 );
	assert( script_len != (size_t *)0 );
	assert( fname != (const char *)0 );
	assert( comment != newline );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( %p, %p, [%s], [%c] )\n", __func__, script, script_len, fname, ( comment == (char)0 ) ? ' ' : comment );
	}

	DEBUG_FUNC_START;

	*script = (char *)0;
	*script_len = (size_t)0;
	input = (char *)0;
	input_len = (size_t)0;

	if ( strcmp( fname, "-" ) == 0 ) {
		/*
		** Read from stdin.
		*/
		fd = 0;
	} else {
		fd = open( fname, O_RDONLY );
	}

	if ( fd < 0 ) {
		UNIX_ERROR( "open() failed" );
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Could not open script file [", stderr );
		(void) fputs( fname, stderr );
		(void) fputs( "].\n", stderr );
		RETURN_INT( -1 );
	}

	for ( ;; ) {
		alloc_size = sizeof( char ) * ( input_len + BUFSIZ + 1 );
		new = (char *)realloc( (void *)input, alloc_size );
		if ( new == (char *)0 ) {
			UNIX_ERROR( "realloc() failed" );
			RETURN_INT( -1 );
		}

		input = new;
		buf = &input[ input_len ];

		read_cnt = read( fd, (void *)buf, (size_t)BUFSIZ );

		if ( read_cnt == (ssize_t)0 ) {
			/*
			** End of file.
			*/
			break;
		}

		if ( read_cnt < (ssize_t)0 ) {
			if ( fd == 0 && errno == EBADF ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Standard input was not open for reading.\n", stderr );
				RETURN_INT( -1 );
			}

			UNIX_ERROR( "read() failed" );
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Failed to read script file [", stderr );
			(void) fputs( fname, stderr );
			(void) fputs( "].\n", stderr );
			RETURN_INT( -1 );
		}
	
		input_len += (size_t)read_cnt;
	}

	if ( fd > 0 ) {
		if ( close( fd ) == -1 ) {
			UNIX_ERROR( "close() failed" );
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Failed to close script file [", stderr );
			(void) fputs( fname, stderr );
			(void) fputs( "].\n", stderr );
			RETURN_INT( -1 );
		}
	}

	if ( comment == (char)0 ) {
		input[ input_len ] = (char)0;
		*script = input;
		*script_len = input_len;
	} else {
		if ( Debug ) {
			(void) fputs( "script with comments:\n", stderr );
			(void) fputc( '[', stderr );
			(void) fputs( input, stderr );
			(void) fputs( "]\n", stderr );
		}
		if ( strip_comments( script, script_len, input, input_len, comment, newline ) == -1 ) {
			RETURN_INT( -1 );
		}

		free( (void *)input );
	}

	if ( Debug ) {
		(void) fputs( "script:\n", stderr );
		(void) fputc( '[', stderr );
		(void) fputs( *script, stderr );
		(void) fputs( "]\n", stderr );
	}

	RETURN_INT( 0 );
}

static int strip_comments( char **result, size_t *result_len, char *input, size_t input_len, char comment, char newline )
{
	char	*new, *comment_ptr, *remaining, *newline_ptr;
	size_t	alloc_size, cp_len, remaining_len, new_len, comment_len;

	*result = (char *)0;
	*result_len = (size_t)0;

	alloc_size = sizeof( char ) * ( input_len + (size_t)1 );
	new = (char *)malloc( alloc_size );
	if ( new == (char *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		return -1;
	}

	remaining = input;
	remaining_len = input_len;
	new_len = (size_t)0;

	while ( remaining_len > (size_t)0 ) {
		comment_ptr = (char *)memchr( (void *)remaining, (int)comment, remaining_len );
		if ( comment_ptr == (char *)0 ) {
			/*
			** No comment character found in remainder
			** of input string.
			*/
			(void) memcpy( (void *)&new[ new_len ], (void *)remaining, remaining_len );
			new_len += remaining_len;
			break;
		}

		assert( comment_ptr >= remaining );
		cp_len = comment_ptr - remaining;

		if ( comment_ptr > input && comment_ptr[ -1 ] != '\n' ) {
			/*
			** Comment character not at beginning of line.
			** This is not a comment.
			*/
			++cp_len;
			assert( remaining_len >= cp_len );
			(void) memcpy( (void *)&new[ new_len ], (void *)remaining, cp_len );
			new_len += cp_len;
			assert( input_len >= new_len );
			remaining += cp_len;
			assert( input + input_len >= remaining );
			remaining_len -= cp_len;
			continue;
		}

		assert( remaining_len >= cp_len );
		(void) memcpy( (void *)&new[ new_len ], remaining, cp_len );
		new_len += cp_len;
		assert( input_len >= new_len );
		remaining += cp_len;
		remaining_len -= cp_len;

		if ( remaining_len > (size_t)0 ) {
			newline_ptr = (char *)memchr( (void *)&comment_ptr[ 1 ], (int)newline, remaining_len );
			if ( newline_ptr == (char *)0 ) {
				/*
				** Discard remaining input.
				*/
				break;
			}
			comment_len = (size_t)( newline_ptr - comment_ptr ) + (size_t)1;
			remaining += comment_len;
			assert( remaining_len >= comment_len );
			remaining_len -= comment_len;
		}
	}

	new[ new_len ] = (char)0;

	*result = new;
	*result_len = new_len;

	return 0;
}

#if 0
int main( int argc, char **argv )
{
	size_t	len;
	char	*ptr;

	if ( argc != 2 ) {
		(void) fputs( "one command line argument needed\n", stderr );
		return 10;
	}

	Debug = 1;

	if ( load_script_file( &ptr, &len, argv[ 1 ], '#' ) == -1 ) {
		return 20;
	}
	return 0;
}
#endif
