/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <regex.h>
#include <assert.h>

#include "tbox.h"

#define	NUM_OF_MATCHES	4


static int append_result( char **, size_t *, const char *, size_t );
static void reg_error( const char *, int, const regex_t *, int );

/*
** This function expands environment variables embedded in strings.
** It expects variables to be specified in format ${HOME} or $(HOME).
*/

char *strenv( const char *orig_str )
{
	static const char	env_var_fmt[] =
		"([$][ \t]*[{(][ \t]*)"		/* start syntax ${ or $( */
		"([A-Za-z][_A-Za-z0-9]*)"	/* variable name */
		"([ \t]*[})])";			/* end syntax } or ) */

	static regex_t	env_var;
	regmatch_t	match[NUM_OF_MATCHES];
	int	env_var_status;
	static int	first_pass = 1;
	const char	*ptr, *env_value;
	char	*var_name;
	static char	*result;
	size_t	var_name_len, result_len;

	assert( orig_str != (const char *) 0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( [%s] )\n", __func__, orig_str );
	}

	DEBUG_FUNC_START;

	if ( first_pass ) {
		if ( Debug ) {
			(void) fprintf( stderr, "Compiling format [%s]\n", env_var_fmt );
		}
		if ( ( env_var_status = regcomp( &env_var, env_var_fmt, REG_EXTENDED ) ) != 0 ) {
			reg_error( "regcomp", env_var_status, &env_var, __LINE__ );

			RETURN_POINTER( (char *)0 );
		}

		first_pass = 0;
	}

	ptr = orig_str;
	result_len = (size_t)0;

	/*
	** This loop searches to expand all environment variables
	** in string that was passed into this function. Results
	** are placed in dynamicly allocated character array.
	*/
	for ( ;; ) {
		/*
		** Search for occurrence of environment variable in string.
		*/
		env_var_status = regexec( &env_var, ptr, (size_t)NUM_OF_MATCHES, match, 0 );

		if ( env_var_status == REG_NOMATCH ) {
			/*
			** Entire string has been searched.
			*/
			break;
		}

		if ( env_var_status != 0 ) {
			/*
			** regexec() had an unexpected failure.
			*/
			reg_error( "regexec", env_var_status, &env_var, __LINE__ );

			RETURN_POINTER( (char *)0 );
		}

		/*
		** Append examined part of the string prior to
		** environment variable to result.
		*/
		if ( append_result( &result, &result_len, ptr, (size_t)match[ 1 ].rm_so ) == -1 ) {
			RETURN_POINTER( (char *)0 );
		}

		/*
		** Copy embedded environment variable name into
		** a temporary variable.
		*/
		assert( match[ 2 ].rm_eo > match[ 2 ].rm_so );
		var_name_len = (size_t) ( match[ 2 ].rm_eo - match[ 2 ].rm_so );
		var_name = (char *)malloc( var_name_len + (size_t)1 );
		if ( var_name == (char *)0 ) {
			UNIX_ERROR( "malloc() failed" );
			RETURN_POINTER( (char *)0 );
		}

		(void) strncpy( var_name, &ptr[ match[ 2 ].rm_so ], var_name_len );
		var_name[ var_name_len ] = (char)0;

		/*
		** Lookup value of environment variable.
		*/
		env_value = getenv( var_name );
		if ( env_value == (const char *)0 ) {
			/*
			** Environment variable was not found.
			*/
			(void) fputs( __FILE__, stderr );
			(void) fprintf( stderr, "(%d)", __LINE__ );
			(void) fputs( ": Environment variable [", stderr );
			(void) fputs( var_name, stderr );
			(void) fputs( "] was not defined.\n", stderr );
			free( var_name );
			RETURN_POINTER( (char *)0 );
		}

		free( var_name );

		/*
		** Append environment variable value to result.
		*/
		if ( append_result( &result, &result_len, env_value, strlen( env_value ) ) == -1 ) {
			RETURN_POINTER( (char *)0 );
		}

		/*
		** Advance ptr past environment variable usage in string.
		*/
		ptr = &ptr[ match[ 3 ].rm_eo ];
	}

	/*
	** Append last examined part of the string
	** to result.
	*/
	if ( append_result( &result, &result_len, ptr, strlen( ptr ) ) == -1 ) {
		RETURN_POINTER( (char *)0 );
	}

	if ( Debug ) {
		(void) fprintf( stderr, "expanded result [%s]\n", result );
	}

	RETURN_POINTER( result );
}

static int append_result( char **result, size_t *result_len, const char *append_value, size_t append_value_len )
{
	static size_t	result_size = 0;
	size_t	alloc_size;
	char	*new;

	alloc_size = ( *result_len + append_value_len + (size_t)1 ) * sizeof( char );

	if ( alloc_size > result_size ) {
		/*
		** Expand result area.
		*/
		new = realloc( (void *)*result, alloc_size );
		if ( new == (char *)0 ) {
			UNIX_ERROR( "realloc() failed" );
			return -1;
		}

		*result = new;
		result_size = alloc_size;
	}

	(void) strncpy( &( *result )[ *result_len ], append_value, append_value_len );
	*result_len += append_value_len;
	( *result )[ *result_len ] = (char)0;

	if ( Debug ) {
		(void) fprintf( stderr, "result [%s]\n", *result );
	}

	return 0;
}

static void reg_error( const char *func, int error_code, const regex_t *regexp, int line )
{
	char    msg[100];

	(void) regerror( error_code, regexp, msg, sizeof( msg ) );
	(void) fputs( __FILE__, stderr );
	(void) fprintf( stderr, "(%d)", line );
	(void) fputs( ": ", stderr );
	(void) fputs( func, stderr );
	(void) fputs( "() failed [", stderr );
	(void) fputs( msg, stderr );
	(void) fputs( "]\n", stderr );
}


#ifdef MT_strenv

#include <stdlib.h>
/*
** This function is used to regression test strenv().
** The following command is used to compile:
**   x=strenv; make "MT_CC=-DMT_$x" "MT_PRE=DEFINE=MT_$x" $x
*/
int main( void )

{
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	test_func[] = "strenv";
	static const char	incorrectly_successful[] = ">>>\n>>> %s() was incorrectly successful.\n";
	static const char	correctly_unsuccessful[] = ">>>\n>>> %s() was correctly unsuccessful.\n";
	static const char	correctly_successful[] = ">>>\n>>> %s() was correctly successful.\n";
	static const char	incorrectly_unsuccessful[] = ">>>\n>>> %s() was incorrectly unsuccessful.\n";
	static const char	blank_line[] = ">>>\n";
	const char	*ptr, *home;
	char	str[1024];

	Debug = 1;

	(void) fprintf( stderr, ">>> Start module test on function %s().\n", test_func );
	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #1\n", stderr );
	(void) fputs( ">>> Check failure for undefined environment variable value.\n", stderr );
	(void) fputs( blank_line, stderr );

	if ( strenv( "ABC/${HELLO}/DEFG" ) != (const char *)0 ) {
		(void) fprintf( stderr, incorrectly_successful, test_func );
	} else {
		(void) fprintf( stderr, correctly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #2\n", stderr );
	(void) fputs( ">>> Check for valid environment variable expansion.\n", stderr );
	(void) fputs( blank_line, stderr );

	home = getenv( "HOME" );
	if ( home == (const char *)0 ) {
		fputs( ">>> HOME environment variable was not set.\n", stderr );
	} else {
		assert( strlen( ptr ) < sizeof( str ) );
		(void) strcpy( str, home );
		(void) strcat( str, "/DEFG" );

		ptr = strenv( "${HOME}/DEFG" );

		if ( strcmp( ptr, str ) == 0 ) {
			(void) fprintf( stderr, correctly_successful, test_func );
		} else {
			(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
		}

	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #3\n", stderr );
	(void) fputs( ">>> Check for valid environment variable expansion.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( str, "/ABC" );
	(void) strcat( str, home );

	ptr = strenv( "/ABC$(HOME)" );

	if ( strcmp( ptr, str ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fputs( ">>> TEST CASE #4\n", stderr );
	(void) fputs( ">>> Check for valid environment variable expansion.\n", stderr );
	(void) fputs( blank_line, stderr );

	(void) strcpy( str, "/ABC" );
	(void) strcat( str, home );
	(void) strcat( str, "/XYZ" );

	ptr = strenv( "/ABC${HOME}/XYZ" );

	if ( strcmp( ptr, str ) == 0 ) {
		(void) fprintf( stderr, correctly_successful, test_func );
	} else {
		(void) fprintf( stderr, incorrectly_unsuccessful, test_func );
	}

	(void) fputs( blank_line, stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}
#endif
