/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "sexpr.h"
#include "dfile.h"
#include "where.h"
#include "_where.h"


/*
** This function is part of the interpreter for comparing data values.
*/

#define NUMERIC_CONSTANT( x )	( ( DATUM_TYPE( x ) == Constant && DATUM_CONST_TYPE( x ) == Real ) )

static int compare_numeric_datum( int *, datum_t *, datum_t * );
static int convert_to_numeric( double *, const char * );
static int assign_numeric( double *, datum_t * );
static const char *assign_string( datum_t * );

int _where_compare_condition( int *ret_result, multi_datum_t *multi_datum )
{
	datum_t	*lhs, *rhs;
	const char *lhs_str, *rhs_str;

	assert( ret_result != (int *)0 );
	assert( multi_datum != (multi_datum_t *)0 );

	DEBUG_FUNC_START;

	/*
	** Decide if comparison is to be done as ASCII strings or
	** double floating point.
	*/

	assert( multi_datum->datum_cnt == 2UL );

	lhs = &multi_datum->datum[ 0 ];
	assert( lhs != (datum_t *)0 );

	rhs = &multi_datum->datum[ 1 ];
	assert( rhs != (datum_t *)0 );

	if ( NUMERIC_CONSTANT( lhs ) || NUMERIC_CONSTANT( rhs ) ) {
		if ( compare_numeric_datum( ret_result, lhs, rhs ) == -1 ) {
			RETURN_INT( WHERE_INVLDNBR );
		}
	} else {
		lhs_str = assign_string( lhs );
		rhs_str = assign_string( rhs );
		*ret_result = strcmp( lhs_str, rhs_str );

		if ( Debug ) {
			(void) fprintf( stderr, "compared string values [%s], [%s]\n", lhs_str, rhs_str );
		}
	}

	if ( Debug ) {
		(void) fprintf( stderr, "result = %d\n", *ret_result );
	}

	RETURN_INT( WHERE_NOERR );
}

static int compare_numeric_datum( int *ret_result, datum_t *lhs_datum, datum_t *rhs_datum )
{
	double	lhs, rhs;

	if ( assign_numeric( &lhs, lhs_datum ) == -1 ) {
		return -1;
	}

	if ( assign_numeric( &rhs, rhs_datum ) == -1 ) {
		return -1;
	}

	if ( lhs < rhs ) {
		*ret_result = -1;
	} else {
		if ( lhs > rhs ) {
			*ret_result = 1;
		} else {
			*ret_result = 0;
		}
	}

	if ( Debug ) {
		(void) fprintf( stderr, "compared numeric values %g, %g\n", lhs, rhs );
	}

	return 0;
}

static int assign_numeric( double *result, datum_t *datum )
{
	int	ret;

	if ( DATUM_TYPE( datum ) == Constant ) {
		if ( DATUM_CONST_TYPE( datum ) == Real ) {
			*result = DATUM_CONST_REAL( datum );
			ret = 0;
		} else {
			ret = convert_to_numeric( result, DATUM_CONST_LITERAL( datum ) );
		}
	} else {
		assert( DATUM_TYPE( datum ) == Variable );
		ret = convert_to_numeric( result, DATUM_VAR_BIND_STR( datum ) );
	}

	return ret;
}

static int convert_to_numeric( double *result, const char *str )
{
	const char	*end;

	*result = strtod( str, (char **)&end );

	if ( str == end || *end != (char)0 ) {
		return -1;
	}

	return 0;
}

static const char *assign_string( datum_t *datum )
{
	if ( DATUM_TYPE( datum ) == Constant ) {
		assert( DATUM_CONST_TYPE( datum ) != Real );
		return DATUM_CONST_LITERAL( datum );
	}

	assert( DATUM_TYPE( datum ) == Variable );
	return DATUM_VAR_BIND_STR( datum );
}
