/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "dfile_dynamic.h"
#include "sexpr.h"
#include "where.h"
#include "dfile_diff.h"

static int input_key_mismatch( int *, dfile_t *, dfile_t *, unsigned short * );
static int read_rec( int *, dfile_t * );


/*
*/
int diff_records( input_dfile_t *original_dfile, input_dfile_t *subsequent_dfile, output_dfile_t *added_dfile, output_dfile_t *deleted_dfile, dfile_t *key_dfile, change_data_t *change_data, dfile_t *change_data_dfile, unsigned short *key_map_tbl, unsigned short key_field_tbl_cnt, dfile_bind_t **non_key_bind_tbl )
{
	int	original_eof_flag = 0, subsequent_eof_flag = 0;
	int	key_cmp, non_key_cmp;
	unsigned long change_seq_nbr = 0;
	char	*change_seq_nbr_desc[] = { "change_seq_nbr" };
	dfile_bind_t	**change_seq_nbr_bind_tbl, *change_seq_nbr_bind;
	char	change_seq_nbr_str[24];
	const size_t	change_seq_nbr_str_len = 8;

	assert( original_dfile != (input_dfile_t *)0 );
	assert( subsequent_dfile != (input_dfile_t *)0 );
	assert( added_dfile != (output_dfile_t *)0 );
	assert( deleted_dfile != (output_dfile_t *)0 );
	assert( key_field_tbl_cnt > (unsigned short)0 );
	assert( non_key_bind_tbl != (dfile_bind_t **)0 );
	assert( change_data != (change_data_t *)0 );

	DEBUG_FUNC_START;

	if ( snprintf( change_seq_nbr_str, sizeof( change_seq_nbr_str ), "%0*lu", change_seq_nbr_str_len, change_seq_nbr + 1UL ) != change_seq_nbr_str_len ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Failed to create change sequence number string.\n", stderr );
		RETURN_INT( -1 );
	}

	if ( assign_field_bind( &change_seq_nbr_bind_tbl, change_seq_nbr_desc, (unsigned short)1, key_dfile->sorted_bind, key_dfile->bind_cnt ) == -1 ) {
		RETURN_INT( -1 );
	}

	change_seq_nbr_bind = *change_seq_nbr_bind_tbl;

	if ( change_data_dfile != (dfile_t *)0 ) {
		change_data->change_seq_nbr = change_seq_nbr_str; 
		change_data->change_seq_nbr_len = change_seq_nbr_str_len; 
	}

	if ( read_rec( &original_eof_flag, original_dfile->dfile ) == -1 ) {
		RETURN_INT( -1 );
	}

	if ( read_rec( &subsequent_eof_flag, subsequent_dfile->dfile ) == -1 ) {
		RETURN_INT( -1 );
	}

	while ( !original_eof_flag && !subsequent_eof_flag ) {
		key_cmp = compare_bind( (const dfile_bind_t **)original_dfile->key_bind_tbl, (const dfile_bind_t **)subsequent_dfile->key_bind_tbl, key_field_tbl_cnt );

		if ( key_cmp < 0 ) {
			/*
			** Found deleted record.
			*/
			if ( input_key_mismatch( &original_eof_flag, deleted_dfile->dfile, original_dfile->dfile, deleted_dfile->input_map_tbl ) == -1 ) {
				RETURN_INT( -1 );
			}

			continue;
		}

		if ( key_cmp > 0 ) {
			/*
			** Found deleted record.
			*/
			if ( input_key_mismatch( &subsequent_eof_flag, added_dfile->dfile, subsequent_dfile->dfile, added_dfile->input_map_tbl ) == -1 ) {
				RETURN_INT( -1 );
			}

			continue;
		}

		if ( compare_non_key_data( &non_key_cmp, change_data, change_data_dfile, original_dfile->dfile->bind, original_dfile->dfile->bind_cnt, non_key_bind_tbl ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( non_key_cmp != 0 ) {
			if ( key_dfile != (dfile_t *)0 ) {
				copy_input_to_output( key_dfile->bind, original_dfile->dfile->bind, key_map_tbl, key_dfile->bind_cnt );


				*change_seq_nbr_bind->field_buffer = change_seq_nbr_str;
				*change_seq_nbr_bind->field_length = change_seq_nbr_str_len;

				if ( dfile_write( key_dfile ) == -1 ) {
					RETURN_INT( -1 );
				}
			}

			++change_seq_nbr;
			if ( snprintf( change_seq_nbr_str, sizeof( change_seq_nbr_str ), "%0*lu", change_seq_nbr_str_len, change_seq_nbr + 1UL ) != change_seq_nbr_str_len ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Failed to create change sequence number string.\n", stderr );
				RETURN_INT( -1 );
			}
		}

		if ( read_rec( &original_eof_flag, original_dfile->dfile ) == -1 ) {
			RETURN_INT( -1 );
		}

		if ( read_rec( &subsequent_eof_flag, subsequent_dfile->dfile ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	while ( !original_eof_flag ) {
		if ( input_key_mismatch( &original_eof_flag, deleted_dfile->dfile, original_dfile->dfile, deleted_dfile->input_map_tbl ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	while ( !subsequent_eof_flag ) {
		if ( input_key_mismatch( &subsequent_eof_flag, added_dfile->dfile, subsequent_dfile->dfile, added_dfile->input_map_tbl ) == -1 ) {
			RETURN_INT( -1 );
		}
	}

	RETURN_INT( 0 );
}

static int input_key_mismatch( int *eof_flag, dfile_t *output_dfile, dfile_t *input_dfile, unsigned short *input_map_tbl )
{
	if ( output_dfile != (dfile_t *)0 ) {
		copy_input_to_output( output_dfile->bind, input_dfile->bind, input_map_tbl, output_dfile->bind_cnt );

		if ( dfile_write( output_dfile ) == -1 ) {
			return -1;
		}
	}

	if ( read_rec( eof_flag, input_dfile ) == -1 ) {
		return -1;
	}

	return 0;
}

static int read_rec( int *eof_flag, dfile_t *dfile )
{
	if ( dfile_read( dfile ) == -1 ) {
		if ( dfile->error != Dfile_all_data_processed ) {
			return -1;
		}

		/*
		** Set EOF flag to true.
		*/
		*eof_flag = 1;
	}

	return 0;
}
