/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dfile_utility.h"
#include "dfile_diff.h"


static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], char **original_dfile_name, char **subsequent_dfile_name, char **added_dfile_name, char **deleted_dfile_name, dfile_tag_t **dfile_tag_tbl, unsigned short *dfile_tag_tbl_cnt, char **key_fields, char **original_filter_file_name, char **subsequent_filter_file_name, char **key_dfile_name, char **change_data_dfile_name )
{
	int	ch, ndx;
	extern char	*optarg;

	assert( argv != (char * const *) 0 );
	assert( original_dfile_name != (char **) 0 );
	assert( subsequent_dfile_name != (char **) 0 );
	assert( added_dfile_name != (char **) 0 );
	assert( deleted_dfile_name != (char **) 0 );
	assert( dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( key_fields != (char **)0 );
	assert( original_filter_file_name != (char **)0 );
	assert( subsequent_filter_file_name != (char **)0 );
	assert( key_dfile_name != (char **) 0 );
	assert( change_data_dfile_name != (char **) 0 );

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	*original_dfile_name = (char *)0;
	*subsequent_dfile_name = (char *)0;
	*added_dfile_name = (char *)0;
	*deleted_dfile_name = (char *)0;
	*dfile_tag_tbl = (dfile_tag_t *)0;
	*dfile_tag_tbl_cnt = (unsigned short)0;
	*key_fields = (char *)0;
	*original_filter_file_name = (char *)0;
	*subsequent_filter_file_name = (char *)0;
	*key_dfile_name = (char *)0;
	*change_data_dfile_name = (char *)0;

	while ( ( ch = getopt( argc, argv, "a:b:c:di:j:k:l:t:x:y:" ) ) != EOF ) {
		switch ( ch ) {
		case 'a':
			*added_dfile_name = optarg;
			break;
		case 'b':
			*deleted_dfile_name = optarg;
			break;
		case 'c':
			*key_dfile_name = optarg;
			break;
		case 'd':
#ifdef NDEBUG
			(void) fputs( "Not compiled for debug.\n", stderr );
#else
			Debug = 1;
#endif
			break;
		case 'i':
			*original_dfile_name = optarg;
			break;
		case 'j':
			*subsequent_dfile_name = optarg;
			break;
		case 'k':
			*key_fields = optarg;
			break;
		case 'l':
			*change_data_dfile_name = optarg;
			break;
		case 't':
			if ( cmd_line_parse_tag( dfile_tag_tbl, dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'x':
			*original_filter_file_name = optarg;
			break;
		case 'y':
			*subsequent_filter_file_name = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *original_dfile_name == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify original dfile name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *subsequent_dfile_name == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify subsequent dfile name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *key_fields == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify unique key fields.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Original DFile Name: [%s]\n", NULL_CHECK( *original_dfile_name ) );
		(void) fprintf( stderr, "Subsequent DFile Name: [%s]\n", NULL_CHECK( *subsequent_dfile_name ) );
		(void) fprintf( stderr, "Added DFile Name: [%s]\n", NULL_CHECK( *added_dfile_name ) );
		(void) fprintf( stderr, "Deleted DFile Name: [%s]\n", NULL_CHECK( *deleted_dfile_name ) );
		(void) fprintf( stderr, "Key DFile Name: [%s]\n", NULL_CHECK( *key_dfile_name ) );
		(void) fprintf( stderr, "Change Data DFile Name: [%s]\n", NULL_CHECK( *change_data_dfile_name ) );
		(void) fprintf( stderr, "Original Filter File Name: [%s]\n", NULL_CHECK( *original_filter_file_name ) );
		(void) fprintf( stderr, "Subsequent Filter File Name: [%s]\n", NULL_CHECK( *subsequent_filter_file_name ) );
		(void) fputs( "Input files:\n", stderr );
		for ( ndx = (unsigned short)0; ndx < *dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "DFile Tag [%s] [%s]\n", ( *dfile_tag_tbl )[ ndx ].tag, ( *dfile_tag_tbl )[ ndx ].tag_value );
		}
		(void) fprintf( stderr, "Key Fields: [%s]\n", NULL_CHECK( *key_fields ) );
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-a dfile]", stderr );
	(void) fputs( " [-b dfile]", stderr );
	(void) fputs( " [-c dfile]", stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " -i dfile", stderr );
	(void) fputs( " -j dfile", stderr );
	(void) fputs( " -k field,field...", stderr );
	(void) fputs( " [-l dfile]", stderr );
	(void) fputs( " [-t %x=abc]", stderr );
	(void) fputs( " [-x file]", stderr );
	(void) fputs( " [-y file]", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-a -> output dfile containing added records\n", stderr );
	(void) fputs( "\t-b -> output dfile containing deleted records\n", stderr );
	(void) fputs( "\t-c -> output dfile containing change data\n", stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-i -> input original DFile\n", stderr );
	(void) fputs( "\t-j -> input subsequent DFile\n", stderr );
	(void) fputs( "\t-k -> key fields\n", stderr );
	(void) fputs( "\t-l -> output dfile containing key values of changed records\n", stderr );
	(void) fputs( "\t-t -> DFile path tags\n", stderr );
	(void) fputs( "\t-x -> input original filter file\n", stderr );
	(void) fputs( "\t-y -> input subsequent filter file\n", stderr );
}
