/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dfile_partition.h"


static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], char **input_dfile_name, char **output_dfile_name, int *thread_input_flag, int *thread_output_flag, dfile_tag_t **dfile_tag_tbl, unsigned short *dfile_tag_tbl_cnt, dfile_tag_t **input_dfile_tag_tbl, unsigned short *input_dfile_tag_tbl_cnt, dfile_tag_t **output_dfile_tag_tbl, unsigned short *output_dfile_tag_tbl_cnt, char **output_partition_tag, char **filter_file_name, char **partition_field_name, unsigned short *hash_slot_cnt, char **range_partition_file )
{
	int	slot_cnt;
	int	ch, ndx;
	extern char	*optarg;

	assert( argv != (char * const *) 0 );
	assert( input_dfile_name != (char **) 0 );
	assert( output_dfile_name != (char **) 0 );
	assert( thread_input_flag != (int *) 0 );
	assert( thread_output_flag != (int *) 0 );
	assert( dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( input_dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( input_dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( output_dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( output_dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( output_partition_tag != (char **) 0 );
	assert( filter_file_name != (char **) 0 );
	assert( partition_field_name != (char **) 0 );
	assert( hash_slot_cnt != (unsigned short *) 0 );
	assert( range_partition_file != (char **) 0 );

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	*input_dfile_name = (char *)0;
	*output_dfile_name = (char *)0;
	*dfile_tag_tbl = (dfile_tag_t *)0;
	*dfile_tag_tbl_cnt = (unsigned short)0;
	*input_dfile_tag_tbl = (dfile_tag_t *)0;
	*input_dfile_tag_tbl_cnt = (unsigned short)0;
	*output_dfile_tag_tbl = (dfile_tag_t *)0;
	*output_dfile_tag_tbl_cnt = (unsigned short)0;
	*filter_file_name = (char *)0;
	*partition_field_name = (char *)0;
	*hash_slot_cnt = (unsigned short)0;
	*range_partition_file = (char *)0;
	*output_partition_tag = (char *) 0;

	/*
	** Initialize I/O threading flags to false.
	*/
	*thread_input_flag = 0;
	*thread_output_flag = 0;

	while ( ( ch = getopt( argc, argv, "df:h:i:l:o:p:t:u:v:rwx:" ) ) != EOF ) {
		switch ( ch ) {
		case 'd':
#ifdef NDEBUG
			(void) fputs( "Not compiled for debug.\n", stderr );
#else
			Debug = 1;
#endif
			break;
		case 'f':
			*partition_field_name = optarg;
			break;
		case 'h':
			slot_cnt = atoi( optarg );
			if ( slot_cnt < 0 || slot_cnt > USHRT_MAX ) {
				/*
				** out of range
				*/
				*hash_slot_cnt = (unsigned short)0;
			} else {
				*hash_slot_cnt = (unsigned short)slot_cnt;
			}
			break;
		case 'i':
			*input_dfile_name = optarg;
			break;
		case 'l':
			*range_partition_file = optarg;
			break;
		case 'o':
			*output_dfile_name = optarg;
			break;
		case 'p':
			*output_partition_tag = optarg;
			break;
		case 'r':
			/*
			** Set threaded input to true.
			*/
			*thread_input_flag = 1;
			break;
		case 't':
			if ( cmd_line_parse_tag( dfile_tag_tbl, dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'u':
			if ( cmd_line_parse_tag( input_dfile_tag_tbl, input_dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'v':
			if ( cmd_line_parse_tag( output_dfile_tag_tbl, output_dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'w':
			/*
			** Set threaded output to true.
			*/
			*thread_output_flag = 1;
			break;
		case 'x':
			*filter_file_name = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *input_dfile_name == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify input dfile name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *output_dfile_name == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify output dfile name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *partition_field_name == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify partition field name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *output_partition_tag == (char *) 0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify output partition tag.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *hash_slot_cnt == (unsigned short) 0 && *range_partition_file == (char *)0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Must specify hash slot count or range partition file.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *hash_slot_cnt > (unsigned short) 0 && *range_partition_file != (char *)0 ) {
		fput_src_code( __FILE__, __LINE__, stderr );
		(void) fputs( "Cannot specify both hash slot count and range partition file.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Input DFile Name: [%s]\n", *input_dfile_name );
		(void) fprintf( stderr, "Output DFile Name: [%s]\n", *input_dfile_name );
		(void) fprintf( stderr, "Filter File Name: [%s]\n", *filter_file_name );
		(void) fprintf( stderr, "Thread Input Flag: [%s]\n", ( *thread_input_flag ) ? "true" : "false" );
		(void) fprintf( stderr, "Thread Output Flag: [%s]\n", ( *thread_output_flag ) ? "true" : "false" );
		(void) fprintf( stderr, "Partition Field Name: [%s]\n", *partition_field_name );
		(void) fprintf( stderr, "Output Partition Tag: [%s]\n", ( *output_partition_tag ) ? *output_partition_tag : "null" );
		(void) fprintf( stderr, "Range Partition File: [%s]\n", ( *range_partition_file ) ? *range_partition_file : "null" );
		(void) fprintf( stderr, "Hash Slot Count: [%hu]\n", *hash_slot_cnt );
		for ( ndx = (unsigned short)0; ndx < *dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "DFile Tag [%s] [%s]\n", ( *dfile_tag_tbl )[ ndx ].tag, ( *dfile_tag_tbl )[ ndx ].tag_value );
		}
		for ( ndx = (unsigned short)0; ndx < *input_dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "Input DFile Tag [%s] [%s]\n", ( *input_dfile_tag_tbl )[ ndx ].tag, ( *input_dfile_tag_tbl )[ ndx ].tag_value );
		}
		for ( ndx = (unsigned short)0; ndx < *output_dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "Input DFile Tag [%s] [%s]\n", ( *output_dfile_tag_tbl )[ ndx ].tag, ( *output_dfile_tag_tbl )[ ndx ].tag_value );
		}
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " -f field", stderr );
	(void) fputs( " [-h n]", stderr );
	(void) fputs( " -i dfile", stderr );
	(void) fputs( " [-l file]", stderr );
	(void) fputs( " -o dfile", stderr );
	(void) fputs( " -p %x", stderr );
	(void) fputs( " [-t %x=abc]", stderr );
	(void) fputs( " [-u %x=abc]", stderr );
	(void) fputs( " [-v %x=abc]", stderr );
	(void) fputs( " [-r]", stderr );
	(void) fputs( " [-w]", stderr );
	(void) fputs( " [-x file]", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-f -> partition field name\n", stderr );
	(void) fputs( "\t-h -> hash slot count\n", stderr );
	(void) fputs( "\t-i -> input DFile\n", stderr );
	(void) fputs( "\t-l -> range partition file name\n", stderr );
	(void) fputs( "\t-o -> output DFile\n", stderr );
	(void) fputs( "\t-p -> output partition tag\n", stderr );
	(void) fputs( "\t-t -> DFile path tags\n", stderr );
	(void) fputs( "\t-u -> input DFile path tags\n", stderr );
	(void) fputs( "\t-v -> output DFile path tags\n", stderr );
	(void) fputs( "\t-r -> thread input (default no threading)\n", stderr );
	(void) fputs( "\t-w -> thread output (default no threading)\n", stderr );
	(void) fputs( "\t-x -> filter file name\n", stderr );
}
