/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stddef.h>
#include <stdio.h>
#include <string.h>
#include <assert.h>
#include <limits.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_utility.h"
#include "sexpr.h"
#include "dfile_dynamic.h"
#include "dfile_sort.h"


static int verify_not_key_field( unsigned short *, unsigned short, unsigned short, const char *, const char * );
static unsigned short calc_ndx( const dfile_bind_t *, const dfile_bind_t * );

/*
** This function creates a map between input and output record layouts.
*/
int map_io_fields( unsigned short **io_field_map, dfile_bind_t **input_sorted_bind, dfile_bind_t *input_bind, unsigned short input_bind_cnt, dfile_bind_t **output_sorted_bind, dfile_bind_t *output_bind, unsigned short output_bind_cnt, unsigned short *key_ndx_tbl, unsigned short key_tbl_cnt, const char *input_dfile_name )
{
	size_t	alloc_size;
	unsigned short	input_bind_ndx, output_bind_ndx, mapped_cnt;
	unsigned short	*map;
	int	cmp;

	assert( input_sorted_bind != (dfile_bind_t **)0 );
	assert( io_field_map != (unsigned short **)0 );
	assert( input_bind != (dfile_bind_t *)0 );
	assert( output_sorted_bind != (dfile_bind_t **)0 );
	assert( output_bind != (dfile_bind_t *)0 );

	DEBUG_FUNC_START;

	*io_field_map = (unsigned short *)0;

	alloc_size = sizeof( unsigned short ) * output_bind_cnt;
	map = (unsigned short *)malloc( alloc_size );
	if ( map == (unsigned short *)0 ) {
		UNIX_ERROR( "malloc() failed" );
		RETURN_INT( -1 );
	}

	mapped_cnt = (unsigned short)0;
	*io_field_map = map;

	while ( input_bind_cnt > (unsigned short)0 && output_bind_cnt > (unsigned short)0 ) {
		cmp = dfile_bind_field_name_cmp( (void *)input_sorted_bind, (void *)output_sorted_bind );

		if ( cmp > 0 ) {
			output_bind_ndx = calc_ndx( *output_sorted_bind, output_bind );
			map[ output_bind_ndx ] = USHRT_MAX;
			++output_sorted_bind;
			--output_bind_cnt;
			continue;
		}

		if ( cmp < 0 ) {
			input_bind_ndx = calc_ndx( *input_sorted_bind, input_bind );
			if ( verify_not_key_field( key_ndx_tbl, key_tbl_cnt, input_bind_ndx, input_dfile_name, ( *input_sorted_bind )->field_name ) == -1 ) {
				RETURN_INT( -1 );
			}

			/*
			** Missing field was not part of key.
			*/

			++input_sorted_bind;
			--input_bind_cnt;
			continue;
		}

		output_bind_ndx = calc_ndx( *output_sorted_bind, output_bind );
		input_bind_ndx = calc_ndx( *input_sorted_bind, input_bind );

		map[ output_bind_ndx ] = input_bind_ndx;

		++mapped_cnt;
		++input_sorted_bind;
		--input_bind_cnt;
		++output_sorted_bind;
		--output_bind_cnt;
	}

	while ( input_bind_cnt > (unsigned short)0 ) {
		input_bind_ndx = calc_ndx( *input_sorted_bind, input_bind );

		if ( verify_not_key_field( key_ndx_tbl, key_tbl_cnt, input_bind_ndx, input_dfile_name, ( *input_sorted_bind )->field_name ) == -1 ) {
			RETURN_INT( -1 );
		}

		/*
		** Missing field was not part of key.
		*/

		++input_sorted_bind;
		--input_bind_cnt;
	}

	while ( output_bind_cnt > (unsigned short)0 ) {
		output_bind_ndx = calc_ndx( *output_sorted_bind, output_bind );

		map[ output_bind_ndx ] = USHRT_MAX;

		++output_sorted_bind;
		--output_bind_cnt;
	}

	if ( mapped_cnt == (unsigned short)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "No output fields match input fields.\n", stderr );
		RETURN_INT( -1 );
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Mapped %hu output fields.\n", mapped_cnt );
	}

	RETURN_INT( 0 );
}

static int verify_not_key_field( unsigned short *key_ndx_tbl, unsigned short key_tbl_cnt, unsigned short missing_field_ndx, const char *input_dfile_name, const char *field_name )
{
	if ( key_ndx_tbl == (unsigned short *)0 ) {
		/*
		** This is a dfile to be merged. This verification is
		** performed later (somewhere else) for dfiles to be merged.
		*/
		return 0;
	}

	for ( ; key_tbl_cnt > (unsigned short)0; --key_tbl_cnt ) {
		if ( *key_ndx_tbl == missing_field_ndx ) {
			FPUT_SRC_CODE( stderr );
			(void) fputs( "Key field [", stderr );
			(void) fputs( field_name, stderr );
			(void) fputs( "] was not defined in input dfile [", stderr );
			(void) fputs( input_dfile_name, stderr );
			(void) fputs( "].\n", stderr );
			return -1;
		}

		++key_ndx_tbl;
	}

	return 0;
}

static unsigned short calc_ndx( const dfile_bind_t *bind, const dfile_bind_t *base_bind )
{
	ptrdiff_t	ndx;

	assert( bind >= base_bind );
	ndx = bind - base_bind;

	assert( ndx >= 0 );
	assert( ndx < USHRT_MAX );

	return (unsigned short)ndx;
}
