/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <regex.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include "tbox.h"
#include "dfile.h"
#include "dfile_dynamic.h"
#include "dfile_utility.h"
#include "fixed2dfile.h"


#define NULL_CHECK( x ) ( ( *x == (const char *)0 ) ? "null" : *x )

static void	print_usage( const char * );

/*
** This function processes the command line arguments.
*/
int get_args( int argc, char * const argv[], const char **input_file_name, const char **output_dfile_name, dfile_tag_t **dfile_tag_tbl, unsigned short *dfile_tag_tbl_cnt, const char **filter_file_name, const char **control_file_name, size_t *input_record_length )
{
	int	ch, ndx;
	extern char	*optarg;
	long	rec_len;

	assert( argv != (char * const *) 0 );
	assert( input_file_name != (const char **) 0 );
	assert( output_dfile_name != (const char **) 0 );
	assert( dfile_tag_tbl != (dfile_tag_t **) 0 );
	assert( dfile_tag_tbl_cnt != (unsigned short *) 0 );
	assert( filter_file_name != (const char **)0 );
	assert( control_file_name != (const char **)0 );
	assert( input_record_length != (size_t *)0 );

#ifndef NDEBUG
	/*
	** Initialize debug to off.
	*/
	Debug = 0;
#endif

	*input_file_name = (const char *)0;
	*output_dfile_name = (const char *)0;
	*dfile_tag_tbl = (dfile_tag_t *)0;
	*dfile_tag_tbl_cnt = (unsigned short)0;
	*filter_file_name = (const char *)0;
	*control_file_name = (const char *)0;
	*input_record_length = (size_t)0;

	while ( ( ch = getopt( argc, argv, "c:di:o:l:t:z:" ) ) != EOF ) {
		switch ( ch ) {
		case 'd':
#ifdef NDEBUG
			(void) fputs( "Not compiled for debug.\n", stderr );
#else
			Debug = 1;
#endif
			break;
		case 'c':
			*control_file_name = optarg;
			break;
		case 'i':
			*input_file_name = optarg;
			break;
		case 'l':
			rec_len = atol( optarg );
			if ( rec_len <= 0 ) {
				FPUT_SRC_CODE( stderr );
				(void) fputs( "Could not convert input record length [", stderr );
				(void) fputs( optarg, stderr );
				(void) fputs( "] to a positive integer.\n", stderr );
				print_usage( argv[ 0 ] );
				return -1;
			}
			*input_record_length = (size_t)rec_len;
			break;
		case 'o':
			*output_dfile_name = optarg;
			break;
		case 't':
			if ( cmd_line_parse_tag( dfile_tag_tbl, dfile_tag_tbl_cnt, optarg ) == -1 ) {
				return -1;
			}
			break;
		case 'z':
			*filter_file_name = optarg;
			break;
		default:
			print_usage( argv[ 0 ] );
			return -1;
		}
	}

	if ( *input_file_name == (char *) 0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Must specify input file name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *output_dfile_name == (char *) 0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Must specify output dfile name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *control_file_name == (char *) 0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Must control file name.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( *input_record_length == (size_t)0 ) {
		FPUT_SRC_CODE( stderr );
		(void) fputs( "Input record length is required.\n", stderr );
		print_usage( argv[ 0 ] );
		return -1;
	}

	if ( Debug ) {
		(void) fprintf( stderr, "Input File Name: [%s]\n", NULL_CHECK( input_file_name ) );
		(void) fprintf( stderr, "Output DFile Name: [%s]\n", NULL_CHECK( output_dfile_name ) );
		(void) fprintf( stderr, "Control File Name: [%s]\n", NULL_CHECK( control_file_name ) );
		(void) fprintf( stderr, "Filter File Name: [%s]\n", NULL_CHECK( filter_file_name ) );
		(void) fprintf( stderr, "Input Record Length: [%u]\n", *input_record_length );
		(void) fputs( "Tags:\n", stderr );
		for ( ndx = (unsigned short)0; ndx < *dfile_tag_tbl_cnt; ++ndx ) {
			(void) fprintf( stderr, "DFile Tag [%s] [%s]\n", ( *dfile_tag_tbl )[ ndx ].tag, ( *dfile_tag_tbl )[ ndx ].tag_value );
		}
	}

	return 0;
}

static void print_usage( const char *exec_name )
{
	(void) fputs( "usage: ", stderr );
	(void) fputs( exec_name, stderr );
	(void) fputs( " [-d]", stderr );
	(void) fputs( " -c file", stderr );
	(void) fputs( " -i file", stderr );
	(void) fputs( " -l n", stderr );
	(void) fputs( " -o dfile", stderr );
	(void) fputs( " [-t %x=abc]", stderr );
	(void) fputs( " [-z file]", stderr );
	(void) fputc( '\n', stderr );
	(void) fputs( "\t-d -> debug\n", stderr );
	(void) fputs( "\t-c -> control file\n", stderr );
	(void) fputs( "\t-i -> input file\n", stderr );
	(void) fputs( "\t-l -> input record length\n", stderr );
	(void) fputs( "\t-o -> output DFile\n", stderr );
	(void) fputs( "\t-t -> DFile path tags\n", stderr );
	(void) fputs( "\t-z -> filter file\n", stderr );
}
