/* Copyright (C) 2009, 2010, 2011, 2012 Keith Crane

This file is part DFILE Tools.

DFILE Tools is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or (at
your option) any later version.

DFILE Tools is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
for more details.

You should have received a copy of the GNU General Public License along
with DFILE Tools; see the file COPYING.  If not, see
<http://www.gnu.org/licenses/>. */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <assert.h>
#include "tbox.h"


/*
** This function is a wrapper for the standard bsearch() function.
** It is used for searching on non-unique keys. The address of
** first element in array containing key value is returned. Function
** arguments are same as bsearch().
*/

void *nubsearch( const void *key, const void *base, size_t num_of_elements, size_t size_of_element, int (*compare_func) ( const void *, const void * ) )
{
	void	*ptr, *prev_element;

	assert( key != (const void *)0 );
	assert( base != (const void *)0 );
	assert( compare_func != (int (*)(const void *, const void *))0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( [%u], [%u] )\n", __func__, num_of_elements, size_of_element );
	}

	DEBUG_FUNC_START;

	ptr = bsearch( key, base, num_of_elements, size_of_element, compare_func );

	if ( ptr == (void *)0 ) {
		/*
		** No array element matched.
		*/
		if ( Debug ) {
			(void) fputs( "No array element matched.\n", stderr );
		}
		RETURN_POINTER( (void *)0 );
	}

	/*
	** Find first occurrence by backing up in array.
	*/
	while ( ptr > base ) {
		prev_element = (void *)&( (char *)ptr )[ -size_of_element ];
		if ( ( *compare_func )( key, prev_element ) != 0 ) {
			/*
			** Previous array entry does not match;
			** stop backing up.
			*/
			break;
		}
		ptr = prev_element;
	}

	/*
	** Return first occurrence.
	*/

	RETURN_POINTER( ptr );
}

/*
** This function is expected to be called after nubsearch() was successfully
** called. It returns address of next array element with matching key value.
** If key value of next array entry is different or end of array is reached,
** null pointer is returned.
*/
void *nubsearch_next( const void *previous_returned_element, const void *base, size_t num_of_elements, size_t size_of_element, int (*compare_func) ( const void *, const void * ) )
{
	void	*next, *end_of_array;

	assert( previous_returned_element != (const void *)0 );
	assert( base != (const void *)0 );
	assert( compare_func != (int (*)(const void *, const void *))0 );

	if ( Debug ) {
		(void) fprintf( stderr, "%s( [%u], [%u] )\n", __func__, num_of_elements, size_of_element );
	}

	DEBUG_FUNC_START;

	/*
	** Calculate address past end of array.
	*/
	end_of_array = (void *)&( (char *)base )[ num_of_elements * size_of_element ];

	/*
	** Calculate address of next array entry.
	*/
	next = (void *)&( (char *)previous_returned_element )[ size_of_element ];

	if ( next >= end_of_array ) {
		/*
		** Next array entry would be past end of array.
		*/
		if ( Debug ) {
			(void) fputs( "Reach physical end of array.\n", stderr );
		}
		RETURN_POINTER( (void *)0 );
	}

	if ( ( *compare_func )( previous_returned_element, next ) != 0 ) {
		/*
		** Next array entry has a different key value.
		*/
		if ( Debug ) {
			(void) fputs( "Key value different.\n", stderr );
		}
		RETURN_POINTER( (void *)0 );
	}

	if ( Debug ) {
		(void) fputs( "Returning next array element.\n", stderr );
	}
	RETURN_POINTER( next );
}

#ifdef MT_nubsearch

/*
** This function is used to regression test nubsearch().
** The following command is used to compile:
**   x=nubsearch; make "MT_CC=-DMT_$x" $x
*/
int int_cmp( const void *, const void * );

int main( void )

{
	static const char	complete_msg[] =  ">>> Module test on function %s() is complete.\n";
	static const char	test_func[] = "nubsearch";
	static const char	incorrectly_successful[] = ">>>\n>>> %s() was incorrectly successful.\n";
	static const char	correctly_unsuccessful[] = ">>>\n>>> %s() was correctly unsuccessful.\n";
	static const char	correctly_successful[] = ">>>\n>>> %s() was correctly successful.\n";
	static const char	incorrectly_unsuccessful[] = ">>>\n>>> %s() was incorrectly unsuccessful.\n";
	static const char	blank_line[] = ">>>\n";

	static int	value_tbl[] = {
		1,
		2, 2,
		3, 3, 3,
		4, 4, 4, 4,
		5, 5, 5, 5, 5
	};
	const size_t	value_tbl_cnt = sizeof( value_tbl ) / sizeof( int );
	const int	max_value = 5;
	int	key, cnt, *ptr;

	Debug = 1;

	for ( key = 1; key <= max_value; ++key ) {
		cnt = 0;
		ptr = nubsearch( (const void *)&key, (const void *)value_tbl, value_tbl_cnt, sizeof( int ), int_cmp );

		while ( ptr != (int *)0 ) {
			if ( *ptr != key ) {
				(void) fputs( "Failure: incorrect value was returned.\n", stderr );
				(void) fprintf( stderr, complete_msg, test_func );
				exit( 0 );
			}

			++cnt;

			ptr = nubsearch_next( (const void *)ptr, (const void *)value_tbl, value_tbl_cnt, sizeof( int ), int_cmp );
		}
		if ( cnt == key ) {
			(void) fputs( blank_line, stderr );
			(void) fprintf( stderr, ">>> Successfully processed %d.\n", key );
			(void) fputs( blank_line, stderr );
		} else {
			(void) fputs( "Failure: incorrect number of array entries were returned.\n", stderr );
			(void) fprintf( stderr, complete_msg, test_func );
			exit( 0 );
		}
	}

	for ( key = max_value; key >= 1; --key ) {
		cnt = 0;
		ptr = nubsearch( (const void *)&key, (const void *)value_tbl, value_tbl_cnt, sizeof( int ), int_cmp );

		while ( ptr != (int *)0 ) {
			if ( *ptr != key ) {
				(void) fputs( "Failure: incorrect value was returned.\n", stderr );
				(void) fprintf( stderr, complete_msg, test_func );
				exit( 0 );
			}

			++cnt;

			ptr = nubsearch_next( (const void *)ptr, (const void *)value_tbl, value_tbl_cnt, sizeof( int ), int_cmp );
		}
		if ( cnt == key ) {
			(void) fputs( blank_line, stderr );
			(void) fprintf( stderr, ">>> Successfully processed %d.\n", key );
			(void) fputs( blank_line, stderr );
		} else {
			(void) fputs( "Failure: incorrect number of array entries were returned.\n", stderr );
			(void) fprintf( stderr, complete_msg, test_func );
			exit( 0 );
		}
	}

	(void) fputs( ">>> Regression test successful.\n", stderr );
	(void) fprintf( stderr, complete_msg, test_func );
	exit( 0 );
}

int int_cmp( const void *x, const void *y )
{
	return *(int *)x - *(int *)y;
}
#endif
